
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 * Some of the code in this file was copied from the xine-ui project.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: oxine.c,v 1.140 2006/02/08 20:44:37 mschwerin Exp $
 *
 */

#include "config.h"

#include <getopt.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#include "codeset.h"
#include "disc.h"
#include "environment.h"
#include "event.h"
#include "filelist_menu.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "main_menu.h"
#include "oxine.h"
#include "playlist_m3u.h"
#include "scheduler.h"

// global variables
oxine_t *oxine;

#define OPTION_VERBOSE      (1000)

static const char *short_options = "?hvf"
#ifdef HAVE_DISC_POLLING
    "P"
#endif
#ifdef HAVE_LIRC
    "L"
#endif
#ifdef HAVE_JOYSTICK
    "J"
#endif
    "V:A:";

static struct option long_options[] = {
    {"help", no_argument, 0, 'h'},
    {"version", no_argument, 0, 'v'},
    {"fullscreen", no_argument, 0, 'f'},
    {"video-driver", required_argument, 0, 'V'},
    {"audio-driver", required_argument, 0, 'A'},
#ifdef HAVE_DISC_POLLING
    {"no-polling", no_argument, 0, 'P'},
#endif
#ifdef HAVE_LIRC
    {"no-lirc", no_argument, 0, 'L'},
#endif
#ifdef HAVE_JOYSTICK
    {"no-joystick", no_argument, 0, 'J'},
#endif
    {"verbose", optional_argument, 0, OPTION_VERBOSE},
    {0, no_argument, 0, 0}
};

static void
show_usage (void)
{
    xine_t *xine = xine_new ();
    xine_init (xine);

    printf ("\n");
    printf (_("Usage: oxine [OPTIONS] [MRL]\n"));
    printf ("\n");
    printf (_("OPTIONS are:\n"));
    printf (_("  -v, --version                Display version.\n"));
    printf (_
            ("      --verbose [=level]       Set verbosity level. Default is 0.\n"));
    printf (_("  -V, --video-driver <drv>     Select video driver by ID.\n"));
    printf (_("                               Available drivers:\n"));
    printf ("                               auto ");

    const char *const *driver_ids;
    if ((driver_ids = xine_list_video_output_plugins (xine))) {
        const char *driver_id = *driver_ids++;
        while (driver_id) {
            if (odk_supports_video_driver (driver_id))
                printf ("%s ", driver_id);
            driver_id = *driver_ids++;
        }
    }
    printf ("\n");
    printf (_("  -A, --audio-driver <drv>     Select audio driver by ID.\n"));
    printf (_("                               Available drivers:\n"));
    printf ("                               ");
    if ((driver_ids = xine_list_audio_output_plugins (xine))) {
        const char *driver_id = *driver_ids++;
        while (driver_id) {
            printf ("%s ", driver_id);
            driver_id = *driver_ids++;
        }
    }
    printf ("\n");
    printf (_("  -f, --fullscreen             Start in fullscreen mode.\n"));
#ifdef HAVE_DISC_POLLING
    printf (_("  -P, --no-polling             Disable disc polling.\n"));
#endif
#ifdef HAVE_LIRC
    printf (_("  -L, --no-lirc                Turn off LIRC support.\n"));
#endif
#ifdef HAVE_JOYSTICK
    printf (_("  -J, --no-joystick            Turn off joystick support.\n"));
#endif
    printf ("\n\n");
    printf (_("examples for valid MRLs (media resource locator):\n"
              "  File:  'path/foo.vob'\n"
              "         '/path/foo.vob'\n"
              "         'file://path/foo.vob'\n"
              "         'fifo://[[mpeg1|mpeg2]:/]path/foo'\n"
              "         'stdin://[mpeg1|mpeg2]'\n"
              "  DVD:   'dvd://VTS_01_2.VOB'\n"
              "         'dvd://path/foo.iso'\n"
              "         'dvd://<track number>'\n"
              "  VCD:   'vcd://<track number>'\n"));
    printf ("\n");

    xine_exit (xine);
}


int
main (int argc, char **argv)
{
    int c;
    int option_index;
    char *video_driver_id = NULL;
    char *audio_driver_id = NULL;

    int verbosity = XINE_VERBOSITY_NONE;

    oxine = ho_new (oxine_t);

    int fullscreen = 0;
#ifdef HAVE_LIRC
    int use_lirc = 1;
#endif
#ifdef HAVE_JOYSTICK
    int use_joystick = 1;
#endif
#ifdef HAVE_DISC_POLLING
    oxine->use_polling = 1;
#endif

#ifdef DEBUG
    char *codeset = get_system_encoding ();
    debug ("   PACKAGE_NAME: %s", PACKAGE_NAME);
    debug ("PACKAGE_VERSION: %s", PACKAGE_VERSION);
    debug ("  OXINE_DATADIR: %s", OXINE_DATADIR);
    debug ("OXINE_LOCALEDIR: %s", OXINE_LOCALEDIR);
    debug ("   XINE_VERSION: %s", XINE_VERSION);
    debug ("           HOME: %s", get_dir_home ());
    debug ("           LANG: %s", getenv ("LANG"));
    debug ("        CODESET: %s", codeset);
    if (codeset)
        ho_free (codeset);
#endif

    /* Set up internationalization */
#ifdef HAVE_SETLOCALE
    if (!setlocale (LC_ALL, ""))
        error ("The current locale is not supported by the C library!");
#endif

#ifdef ENABLE_NLS
    bindtextdomain (PACKAGE, OXINE_LOCALEDIR);
    textdomain (PACKAGE);
#endif

    /* Parse command line arguments */
    while ((c = getopt_long (argc, argv, short_options,
                             long_options, &option_index)) != EOF) {
        switch (c) {
            case OPTION_VERBOSE:
                if (optarg != NULL)
                    verbosity = strtol (optarg, &optarg, 10);
                else
                    verbosity = 1;
                break;
            case 'A':
                audio_driver_id = ho_strdup (optarg);
                break;
            case 'V':
                video_driver_id = ho_strdup (optarg);
                break;
            case 'f':
                fullscreen = 1;
                break;
#ifdef HAVE_DISC_POLLING
            case 'P':
                oxine->use_polling = 0;
                break;
#endif
#ifdef HAVE_LIRC
            case 'L':
                use_lirc = 0;
                break;
#endif
#ifdef HAVE_JOYSTICK
            case 'J':
                use_joystick = 0;
                break;
#endif
            case 'v':
                printf (_("This is %s - A free Video-Player v%s\n"),
                        PACKAGE_NAME, PACKAGE_VERSION);
                printf (_("(c) 2005 by the oxine project\n"));
                exit (1);
                break;
            case 'h':
            case '?':
                show_usage ();
                exit (0);
                break;
            default:
                error (_("Invalid argument %d!"));
                show_usage ();
                exit (1);
                break;
        }
    }

    oxine->xine = xine_new ();
    oxine->user_interface_is_visible = 0;
    oxine->control_menu_is_visible = 0;
    oxine->otk_clear_job = 0;
    oxine->playlist = playlist_new ();
    oxine->removable_discs = filelist_new (ALLOW_FILES_ANY, "/");
    oxine->toplevel_filelist = NULL;
    oxine->current_filelist = NULL;
    oxine->mostlistened_filelist = NULL;
    pthread_mutex_init (&oxine->download_mutex, NULL);

    /* The config files */
    debug ("reading configuration from file %s", get_file_config ());
    xine_config_load (oxine->xine, get_file_config ());

    /* Init the xine engine */
    xine_engine_set_param (oxine->xine, XINE_ENGINE_PARAM_VERBOSITY,
                           verbosity);
    xine_init (oxine->xine);

    /* Preferred drivers */
    xine_config_register_string (oxine->xine, "video.driver", "auto",
                                 _("video driver to use"),
                                 _("video driver to use"), 10, NULL, NULL);
    xine_config_register_string (oxine->xine, "audio.driver", "auto",
                                 _("audio driver to use"),
                                 _("audio driver to use"), 10, NULL, NULL);

    xine_cfg_entry_t centry;
    if (video_driver_id) {
        xine_config_lookup_entry (oxine->xine, "video.driver", &centry);
        centry.str_value = video_driver_id;
        xine_config_update_entry (oxine->xine, &centry);
        ho_free (video_driver_id);
    }

    if (audio_driver_id) {
        xine_config_lookup_entry (oxine->xine, "audio.driver", &centry);
        centry.str_value = audio_driver_id;
        xine_config_update_entry (oxine->xine, &centry);
        ho_free (audio_driver_id);
    }

    start_scheduler ();

    oxine->odk = odk_init (oxine->xine);
    if (!oxine->odk) {
        fatal ("Could not initialize odk!");
        abort ();
    }

    oxine->otk = otk_init (oxine->odk);
    if (!oxine->otk) {
        fatal ("Could not inizialize otk");
        abort ();
    }
    otk_set_event_handler (oxine->otk, oxine_event_handler, oxine);

#ifdef HAVE_DISC_POLLING
    if (oxine->use_polling)
        disc_start_polling (oxine);
#endif
#ifdef HAVE_LIRC
    if (use_lirc)
        odk_listen_to (oxine->odk, "lirc");
#endif
#ifdef HAVE_JOYSTICK
    if (use_joystick)
        odk_listen_to (oxine->odk, "js");
#endif

    /* Register shutdown command */
    xine_config_register_string (oxine->xine, "misc.shutdown_command", "",
                                 _("what command is executed on shutdown"),
                                 _("what command is executed on shutdown"),
                                 10, NULL, NULL);

    /* Register device of photo CDs */
    xine_config_register_string (oxine->xine, "media.photo_cd.device",
                                 "/dev/cdrom",
                                 _("device used for photo CD"),
                                 _("The path to the device, usually a "
                                   "CD or DVD drive, which you intend "
                                   "to use for playing photo CDs."), 10,
                                 NULL, NULL);

    /* Register auto reloading of playlist */
    xine_config_register_bool (oxine->xine, "gui.playlist.auto_reload",
                               1, _("Automatically reload old playlist"),
                               _("If this is enabled and if you don't "
                                 "specify any MRL on the command line, "
                                 "oxine will automatically load the "
                                 "previous playlist."), 10, NULL, NULL);

    /* Register auto playing of playlist */
    xine_config_register_bool (oxine->xine, "gui.playlist.auto_play",
                               0, _("Automatically play reloaded playlist"),
                               _("If this is enabled and "
                                 "gui.playlist_auto_reload is also enabled "
                                 "the reloaded playlist will automatically "
                                 "be played."), 10, NULL, NULL);

    /* Register position in playlist */
    xine_config_register_num (oxine->xine, "gui.playlist.current_track",
                              -1, _("The last played track in the playlist"),
                              _("This is the number in the playlist of the "
                                "last played track."), 10, NULL, NULL);

    /* Make sure all of these entries have a value. */
    xine_config_lookup_entry (oxine->xine, "media.dvd.device", &centry);
    if (!centry.str_value || (strlen (centry.str_value) == 0)) {
        centry.str_value = "/dev/dvd";
        xine_config_update_entry (oxine->xine, &centry);
    }
    xine_config_lookup_entry (oxine->xine, "media.audio_cd.device", &centry);
    if (!centry.str_value || (strlen (centry.str_value) == 0)) {
        centry.str_value = "/dev/cdrom";
        xine_config_update_entry (oxine->xine, &centry);
    }
    xine_config_lookup_entry (oxine->xine, "media.vcd.device", &centry);
    if (!centry.str_value || (strlen (centry.str_value) == 0)) {
        centry.str_value = "/dev/cdrom";
        xine_config_update_entry (oxine->xine, &centry);
    }

    /* Add anything that was not recognized to the playlist */
    int i = optind;
    filelist_t *arglist = filelist_new (ALLOW_FILES_ANY, "/");
    for (; i < argc; i++) {
        char *mrl = argv[i];
        char *title = create_title (mrl);
        filelist_add (arglist, title, mrl, FILE_TYPE_UNKNOWN);
        ho_free (title);
    }
    fileitem_t *fileitem = filelist_first (arglist);
    while (fileitem) {
        filelist_expand (arglist, fileitem);
        playlist_add_fileitem (oxine->playlist, fileitem);
        fileitem = filelist_next (arglist, fileitem);
    }
    filelist_free (arglist);

    /* Show main window. */
    odk_show_window (oxine->odk, fullscreen);

    oxine->filelist_menu = show_music_menu_cb;
    oxine->repaint_menu = show_main_menu_cb;
    oxine->playback_ended_menu = show_main_menu_cb;
    oxine->backto_menu = show_main_menu_cb;

    /* If the user wants this we reload the last playlist and start playing
     * the last song that was played before oxine exited. */
    xine_config_lookup_entry (oxine->xine, "gui.playlist.auto_reload",
                              &centry);
    int auto_reload = centry.num_value;

    xine_config_lookup_entry (oxine->xine, "gui.playlist.auto_play", &centry);
    int auto_play = centry.num_value;

    xine_config_lookup_entry (oxine->xine, "gui.playlist.current_track",
                              &centry);
    int current_track = centry.num_value;

    playitem_t *item2play = NULL;
    if (playlist_length (oxine->playlist) == 0) {
        if (auto_reload)
            m3u_load (oxine->playlist, get_file_lastplaylist ());

        if (auto_reload && auto_play && (current_track != -1)) {
            int list_length = playlist_length (oxine->playlist);
            if ((current_track > 0) && (current_track < list_length)) {
                item2play = playlist_set_current_pos (oxine->playlist,
                                                      current_track);
            } else {
                item2play = playlist_get_first (oxine->playlist);
            }
        }
    } else {
        item2play = playlist_get_first (oxine->playlist);
    }
    playlist_play_item (oxine, item2play);

    /* This is where we start the event loop. */
    odk_run (oxine->odk);

    stop_scheduler ();

    if (oxine->otk)
        otk_free (oxine->otk);
    if (oxine->odk)
        odk_free (oxine->odk);
#ifdef HAVE_DISC_POLLING
    if (oxine->use_polling)
        disc_stop_polling (oxine);
#endif

    /* Save and free the playlist */
    if (oxine->playlist) {
        m3u_save (oxine->playlist, get_file_lastplaylist ());
        playlist_free (oxine->playlist);
    }

    if (oxine->toplevel_filelist)
        filelist_free (oxine->toplevel_filelist);
    if (oxine->removable_discs)
        filelist_free (oxine->removable_discs);

    pthread_mutex_destroy (&oxine->download_mutex);
    xine_config_save (oxine->xine, get_file_config ());
    xine_exit (oxine->xine);
    ho_free (oxine);

    environment_free ();

    heapstat ();

    return 0;
}
