/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@rebelbase.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <time.h>

#include <gnome.h>

#include <pan/base/argset.h>
#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/log.h>

#include <pan/globals.h>
#include <pan/prefs.h>
#include <pan/util.h>

/***
****
****  GUI
****
***/

void
pan_info_dialog (const char *format, ...)
{
	va_list args;
	char *str = NULL;
	g_return_if_fail (format != NULL);
	va_start (args, format);
	str = g_strdup_vprintf (format, args);
	va_end (args);
	g_warning (str);
	pan_lock();
	gnome_ok_dialog_parented (str, GTK_WINDOW(Pan.window));
	pan_unlock();
	g_free (str);
}

void
pan_error_dialog_parented (gpointer parent, const char *format, ...)
{
	va_list args;
	char *str = NULL;
	g_return_if_fail (format != NULL);
	va_start (args, format);
	str = g_strdup_vprintf (format, args);
	va_end (args);
	g_warning (str);
	pan_lock();
	gnome_error_dialog_parented (str, GTK_WINDOW(parent));
	pan_unlock();
	g_free (str);
}

void
pan_error_dialog (const char *format, ...)
{
	va_list args;
	char *str = NULL;
	g_return_if_fail (format != NULL);
	va_start (args, format);
	str = g_strdup_vprintf (format, args);
	va_end (args);
	g_warning (str);
	pan_lock();
	gnome_error_dialog_parented (str, GTK_WINDOW(Pan.window));
	pan_unlock();
	g_free (str);
}

/***
****
****  IDLE FUNC
****
***/

/**
 * Thus spoke the GTK FAQ: ``Callbacks require a bit of attention.
 * Callbacks from GTK+ (signals) are made within the GTK+ lock. However
 * callbacks from GLib (timeouts, IO callbacks, and idle functions) are
 * made outside of the GTK+ lock. So, within a signal handler you do not
 * need to call gdk_threads_enter(), but within the other types of
 * callbacks, you do.''
 *
 * pan_timeout_add() and run_in_main_thread_idle() are wrappers around
 * glib-level callbacks that ensure that pan_lock() works properly
 * to give them a gdk_threads_enter() lock.
 */

static gboolean main_thread_is_in_glib_callback = FALSE;

static gint
pan_timeout_wrapper (gpointer data)
{
	ArgSet * argset = (ArgSet*) data;
	GtkFunction func = (GtkFunction) argset_get (argset, 0);
	gpointer user_data = (gpointer) argset_get (argset, 1);
	gint retval;

	/* sanity clause */
	pan_warn_if_fail (pthread_self() == Pan.main_t);

	/* call the timer func */
	main_thread_is_in_glib_callback = TRUE;
	retval = (*func)(user_data);
	main_thread_is_in_glib_callback = FALSE;

	/* cleanup */
	if (!retval)
		argset_free (argset);
	return retval;
}
guint
pan_timeout_add (guint32 interval, GtkFunction func, gpointer arg)
{
	return gtk_timeout_add (interval, pan_timeout_wrapper, argset_new2 (func, arg));
}

/***
****
****  LOCKING
****
***/

static pthread_t has_lock_thr = 0;

void
pan_lock_from (const gchar * file, const gchar * func, int line)
{
	static const gchar * last_file = NULL;
	static const gchar * last_func = NULL;
	static int last_line = -1;
	const pthread_t thr = pthread_self();

	/**
	 * If pan_lock() is called from the main thread while it has a GUI lock
	 * (typically from a gtk signals, like a button press signal etc.)
	 * then we don't need to lock.
	 *
	 * However if pan_lock() is called from a worker thread, or the main
	 * thread inside a glib idle function (via run_in_main_thread_nolock())
	 * then we _do_ need to obtain a gtk lock.
	 */
	if (thr==Pan.main_t && !main_thread_is_in_glib_callback)
	{
		debug4 (DEBUG_LOCK,"mainthread %lu attempted unnecessary lock from %s:%d (%s)", thr, file, line, func);
	}
	else if (thr == has_lock_thr)
	{
		g_error (_("thread %lu attempted double lock!\nfirst lock was in %s:%d (%s),\nnow trying for another one from %s:%d (%s)"),
			(gulong)thr,
			last_file, last_line, last_func,
			file, line, func);
	}
	else
	{
	       	/* if (thr==Pan.main_t && main_thread_is_in_glib_callback)
			odebug3 ("idle func %s:%d (%s) getting a pan lock", file, line, func);*/

		gdk_threads_enter();
		last_file = file;
		last_func = func;
		last_line = line;
		has_lock_thr = thr;
		debug3 (DEBUG_LOCK,"thread %lu entered gdk_threads from %s %d", thr, file, line);
	}
}

void
pan_unlock_from (const gchar* file, const gchar * func, int line)
{
	const pthread_t thr = pthread_self();

	if (thr==Pan.main_t && !has_lock_thr)
	{
		debug4 (DEBUG_LOCK,"mainthread %lu attempted unneccessary unlock from %s:%d (%s)", (gulong)thr, file, line, func);
	}
	else if (has_lock_thr != thr)
	{
		g_error (_("thread %lu attempted to remove a lock it didn't have from %s:%d (%s)"), (gulong)thr, file, line, func);
	}
	else
	{
		has_lock_thr = (pthread_t)0;
		gdk_threads_leave();
		debug4 (DEBUG_LOCK,"thread %lu left gdk_threads from %s:%d (%s)", pthread_self(), file, line, func);
	}
}

/***
****
****  LINKIFY TEXT
****
***/

/* This is pretty much taken straight from
   slrn's art.c, written by John E. Davis */
static const gchar*
find_url (const gchar  * in,
          guint        * setme_url_len)
{
	const gchar * ptr;

	while ((ptr = pan_strstr (in, "://")) != NULL)
	{
		const gchar * begin;
		const gchar * end;
		gboolean again;

		/* walk back to the beginning of the word */
		begin = ptr;
		while ((begin>in) && isalpha((guchar)begin[-1]))
			--begin;

		/* all registered and reserved scheme names are >= 3 chars long */
		if (begin+3 > ptr) {
			in = ptr + 3;
			continue;
		}

		end = ptr;
		again = TRUE;
		while (again)
		{
			gchar ch = *end;

			switch (ch)
			{
				case ',':
				case ';':
				case '.':
					ch = *++end;
					if (isspace((int)ch)) {
						--end;
						again = FALSE;
					}
					break;

				case ' ':
				case '\t':
				case '\n':
				case '\"':
				case '}':
				case '{':
				case ')':
				case '>':
				case '<':
				case 0:
					again = FALSE;
					break;

				default:
					++end;
					break;
			}
		}

		*setme_url_len = (guint)(end - begin);
		if (*setme_url_len < 7)
			continue;

		if (!strncmp (end-3, "://", 3))
			continue;

		return begin;
	}

	return NULL;
}

gchar*
linkify_text (const gchar * text)
{
	guint len = 0;
	gchar * retval;
	GString * str;
	const gchar * prev = text;
	const gchar * pch;
	debug_enter ("linkify_text");

       	str = g_string_new (NULL);
	while ((pch = find_url(prev,&len)) != NULL)
	{
		gchar * url = g_strndup (pch, len);
		if (pch != prev)
			pan_g_string_append_len (str, prev, (guint)(pch-prev));

		g_string_append (str, "<a href=\"");
		g_string_append (str, url);
		g_string_append (str, "\">");
		g_string_append (str, url);
		g_string_append (str, "</a>");

		prev = pch + len;
		g_free (url);
	}
	g_string_append (str, prev);

	retval = str->str;
	g_string_free (str, FALSE);

	debug_exit ("linkify_text");
	return retval;
}

/***
****
****  UNSORTED
****
***/

void
get_date_display_string (time_t date, const gchar * fmt, gchar * fillme, gint len)
{
	struct tm tm_date;

	/* sanity clause */
	g_return_if_fail (is_nonempty_string(fmt));
	g_return_if_fail (fillme != NULL);
	*fillme = '\0';

	/* get the date string */
        if (display_article_dates_in_local_time)
                localtime_r (&date, &tm_date);
        else
                gmtime_r (&date, &tm_date);

	/* try to put the date in the buffer */
	if (!strftime (fillme, len, fmt, &tm_date))
		*fillme = '\0';
}

void
open_outside_file (const char *fname)
{
	const char * mime_type = NULL;
	const char * mime_prog = NULL;
	char **argv;
	int argc;

	if (!((mime_type = gnome_mime_type (fname))))
		return;
	if (!((mime_prog = gnome_mime_program (mime_type))))
		return;
	if (!strstr (mime_prog, "%f"))
		return;

	argv = g_strsplit (mime_prog, " ", -1);
	for (argc=0; argv[argc]; argc++)
		if (!strcmp(argv[argc], "%f"))
			replace_gstr (argv+argc, g_strdup(fname));

	gnome_execute_async (NULL, argc, argv);

	g_strfreev (argv);
}

static gint
try_to_execute_url (const gchar * template, const gchar * url)
{
	gchar * str;
	gchar ** argv;
	gint argc;
	gint retval;

	/* sanity clause */
	g_return_val_if_fail (is_nonempty_string(url), -1);
	if (template == NULL)
		template = "netscape %s";

	/* add the URL to the template */
	if (strstr(template,"%s"))
		str = pan_substitute (template, "%s", url);
	else
		str = g_strdup_printf ("%s %s", template, url);
	replace_gstr (&str, pan_substitute(str, "%%%%", "%%"));

	/* break the command into parts as gnome_execute_async wants */
	argc = 0;
	argv = NULL;
	if (poptParseArgvString(str, &argc, &argv) != 0) {
		g_warning ("Parse error of `%s'", template);
		return -1;
	}

	/* try to execute it */
	log_add_va (LOG_INFO, _("Trying to open URL `%s' with `%s'"), url, str);
	pan_lock ();
	retval = gnome_execute_async (NULL, argc, argv);
	pan_unlock ();

	/* cleanup */
	free (argv);
	g_free (str);
	return retval;
}


void
pan_url_show (const gchar * url)
{
	gboolean is_http;
	gchar * browser;

	/* sanity clause */
	g_return_if_fail (is_nonempty_string(url));

	/* if it's a browser url, we have some user configuration to contend with */
	is_http = !strncmp (url, "http://", 7);
	if (is_http)
	{
		/* first, see if user has selected a browser in Pan. */
		if (is_nonempty_string(external_web_browser)) {
			gint i;
			gint result = -1;
			gchar ** tries = g_strsplit (external_web_browser, ":", -1);
			for (i=0; result==-1 && tries[i]!=NULL; ++i) {
				g_message ("trying [%s]", tries[i]);
				result = try_to_execute_url (tries[i], url);
			}
			g_strfreev (tries);
			if (result != -1)
				return;
		}

		/* http://tuxedo.org/~esr/BROWSER/ */
		browser = getenv ("BROWSER");
		if (is_nonempty_string(browser)) {
			gint i;
			gint result = -1;
			gchar ** tries = strsplit_command_line_arguments (browser);
			for (i=0; result==-1 && tries[i]!=NULL; ++i) {
				g_message ("trying [%s]", tries[i]);
				result = try_to_execute_url (tries[i], url);
			}
			g_strfreev (tries);
			if (result != -1)
				return;
		}
	}

	/* just punt it to gnome, which is better at this */
	pan_lock ();
	gnome_url_show (url);
	pan_unlock ();
}
