/*
**  IMAPCacheManager.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#import <Pantomime/IMAPCacheManager.h>

#import <Pantomime/Constants.h>
#import <Pantomime/IMAPCacheObject.h>

static int currentVersion = 1;

@implementation IMAPCacheManager

- (id) init
{
  self = [super init];
  
  [IMAPCacheManager setVersion: currentVersion];

  [self setUIDValidity: 0];
  [self setIMAPCacheObjects: [NSArray array]];
 
  return self;
}

- (id) initWithPath: (NSString *) thePath
{
  self = [self init];
  
  [self setPathToCache: thePath];

  return self;
}

- (void) dealloc
{
  RELEASE(imapCacheObjects);

  [super dealloc];
}


- (BOOL) synchronize
{
  return [NSArchiver archiveRootObject: self toFile: [self pathToCache]];
}

//
// NSCoding protocol
//

- (void) encodeWithCoder: (NSCoder *) theCoder
{
  [IMAPCacheManager setVersion: currentVersion];
   
  [theCoder encodeObject: [NSNumber numberWithInt: [self uidValidity]] ];
  [theCoder encodeObject: [self imapCacheObjects]];
}

- (id) initWithCoder: (NSCoder *) theCoder
{
  int version;
  
  version = [theCoder versionForClassName: NSStringFromClass([self class])];
  
  self = [super init];

  // Initial version of the serialized filter object
  if ( version == 0 )
    {
      [self setUIDValidity: [[theCoder decodeObject] intValue]];

      // We ignore the previous cached objects
      [self setIMAPCacheObjects: [NSArray array]];
    }
  // Version 1
  else
    {
      [self setUIDValidity: [[theCoder decodeObject] intValue]];
      [self setIMAPCacheObjects: [theCoder decodeObject]];
    }
  
  return self;
}


//
// access/mutation methods
//


- (void) addObject: (IMAPCacheObject *) theIMAPCacheObject
{
  [imapCacheObjects addObject: theIMAPCacheObject];
}

- (void) removeObject: (IMAPCacheObject *) theIMAPCacheObject
{
  [imapCacheObjects removeObject: theIMAPCacheObject];
}

- (IMAPCacheObject *) findIMAPCacheObject: (int) theUID
{
  IMAPCacheObject *anObject;
  int i;

  for (i = 0; i < [[self imapCacheObjects] count]; i++)
    {
      anObject = [[self imapCacheObjects] objectAtIndex: i];

      if ([anObject uid] == theUID)
	{
	  return anObject;
	}
    }
  
  return nil;
}

- (NSArray *) imapCacheObjects
{
  return imapCacheObjects;
}

- (void) setIMAPCacheObjects: (NSArray *) theIMAPCacheObjects
{
  if ( theIMAPCacheObjects )
    {
      NSMutableArray *newIMAPCacheObjects;

      newIMAPCacheObjects = [[NSMutableArray alloc] initWithArray: theIMAPCacheObjects];
      RELEASE(imapCacheObjects);
      imapCacheObjects = newIMAPCacheObjects;
    }
  else
    {
      RELEASE(imapCacheObjects);
      imapCacheObjects = nil;
    }
}


- (NSString *) pathToCache
{
  return pathToCache;
}

//
//
//
- (void) setPathToCache: (NSString *) thePathToCache
{
  RETAIN(thePathToCache);
  RELEASE(pathToCache);
  pathToCache = thePathToCache;
}

//
//
//
- (int) uidValidity
{
  return uidValidity;
}

//
//
//
- (void) setUIDValidity: (int) theUIDValidity
{
  uidValidity = theUIDValidity;
}

//
//
//
- (void) flush
{
  [imapCacheObjects removeAllObjects];
  [self setUIDValidity: 0];
  [self synchronize];
}

//
// class methods
//

+ (id) imapCacheObjectsFromDiskWithPath: (NSString *) thePath;
{
  IMAPCacheManager *anIMAPCacheManager;
  
  anIMAPCacheManager = [NSUnarchiver unarchiveObjectWithFile: thePath ];
  
  if ( anIMAPCacheManager )
    {
      [anIMAPCacheManager setPathToCache: thePath];
    }
  else
    {
      anIMAPCacheManager = [[IMAPCacheManager alloc] initWithPath: thePath];
    }

  return anIMAPCacheManager;
}

@end
