/*
**  TCPConnection.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#import <Pantomime/TCPConnection.h>

#import <Pantomime/Constants.h>

#import <stdio.h>
#import <stdlib.h>
#import <netinet/in.h>
#import <sys/ioctl.h>
#import <sys/socket.h>
#import <sys/time.h>
#import <sys/types.h>
#import <netdb.h>
#import <string.h>
#import <unistd.h>	/* For read() and write() and close() */

#ifdef MACOSX
#import <sys/uio.h>	/* For read() and write() */
#endif

@implementation TCPConnection

//
//
//
- (void) dealloc
{
  RELEASE(name);

  [super dealloc];
}


//
//
//
- (id) initWithName: (NSString *) theName
	       port: (int) thePort
{
  return [self initWithName: theName
	       port: thePort
	       connectionTimeout: 30
	       readTimeout: 30
	       writeTimeout: 30];
}


//
// This methods throws an exception if the connection timeout
// is exhausted and the connection hasn't been established yet.d
//
- (id) initWithName: (NSString *) theName
	       port: (int) thePort
  connectionTimeout: (int) theConnectionTimeout
	readTimeout: (int) theReadTimeout
       writeTimeout: (int) theWriteTimeout
{
  struct sockaddr_in server;
  struct hostent *host_info;
  int nonblock = 1;

  if ( theName == nil || thePort <= 0 )
    {
      AUTORELEASE(self);
      NSLog(@"TCPConnection: Attempted to initialize with a nil name or a negative or zero port value.");
      return nil;
    }
  
  // We set our ivars through our mutation methods
  [self setName: theName];
  [self setPort: thePort];
  [self setConnectionTimeout: theConnectionTimeout];
  [self setReadTimeout: theReadTimeout];
  [self setWriteTimeout: theWriteTimeout];
  
  // We get the file descriptor associated to a socket
  fd = socket(PF_INET, SOCK_STREAM, 0);

  if ( [self fd] == -1 ) 
    {
      AUTORELEASE(self);
      NSLog(@"TCPConnection: An error occured while creating the endpoint for communications");
      return nil;
    }

  // We get our hostent structure for our server name
  host_info = gethostbyname([[self name] cString]);
  
  if ( !host_info )
    {
      AUTORELEASE(self);
      NSLog(@"TCPConnection: Unable to get the hostent structure.");
      return nil;
    }

  server.sin_family = host_info->h_addrtype;
  memcpy((char *)&server.sin_addr, host_info->h_addr, host_info->h_length);
  server.sin_port = htons( [self port] );

  // We set the non-blocking I/O flag on [self fd]
  if ( ioctl([self fd], FIONBIO, &nonblock) == -1 )
    {
      AUTORELEASE(self);
      NSLog(@"TCPConnection: Unable to set the non-blocking I/O flag on the socket");
      return nil;
    }
  
  // We initiate our connection to the socket
  if ( connect([self fd], (struct sockaddr *)&server, sizeof(server)) == -1 )
    {
      if ( errno == EINPROGRESS )
        {
          // The  socket is non-blocking and the connection cannot be completed immediately.
          fd_set fdset;
          struct timeval timeout;
          int value;
          
          // We remove all descriptors from the set fdset
          FD_ZERO(&fdset);
          
          // We add the descriptor [self fd] to the fdset set
          FD_SET([self fd], &fdset);
	 
	  // We set the timeout for our connection
          timeout.tv_sec = [self connectionTimeout];
          timeout.tv_usec = 0;
          
          value = select ([self fd] + 1, NULL, &fdset, NULL, &timeout);

	  // An error occured..
          if ( value == -1 )
            {
	      AUTORELEASE(self);
              NSLog(@"TCPConnection: An error occured while calling select().");
              return nil;
            }
	  // Our fdset has ready descriptors (for writability)
          else if ( value > 0 )
            {
              int soError, size;
              
	      size = sizeof(soError);
             
              // We get the options at the socket level (so we use SOL_SOCKET)
              // returns -1 on error, 0 on success
              if ( getsockopt([self fd], SOL_SOCKET, SO_ERROR, &soError, &size) == -1 )
                {
		  AUTORELEASE(self);
                  NSLog(@"TCPConnection: An error occured while trying to get the socket options.");
                  return nil;
                }

              if ( soError != 0)
                {
		  AUTORELEASE(self);
                  NSLog(@"TCPConnection: connect failed.");
                  return nil;
                }
            }
	  // select() has returned 0 which means that the timeout has expired.
          else
            {
	      AUTORELEASE(self);
              NSLog(@"TCPConnection: The connection timeout has expired.");
              return nil;
            }
        } // if ( errno == EINPROGRESS ) ...
      else
        {
	  AUTORELEASE(self);
          NSLog(@"TCPConnection: A general socket error occured.");
          return nil;
        }
    } // if ( connect(...) )
 
  // FIXME - We clear the non-blocking I/O flag
  //nonblock = 0;
  
  //if (ioctl([self fd], FIONBIO, &nonblock) == -1)
  // {
  //   AUTORELEASE(self);
  //   NSLog(@"TCPConnection: An error occured while clearing the non-blocking I/O flag");
  //   return nil;
  // }
  
  return self;
}


//
// access / mutation methods
//

- (NSString *) name
{
  return name;
}

- (void) setName: (NSString *) theName
{
  RELEASE(name);
  RETAIN(theName);
  name = theName;
}

- (int) port
{
  return port;
}

- (void) setPort: (int) thePort
{
  port = thePort;
}

- (int) connectionTimeout
{
  return connectionTimeout;
}

- (void) setConnectionTimeout: (int) theConnectionTimeout
{
  if ( theConnectionTimeout > 0 )
    {
      connectionTimeout = theConnectionTimeout;
    }
  else
    {
      connectionTimeout = 30;
    }
}

- (int) readTimeout
{
  return readTimeout;
}

- (void) setReadTimeout: (int) theReadTimeout
{
  if ( theReadTimeout > 0 )
    {
      readTimeout = theReadTimeout;
    }
  else
    {
      readTimeout = 30;
    }
}

- (int) writeTimeout
{
  return writeTimeout;
}

- (void) setWriteTimeout: (int) theWriteTimeout
{
  if ( theWriteTimeout > 0 )
    {
      writeTimeout = theWriteTimeout;
    }
  else
    {
      writeTimeout = 30;
    }
}


//
// This method is used to return the file descriptor
// associated with our socket.
//
- (int) fd
{
  return fd;
}


//
// other methods
//

- (void) close
{
  if ( close([self fd]) < 0 )
    {
      NSLog(@"TCPConnection: An error occured while closing the file descriptor associated with the socket");
    }
}


//
// The current line length limit that we read from a socket is 4096 bytes
// including the null char.
//
- (NSString *) readLine
{
  return [self readLineBySkippingCR: NO];
}

- (NSString *) readLineBySkippingCR: (BOOL) aBOOL
{
  NSString *aString;
  char c, buf[4096];
  int i;

  bzero(buf, 4096);
  i = 0;

  while ( YES )
    {
      c = [self _readByte];
      
      if (! aBOOL)
	{
	  buf[i] = c;
	  i++;
	}

      if ( c == '\n' || i == 4094)
	{
	  break;
	}
      
      // We skip the \r
      if (aBOOL && c != '\r' )
        {
          buf[i] = c;
          i++;
        }
    }
  
  aString = [NSString stringWithCString: buf];

  if (aString == nil || [aString length] == 0 )
    {
      return nil;
    }
  else
    {
      return aString;
    }
}


//
// Read a string of size theLenght (excluding the null char).
//
- (NSString *) readStringOfLength: (int) theLength
{
  NSString *aString;
  char *buf;
  int i;

  buf = (char *) malloc( (theLength + 1) * sizeof(char));
  bzero(buf, theLength + 1);
 
  for (i = 0; i < theLength; i++)
    {
      buf[i] = [self _readByte];
    }
  
  aString = [NSString stringWithCString: buf];
  free(buf);

  if ( [aString length] == 0 )
    {
      return nil;
    }
 
  return aString;
}


//
// Read "theLength" bytes.
//
- (NSData *) readDataOfLength: (int) theLength
{
  NSData *aData;
  char *buf;
  int i;

  buf = (char *) malloc( theLength * sizeof(char));
  bzero(buf, theLength);
  
  for (i = 0; i < theLength; i++)
    {
      buf[i] = [self _readByte];
    }
  
  aData = [[NSData alloc] initWithBytesNoCopy: buf  
			  length: theLength];

  if ( [aData length] == 0 )
    {
      RELEASE(aData);
      return nil;
    }
  
  return AUTORELEASE(aData);
}


//
// Write a 'line' to the socket. A line is a simple string object
// terminated by a CRLF.
//
- (BOOL) writeLine: (NSString *) theLine
{
  return [self writeString: [NSString stringWithFormat: @"%@\r\n", theLine] ];
}


//
// Write a string to the socket. We currently write the cString representation
// of a Unicode string.
//
- (BOOL) writeString: (NSString *) theString
{
  char *cString;
  int i, len;

  cString = (char *)[theString cString];
  len = strlen( cString );

  for ( i = 0; i < len; i++ )
    {
      [self _writeByte: cString[i]]; 
    }

  return YES;
}


//
// Write bytes to the socket.
//
- (BOOL) writeData: (NSData *) theData
{
  char *bytes;
  int i, len;

  bytes = (char*)[theData bytes];
  len = [theData length];
  
  for ( i = 0; i < len; i++ )
    {
      [self _writeByte: bytes[i]]; 
    }
  
  return YES;
}

@end


//
// private methods
// 
@implementation TCPConnection (Private)

- (char) _readByte
{
  char c;
  
  if ( read([self fd], &c, 1) == -1 )
    {
      // The socket is non-blocking and there was no data immediately available for reading.
      if ( errno == EAGAIN )
	{
	  fd_set fdset;
          struct timeval timeout;
          int value;
	  
	  // We remove all descriptors from the set fdset
          FD_ZERO(&fdset);
          
          // We add the descriptor [self fd] to the fdset set
          FD_SET([self fd], &fdset);

	  // We set the timeout for our connection
          timeout.tv_sec = [self readTimeout];
          timeout.tv_usec = 0;

	  value = select([self fd] + 1, &fdset, NULL, NULL, &timeout);

	  // An error has occrured, we generate our read exception
	  if ( value == -1 )
	    {
	      NSLog(@"Error occured for read()");
	    }
	  // Our fdset has ready descriptors (for readability)
	  else if ( value > 0 )
	    {
	      //NSLog(@"We can read..");
	      read([self fd], &c, 1);
	    }
	  // select() has returned 0 which means that the timeout has expired.
	  // We generate an exception.
	  else 
	    {
	      NSException *anException;

	      NSLog(@"Timeout has expired for read()");
	      
	      anException = [NSException exceptionWithName: @"PantomimeReadTimeoutException"
					 reason: @"Timeout has expired for read()"
					 userInfo: nil];
	      [anException raise];
	      
	    }
	}
    }
  
  return c;
}


//
//
//
- (void) _writeByte: (char) byte
{
  if ( write([self fd], &byte, 1) == -1 )
    {
      // The socket is non-blocking and there was no room in the socket
      // connected to fd to write the data immediately.
      if ( errno == EAGAIN )
	{
	  fd_set fdset;
          struct timeval timeout;
          int value;

	  // We remove all descriptors from the set fdset
          FD_ZERO(&fdset);
          
          // We add the descriptor [self fd] to the fdset set
          FD_SET([self fd], &fdset);

	  // We set the timeout for our connection
          timeout.tv_sec = [self writeTimeout];
          timeout.tv_usec = 0;

	  value = select([self fd] + 1, NULL, &fdset, NULL, &timeout);

	  // An error has occrured, we generate our write exception
	  if ( value == -1 )
	    {
	      NSLog(@"Error occured for write()");
	    }
	  // Our fdset has ready descriptors (for writability)
	  else if ( value > 0 )
	    {
	      //NSLog(@"We can write..");
	      write([self fd], &byte, 1);
	    }
	  // select() has returned 0 which means that the timeout has expired.
	  // We generate an exception.
	  else 
	    {
	      NSLog(@"Timeout has expired for write()");
	    }
	}
    }
}

@end
