#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: dlregis.c,v 1.1 2000/01/10 03:54:20 knepley Exp $";
#endif

#include "mesh.h"

#undef __FUNCT__  
#define __FUNCT__ "MeshInitializePackage"
/*@C
  MeshInitializePackage - This function initializes everything in the Mesh package. It is called
  from PetscDLLibraryRegister() when using dynamic libraries, and on the first call to MeshCreate()
  when using static libraries.

  Input Parameter:
  path - The dynamic library path, or PETSC_NULL

  Level: developer

.keywords: Mesh, initialize, package
.seealso: PetscInitialize()
@*/
int MeshInitializePackage(char *path) {
  static PetscTruth initialized = PETSC_FALSE;
  char              logList[256];
  char             *className;
  PetscTruth        opt;
  int               ierr;

  PetscFunctionBegin;
  if (initialized == PETSC_TRUE) PetscFunctionReturn(0);
  initialized = PETSC_TRUE;
  /* Register Classes */
  ierr = PetscLogClassRegister(&MESH_COOKIE,      "Mesh");                                                CHKERRQ(ierr);
  ierr = PetscLogClassRegister(&PARTITION_COOKIE, "Partition");                                           CHKERRQ(ierr);
  /* Register Constructors and Serializers */
  ierr = MeshRegisterAll(path);                                                                           CHKERRQ(ierr);
  ierr = MeshSerializeRegisterAll(path);                                                                  CHKERRQ(ierr);
  ierr = MeshOrderingRegisterAll(path);                                                                   CHKERRQ(ierr);
  ierr = PartitionRegisterAll(path);                                                                      CHKERRQ(ierr);
  ierr = PartitionSerializeRegisterAll(path);                                                             CHKERRQ(ierr);
  /* Register Events */
  ierr = PetscLogEventRegister(&MESH_Reform,                "MeshReform",       MESH_COOKIE);             CHKERRQ(ierr);
  ierr = PetscLogEventRegister(&MESH_IsDistorted,           "MeshIsDistorted",  MESH_COOKIE);             CHKERRQ(ierr);
  ierr = PetscLogEventRegister(&MESH_Partition,             "MeshPartition",    MESH_COOKIE);             CHKERRQ(ierr);
  ierr = PetscLogEventRegister(&MESH_SetupBoundary,         "MeshSetupBd",      MESH_COOKIE);             CHKERRQ(ierr);
  ierr = PetscLogEventRegister(&MESH_MoveMesh,              "MeshMoveMesh",     MESH_COOKIE);             CHKERRQ(ierr);
  ierr = PetscLogEventRegister(&MESH_CalcNodeVelocities,    "MeshCalcNodeVel",  MESH_COOKIE);             CHKERRQ(ierr);
  ierr = PetscLogEventRegister(&MESH_CalcNodeAccelerations, "MeshCalcNodeAccl", MESH_COOKIE);             CHKERRQ(ierr);
  ierr = PetscLogEventRegister(&MESH_CreateLocalCSR,        "MeshCreateLocCSR", MESH_COOKIE);             CHKERRQ(ierr);
  ierr = PetscLogEventRegister(&MESH_CreateFullCSR,         "MeshCreateFulCSR", MESH_COOKIE);             CHKERRQ(ierr);
  ierr = PetscLogEventRegister(&MESH_CreateDualCSR,         "MeshCreateDulCSR", MESH_COOKIE);             CHKERRQ(ierr);
  ierr = PetscLogEventRegister(&MESH_LocatePoint,           "MeshLocatePoint",  MESH_COOKIE);             CHKERRQ(ierr);
  /* Process info exclusions */
  ierr = PetscOptionsGetString(PETSC_NULL, "-log_info_exclude", logList, 256, &opt);                      CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = PetscStrstr(logList, "mesh", &className);                                                      CHKERRQ(ierr);
    if (className) {
      ierr = PetscLogInfoDeactivateClass(MESH_COOKIE);                                                    CHKERRQ(ierr);
    }
    ierr = PetscStrstr(logList, "partition", &className);                                                 CHKERRQ(ierr);
    if (className) {
      ierr = PetscLogInfoDeactivateClass(PARTITION_COOKIE);                                               CHKERRQ(ierr);
    }
  }
  /* Process summary exclusions */
  ierr = PetscOptionsGetString(PETSC_NULL, "-log_summary_exclude", logList, 256, &opt);                   CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = PetscStrstr(logList, "mesh", &className);                                                      CHKERRQ(ierr);
    if (className) {
      ierr = PetscLogEventDeactivateClass(MESH_COOKIE);                                                   CHKERRQ(ierr);
    }
    ierr = PetscStrstr(logList, "partition", &className);                                                 CHKERRQ(ierr);
    if (className) {
      ierr = PetscLogEventDeactivateClass(PARTITION_COOKIE);                                              CHKERRQ(ierr);
    }
  }
  PetscFunctionReturn(0);
}

#ifdef PETSC_USE_DYNAMIC_LIBRARIES
EXTERN_C_BEGIN
#undef __FUNCT__  
#define __FUNCT__ "PetscDLLibraryRegister"
/*
  DLLibraryRegister - This function is called when the dynamic library it is in is opened.

  This one registers all the mesh generators and partitioners that are in
  the basic Mesh library.

  Input Parameter:
  path - library path
*/
int PetscDLLibraryRegister(char *path) {
  int ierr;

  ierr = PetscInitializeNoArguments();
  if (ierr) return(1);
  /*
      If we got here then PETSc was properly loaded
  */
  ierr = MeshInitializePackage(path);                                                                     CHKERRQ(ierr);
  return(0);
}
EXTERN_C_END

/* --------------------------------------------------------------------------*/
static char *contents = "PETSc Mesh library\n";
static char *authors  = "Matt Knepley    knepley@cs.purdue.edu\n\
  http://www.cs.purdue.edu/homes/knepley/comp_fluid\n";

#include "src/sys/src/utils/dlregis.h"

/* --------------------------------------------------------------------------*/

#endif /* PETSC_USE_DYNAMIC_LIBRARIES */
