# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.

# Embedded icon is taken from www.openclipart.org (public domain)

from core import models
from core.translation import _t, new

#---PIL
def init():
    #lazily import
    global Image, ImageChops, ImageFilter
    import Image, ImageChops, ImageFilter

def drop_shadow(image, horizontal_offset=5, vertical_offset=5, background_colour=(255,255,255,0), 
                shadow_colour=0x444444, border=8, shadow_blur=3, 
                force_background_colour=False, cache={}):
    """
    Add a gaussian blur drop shadow to an image.  

    image           - The image to overlay on top of the shadow.
    offset          - Offset of the shadow from the image as an (x,y) tuple.  
                        Can be positive or negative.
    background_colour     - Background colour behind the image.
    shadow_colour   - Shadow colour (darkness).
    border          - Width of the border around the image.  This must be wide
                        enough to account for the blurring of the shadow.
    shadow_blur      - Number of times to apply the filter.  More shadow_blur 
                        produce a more blurred shadow, but increase processing 
                        time.
    """
    #get info
    size        = image.size
    mode        = image.mode
       
    back        = None
            
    #assert image is RGBA
    if mode != 'RGBA':
        if mode != 'RGB':
            image   = image.convert('RGB')
            mode    = 'RGB'
        #create cache id
        id  = ''.join([str(x) for x in ['shadow_',size,horizontal_offset,vertical_offset,
                    border, shadow_blur,background_colour,shadow_colour]])
                    
        #look up in cache
        if cache.has_key(id):
            #retrieve from cache
            back, back_size = cache[id]
            
    if back is None:
        #size of backdrop
        back_size       = (size[0] + abs(horizontal_offset) + 2*border,
                        size[1] + abs(vertical_offset) + 2*border)

        #create shadow mask
        if mode == 'RGBA':
            image_mask  = image.split()[-1]
            shadow      = Image.new('L',back_size,0)
        else:
            image_mask  = Image.new(mode,size,shadow_colour)
            shadow      = Image.new(mode,back_size,background_colour)
            
        shadow_left     = border + max(horizontal_offset, 0)
        shadow_top      = border + max(vertical_offset, 0)
        shadow.paste(image_mask, (shadow_left, shadow_top, 
                                shadow_left + size[0], shadow_top + size[1]))
        del image_mask #free up memory
                            
        #blur shadow mask

        #Apply the filter to blur the edges of the shadow.  Since a small 
        #kernel is used, the filter must be applied repeatedly to get a decent 
        #blur.
        n   = 0
        while n < shadow_blur:
            shadow = shadow.filter(ImageFilter.BLUR)
            n += 1

        #create back
        if mode == 'RGBA':
            back  = Image.new('RGBA',back_size,shadow_colour)
            back.putalpha(shadow)
            del shadow #free up memory
        else:
            back        = shadow
            cache[id]   = back, back_size

    #Paste the input image onto the shadow backdrop  
    image_left  = border - min(horizontal_offset, 0)
    image_top   = border - min(vertical_offset, 0)
    if mode == 'RGBA':
        back.paste(image, (image_left, image_top),image)
        if force_background_colour:
            mask    = back.split()[-1]
            back.paste(Image.new('RGB',back.size,background_colour),(0,0),
                ImageChops.invert(mask))
            back.putalpha(mask)
    else:
        back.paste(image, (image_left, image_top))
    
    return back

#---Phatch
class Action(models.Action):
    """Drops shadow"""
    
    label       = _t('Shadow')
    author      = 'Stani'
    email       = 'spe.stani.be@gmail.com'
    cache       = True
    init        = staticmethod(init)
    pil         = staticmethod(drop_shadow)
    version     = '0.1'
    tags        = [_t('default'),_('border')]
    update_size = True
    __doc__     = _t('Drops a blurred shadow under a photo')
    
    def interface(self,fields):
        fields[_t('Horizontal Offset')]        = self.PixelField('2%',
                                                choices=self.SMALL_PIXELS)
        fields[_t('Vertical Offset')]          = self.PixelField('2%',
                                                choices=self.SMALL_PIXELS)
        fields[_t('Border')]                   = self.PixelField('4%',
            choices=[str(int(x)*2) for x in self.SMALL_PIXELS])
        fields[_t('Shadow Blur')]              = self.SliderField(3,1,20)
        fields[_t('Background Colour')]        = self.ColourField('#FFFFFF')
        fields[_t('Shadow Colour')]            = self.ColourField('#444444')
        fields[_t('Force Background Colour')]  = self.BooleanField(True)
 
    def values(self,info):
        #pixel fields
        x, y        = info[new('Pil','Size')]
        return super(Action,self).values(info, 
            pixel_fields={
                'Horizontal Offset' : x, 
                'Vertical Offset'   : y,
                'Border'            : (x+y)/2,
            })
        
    icon = \
'x\xda\x01\xb3\rL\xf2\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x000\x00\
\x00\x000\x08\x06\x00\x00\x00W\x02\xf9\x87\x00\x00\x00\x04sBIT\x08\x08\x08\
\x08|\x08d\x88\x00\x00\rjIDATh\x81\xd5\x99y\x8c]\xd5}\xc7?\xbfs\xdf{\xf3\x96\
\x99y\x9e}\xc6\xdb\x8c\xc7qm\x0701v\x026\xb6\xa9\x81@\x1a\x16\x11\xa1\x98\
\xd6\x14ZLD#$*UU\x97(U\xa2\xfe\x01\xa84\xffT\xa8\tA6\xb6\x90\xa2V\x854m\x80\
\xe0@Y\x0c\x1e\x07\xe3\x04\xdb\xb8\x10\xdbxa\xbc\x8cg<\xfb\xf6\xb6{\xef\xf9\
\xf5\x8f{\xdf2\xe3\x19c\xbb"U\xaf\xf4\xd39\xef\xdc\xf3\xee\xf9~\x7f\xdb\xf9\
\x9d{EU\xf9\xff|\x99\xffk\x00\xff\xdb+\xf2y=x\xdbZ\xa9\x17\xc3\x07\x02;\x1f\
\xda\xad\xdf\xfe\xbc\xd6\xf9\xdc,\x10\x8d\xb1\xe3\xce\xfbh\xff\xcaM\xfc\xd9\
\xb6\xf5\xf2\xe8l\xf3~r\x83\xd4>\xbbZ\xd2W\xba\xce\x15Y`\xc7F\x89\xab\xcb\
\xf7\x14\x0e\x9c\xee\xe2\xc5\xef\xab\xda\xca\xfb\xcf\xad\x93o\xaf\xd9\xc8]\
\xcd\xad\xd0\xdc\n#\x83<\xfd\xdc:\xf9\xed\x96\xdd\xfaVq\xce\xd3_\x97\xaa\xea\
1\x1e\xadm\xe6{\xf9a\xfa\x80eW\x82E\xae$\x88\xb7\xaf\x95\xb5\xf3:\xe9Z\xd8\t\
\x87?\xe4\xd4\xe0\x00Odj\xd9\xf1\xd8/4\xbf}\x83|\xb1}\x11\xfbo\xbe\x93Xq\xbe\
\xf5a\xe7\x7f0\xdas\x8e\x95g\xdf\xa1{\xfe\x8d<\xd0\xd8\xc8\x93+\xd7\xd0\xb6\
\xb0\x13\xf6\xee\x82#\x1f0\xff\xc1=z\xf6r\xb1\\\x91\x05\xd4\xb0\xe2\xea\xeb`\
\xee\x02Xv5\x0bO\x9d\xe0\x99\xff\xde\xcf\xe3\xcf\xad\x97\x1f\xd4\xd4\xf2\xe0\
\x8d\xb7\x10S\x0b\x95v\xd9x;\xe9\x97\x7f\xcak\xe9\xaf\xe3_\xfb\x15\x96.Y\x0e\
"\xc1\xbd\xd6\x05\xf0\xf1A6\x00\xff\xf2;!\x80rM}c\xd8\x17X\xd0\t\xf3;h\xe8=\
\xc5\x93(\xf8Y\x98\x18\x03\xaf \xa8\x051\x10\x89)\x1bo\xe6\x0b\x89\x1a\xa8\
\x8a\x87\xe4L@\xa2\xb5\r\xe0wH U\xc3\xaax\x02T\x03 n\x16rc\xc0\xa4ar\x10\x06\
\x0e\x19,\x10I(\x91\x04xY\xf0\xb2\x82\x01\x12s,\xa9\x06H\xd5[\xe2\xb5\x10M@,\
\x0e\xf5\xcd\xdcv%X\xae\x88@]#W\xab\x85B\x162\xc3\xc2\xf0\x19ar\xc8\x90\x9c\
\xa7\xd4\xadR\x16-\xf6\x88$\x01\x13\x07\xa7\x1e\xbc\xf3\xa0\x1e^\x06F\x8e\
\x1bF\x8f\t\x83\x07"\xa4\xea-u\xf3\x95d\x9d\xd2\xd4L\xe7\xb6\xf5\xf2\x80\xb1\
D\x80\x9a\x92\x08\x8e\x9b\xe3\xfb\x8f\xfcZ3\x97E\xe0\x85M\xe2L\x9c\xe5q+4\
\x88\xe0\xa08\x08\x8ecH~\xf1K\xa42\xc30\xf8\xa9a\xf4\x9c\xa1\xeejK\xfb\xbd\
\x1eN\xcd\x17\xa0v#\xc4\x16@t>D\x8a~f\xc1\xed%R8C\xe3\x82n\x1aW\xbe\x8a\x9f\
\x19\xe0\\\x97\xe1\xcc\x01\x87t\x9be\xf1"\xcb\xdcy<\x1f\x8dC4V\x96\xbdoC\xf7\
\t\x8e\x03\xcf\xcc\x84s\xd6,\xb4u\x9d|\xb5m.\xaf}\xed\x1b`\x9c\xc0WU\xc1\xba\
01\x08\xe7\x8f\x1a4.t\xdc\xe1\x13k\x98\x07\xf5\x7f\x0c\xd57^\x9a\t\xd5\x85\
\xd1W`\xf8E\n#c|\xfa\x8a\x83)(-K,\xc9\x06p\xa2\xe5\xa9\x07\xf7\xc1o\xbaX\xbb\
\xa5K\x7fuY\x16Hx\xec\xad\xa9\xc5w"8\xc51\xeb\xc1x\xbfp\xeecCr\xa1\xb2\xf0v\
\x1fi\xf8C\x98sO\x10\xa9\x9a\x03\xa4\xa8\x9b\x19\x9eZq/}\'\xd4\xdcL,\xf1#\
\x96\xdc\xd7\xc5\xa9\xd7\x0cg\x0f9\xb4^e\xa9i\xd2\x12\x89\x81>\xdc\xc94\x1f\
\xcc\x86s\xd6\x9d\xf8\xfe\xf7t\xac\xe7\x0c\xef\x13\x1a\xc8\xfa\x90\x19\x86\
\xde\xa3\x86\x9a\xa5J\xfb\xd7,\x92\xbe\x15\xd2\x7f\x00\x9a\x07\x9b\x05\x9b\
\x0b\xdb\x99\xa4\xe2\x9e\x86"\x02M\x8f \x89\xe5\xb4\xdfn\xa9Y\xaa\xf4\x1d1d\
\x86\x83\xf5B\x02\x07\x1f\xfb\x85\xe6/\x9b\x00\xc0\xe4\x04/\r\x9c\x0f\\\'?\
\x01\xfd\xc7\x0c\x91:\x98\x7f\x93\x85\xc450\xe7\x1beP~ \xeagQ?\x17\xb6\x17Ji\
n\x89\x8c\x0b\xcd\x8fB\xb4\x95y7YL\x1a\xfa\x8f\x1b\n\x930>\n\x93\xe3\xec\xba\
\x18\xc6\x8b\xd7B\x96\x9dg\xba\xc1\xcf\xc3\xe89C./t\xde\xe5\x03\x02\xf5\x9b\
\x02\xcd\xfbY\xd4\x9b\x0e4\x83z\x99\x0b\xc1O#W"\x82@\xdd7\x01X|\xb7O.#\x8c\
\xf4\x18\xfaN\x03\x86s\x17\x83x\xd14\x1a\x85\xf3g?\xa5\xb0d1\xb1\xe1S\xc2\
\xdc[,&\x02\x1ai\x055A\x82W\t\xbdL@ajJ\x90\xa9})\xb7\x02\xa8\x84?E 6\x17$\
\x82\x89z\xb4\xdeh\xe9{\xc7\xd0z5\xac^\xc3\x0f^\xbcS\x1e\x1a\x1b\xe6\x9f\xbc\
<?\x99\x9eNg\xccB[o\x92E\x11\xe5o:\x97\xb2e\xf95D\xbdaal\xd4\xb0t\xb3\x0f\n6\
y\x03\xd4\xde\x1a\x00\xd6\x008"\xa8RbP"\x05H\x05\xe8"\'\x11)\x83\x97\xb0\xac\
\x18\xdc\x01\xf9\x13\x00\x1c~\xde!\xdd`i\\\xacDbp\xea$\x1c>\xc4\xe4\xd9n\xb6\
E|\x9e*\xd6MS,\xb0c\xad,31\xbe\xbb|9\x9bW\xac\xc6\xd4\xa4!;\x06\xa7\x8e\x08M\
k-j\xc1Z\x10\xd3\x80Vj_C\xf0\xc5K+\xb5/\x81\xa6+\x86\x03\xed\x97\t\x88\tu\
\xe0\xb4 \x04\x04\x9aVY\x06\xde\x17\xe6\xccW\xa2U\xd0\xde\t\xed\x9d\xa4\xf6\
\xee\xe2\xcf?:@\x06\xf8\xce\x05\x04\xac\xe1\xcd;\xee\xa5\xad\xa95\\\xd0\x077\
#\xe4\'\x84t\x87\xc5Z\xc0\x82\xcd\xf7B\xb4#\xb4\x00\xe56\x04\xad\xd3\x08\x14\
\x9b\xe2\x88\x1aAB\xf0*\x80\x1fda\x9b\xebE\xc2\xfe\x9c%\xca\xd9\xff2\xb8\x19\
!\x9eR$L\xe6\x85\x02\xd6\xc2\x9bE\xccSc@8\x19\x89\xd2V"\xe4\x07\x16H\xb4\x01\
NP\xf7X\x1f${\x1a\xe2\xd7N\x05_t\'@\xc3x\x08}\xa5DC\x8b\xae\xe2K\xd8\x0f\xb4\
\x8f\x80ZE\xb3g1!K\'\x06\x89\xb9\xc1\xfa\xa9zp\x1c8\xfa\x11\x1c\xfb\x88\xef|\
\xabK_\x9f\x91\x80\xfa\xdc\xf5\xc6\xcb\xbc\x7f\xc77Y\x9cH\x06`\xf3\x13B\xbcQ\
\xa1X\x1e+\xf8\xb9~\x8c;\x86j4\xc0i\x03\xbc\xa5x\x98a\x13\xd3"\x91\n\x9fW\
\x11\x8c\x04\xf9\x00\xaf\x1f\xf5\x0b`\x08\xc6\x04\x12\xcdJ~@\xb0\xbe2\xd4\
\x0b{w\xf1\xd3\x87\xba\xf4\xa9\xca\xe7NI\xa3\x0f\xef\xd1\xa1\xd1any\xfd%\x06\
\xdcB\x00\xd8\xcbC4\x15\xf8~\xd1\x02j\x15od\x1f6L\xa1\xd6\xcb\xa2n.\xe8\xbbe\
\xf1C\xb1n\x0e[\x08\xc7\xbd,\xea\x86\xa9\xd7\xcb`\xbd\x0c\xeaN`G\xf6\xa1\xfe\
\xd4ub\xa9`\xfd\xdc$\xbc\xbd\x93#\xc4\xf8\x93\xe9\x8a\xb9`\x1fx\xb8K\xbb\x07\
\xfa\xb8\xf5\xadW\xc9x\x1e\xf8.D\x92Z\x01>h\xedd7\xfe\xd8\xc7%P\xd6\x0b\xc1z\
\xb92\x89B\xb6\x0c<\x04]\x9a_\x9c\xe3\xe5p\x87\xde\xc3\xcf\r\x04\x16.\xae\
\xa3\x10I)\xbe+\xf4\xf6\xc0\xc4\x08/<\xf0K\x9d\xfcL\x02\x00[\xde\xd5\x83g\
\xba\xf9\xd9\xc8`h\xfe0\xcb\xa8\x86\xe0\xc3l\xe4\x8f\x1d\xc5\x9f\xe8.\x03t\
\xa7\x02\xd4ic\xbe\x9b\xc3/L\xb3\xd2\xc8\x87\xd8lO\xf0\xcc\xa2\x054H ~\x18S\
\x0b:\xe0Kk\xf8\xee\xb6\xf5\xb2y:\xd6\x197\xb2\xe7\xd6\xc9\xfd\xabo\xe4\xfe\
\xc6\x16\xc8\xf5\x817Y\xd6~)`C\x12:v\x18\x9c\x1e\xa4j\x01\x98T)\x9dj\xc5\x06\
W\xcc8\x10\xf8\xbe5\xa0\xee\x10\xe4O\x81f1\xa1\x1aU\xc2\xe7\x86\xf3\xbcqp\
\xa2\x8aq`\xe5\xf5Hf\x82\xe7\xb7\xae\x93\xfeo\xed\x9e%\x88\x01\xb6o\x90\xeb:\
\x96\xf0\xdc\x8a\xd5\xc1I+R\x05\xeex\xb8\xc0\x948\xa8\xe8{ch\xe6#\x88\xa4\
\x91\xc8\x1cT\xe2\x88\x89\x05\xd9\xa8H\x00Em\x1e4\x87\xe6\xfb\x112\x88\tJuK\
\x10\xb86\x0cn\x91\xe0\xb9\xeeD\xb0\xbe\x84\x04\xd7\xde\x8c\x93\xcb\xf0\x9f\
\xdb7\xc8\xba\x87\xde\xd1\x0f. \xf0\xecFil\xae\xe3\xe5\xf5_\r\xde(\x88\x03U)\
ex\xd0\x94\xb4_\x12;\xd5\xdc\xd6\x82\xe6F\xb1v4\xc8J*\x88\x89\x82\x89\x04\
\x87\x08u\x91\xf0\x0c\x8c\x01c\x02\xff\r\x8f\xc6\xa5\x14\xabR&\x94\x1d\x10\
\xe6\xa4mi\x0f\x10\x81\x15_&\xd1}\x82\xad\xc0u\x17\x10\x88\xb8\xfc\xf3\xca5\
\xb4E\xc3Z\xdcq \x91\x86\xde\xa3!\x86\n+\xa8RJ\xad\xd62%\x83\x04\x04\x15\xfc\
\x02P(-.\x1a\xb6\x81\x17\x965o+\xc0K0\xe6\xe7!\xdf\x0b\x89\x85\x01\x0e\x80\
\xb1\x118\xb8\x0f_\x85\x7f\x9f\xcd\x85\x8e\x1d?\x02M\xad\x90L\x05\x16\x88%\
\x95x\x8d2qFH\xce\xd3\xb2\xff\xfbeK\x94HT\x8cSa-\x91\xd0\rl\xe02E\xcd\x0b\
\x04;/\xc1^`-8&\xf8\xefx\xb7PU\xab\xc4\x92\xca\xf8x\x00\xfc\xf8\x11\xb6S\xe0\
\xf1\x87\x7f\xa5\x9f\x16\x01_P\xccm]/\xab"\x0e\x8f\xb5/b\xf3\xb2k\x8865C\xff\
qad\xc80\xf76?\xd0\xb4\x0f\xbeG\xb9\xef\x97\xfb\xc5LU./\xca\xe6\x97\xd0u\xc4\
\t]\xc8\t<\xcc8SE"p\xfa\x15\x87\xfa&\xcb\xb1\xd3\xea\x9f8\xca6\xcf\xe7\x89\
\x87\xbb\xb4{\x9a\xc2g?\x13?\x7f\xbd4xQ\xb6\xd4\xd5\xf3\x177\xac\xa5m\xe4\
\x84Cz\xa5%1W\x83t\xe7\x05\xc0mH\xc4/\x06v1\xcd\xfa\xd3\x16\x92\x10\x9c\t\
\xc4\x89\x80q\x84H\xa2\x0e\xa7*\x85\xda1DGq"0yZ\x18;`X\xb8\xd2\xe7\xed7\x19\
\xb8\xe3\x05m\x9a\x11$\x179\xd0<\xb8W\x07\xb7\xec\xd6\x7f\x1c\x19\xe4\xdf\
\x92i\xa8oW\x06\x0f\x98\x00pE:\x85\x19Rkq\xc3\xf3\xcaR\x99\xbdP\x10\xa7\x16\
\'\xd5\x89I-B\x12\xf31\xb1\xa6\xa0L\xc9\xc3\xd0~C}\xbbRU\x03s\x17\xd2\xb8u\
\x83\xac\xb8l\x02\xc5+U\xc3\xba\xda\x06H\xb7X\xaa\xab\x95\xfe=\xa53\xfe\x05\
\xbe^\x19\x0fSK\x8f\n\x0bY0U\xcd8\xd5\x8bp\xaa;1\xc9\x85H\xb4\x1e\xebN\xa0\
\x16\xfa\xf78T\xd7*\xe9\x16\x8b\x13\x83\xf9\x1d`,\xb7_\x11\x81gWK\xb4\xa1\
\x85\x15"PU\x03\x8d\x9d\x16\x93\x85\x91\x8f\x83\x02^\x98\xea\xe7\x01+\xa6\
\x94\x04\xc5\x18)f,\x13o\xc1I-\xc0\xa9\xee\xc0\xa9\xe9\xc4Dk\xf1\xc7\x8fcs}\
\x8c|d\x90|\xb0NUM\xe0v\x8d-\x90Lq\xd7\x15\x11\x88\xc4\xb9\xb6\xa9\x95(\x04~\
\x9b\xac\x83\xe6\xa5\x16\xf7\x8c0|\xd0\x04\xe7\x03\xca\x16\x08\xf1\x97\xc7\
\x8a\xb1\x10\xba\x9c\xa9\xaa\xc7$[p\x12-H\xbc\x11\xebN\x92=\xbb\x93\xc2\xc8a\
\x86\x0f\x08n\x8f\xd0\xba\xd4\'U_\xde\xbcB\x12k~\xb8Q\xaag\xc4x1\x02\x08\xd7\
\xcf\xa9+\xfft\xa2P\xd3\xa4\x98k|\xce\x7fb\x18\xda\xeb\x90^a\x91\xc8\xcc\x89\
`\xfa\xf9X\xc5\xa0n\x06w\xf4(\xee\xf0~\xd4\x1dD\x0b.\xa3\x87\x1c\xe2\xa2\xc4\
Z|zzA\xfb\xc2l\xa6\xe0\xe6\xe1\xf4\xa7L\xc6-\xb5\xc0\xc4\xe5\x11\xb0\x14\xf6\
\xed\x81\xbe\xde\xe0A\x85\x02\xea\x16\xc8zy\xdc\xe5W\xd9tm\xd60\xf4\xaeC\xd5\
<K\xbc\xbd\xe2]\xba\x94\xb3N\xf9\xe0\xae\xd8\xec\x00\x9e;\x80\x8d\x04\xec\
\xf2\xa7\x0c\xf9\xb3\x0eu\xf3,\x8d\x8b,\xbbw1\xdes\x8a;\xac\xe2\x19\xf0\x00\
\xd7\x07\xcfw\x18|d\xb7\xce\xf8v\xe23?p\xecX+\xcb\x0c\x8c{\x86\xb1-]L\xa0\
\xaa;\xd6H\xc7U_\xe6\xe4\x8a\xeb\xca/w\xc7\xfb\r\x91:%\xd2\xa88\xd5\x16K\xd9\
\x8dJ\xe5\x03\xa0\xe3\x86\xc2\xa0\xe0\x0f\x0b5M\xe5\x97\xbb\xb1\x04\xbc\xb0\
\x83\xaeM\xaf\xea\xba\x8b\x02\xba,\x0b\x00\x7f\xbaG\x0f\x97\xd8\x8a\x18D\x0c\
\xd0\xf3\xd2\x02\xce\xc7\x924G\xe3Jr\x8e\x92\x1b\xb3L\x0c\x1b&\xfb\x84\xcc\
\xd1\x08\xbe\x05\x89)&\x06Z\x00-\x08\xc6@<\xad\xd45*\xd5\x9d~\xe9\xf5\xba\
\x18\xc8eat\x94\xfd"\x12%\xd8\xa8\xad^\xc2\xe7\xa3\xcf$ R\xd4\x1fN(Q r\xe6\
\x1c\xef\xfa>\xf7F"\xc1\xc9)\x9a\x80D\x9de$\x13\xd4O\xad\xf3\xc0\xcf\x0b\xbe\
\x1f\xd42N\x95\xe28\xf0\xeb\xf7\xb0d\xe0\xaa&L,U\xfeJ3\xd4\x0f\xa3\x1e\x9f\
\x00I\x02\xf7\xf1D\xc4\x07\xfc\x8b\x11\x99\x95@\x05p\x13\xce\x8b\x86\x12\x03\
b\xe7\'\xd8\xdf\xd7\xc3\xbd\xf3\x16\x06 \xbaO\xc2{]\xf4\xbc}\x8c\x1f.\xaf\
\xe7\xb6\xcd\x7f\xc4\x86\x86\xd6\xf2\xbab\xe0\x93\xdf\xc2\xce\x0f\xf8\xf1\
\xae^\xde\xb8\xfb\x10\x9bV-\xe7\x9e\x15\xab\x885\xcf\x85\xc1\x0189\xce\xc9\
\x90\x80KP\x05\xba!\x94Y-2#\x81iZ\x9f\x02\xbc(\xfb\x86\xf8p\xed\x87\x81\x8f\
\xfff/#o}\xcc\xf6g\x8e\xf2F\xde\xc7\xd4\xc58\x9e\xae\xa6c\xd3f\x16&S\xc13GG\
\xe0\x97\xafs\xf0\xa9\x0fy\xd5\x07w\xcfy\x9e]~\x94\xe7\xef;\xc4=+:\xb8od\x90\
\xe8\xce\xd3t\x03U\\\xf8V\xc0#,\x86/\xc0:\x93u$\xf0s\xa7B\xf3\x95\xe0\xab\
\x8a\xfd\x7fX\xc5\xdf\x8d\xba\x0c<{\x84\x9f\x0f\xe4\xd1p<\n\xc4\x96\xcd\xa1\
\xf9\x89\x8d\xfc\xfd\xdd\x9bH\t\xf0\xb3\x7fe\xe2\xaf\xdf\xe0\xafNL0P\xa1\xdd\
<Ph\x8bc\x16\xa4\x88\xbc?\xc8\xb9\xf0^>l\x0b\x15s\xad\xaaN\xab\xb0f\' \x15\
\x04\x8a\xa0\x8a\xc0K\x04f\x91h\xf8\xbf\xc8\xadm\xfc\xde_\xfe>\x7f\x1b\x89b\
\x9e\xde\xc5\x93?\xef\xe6 e\xf7(T\xf4+\x01\xe7\xa7\xfdv+\x08\\\x9a\x05\xa6\
\x91pf\xd0~\xb1\x8d2\rt\xd8/\xc6\x8e\xdc\xbf\x98\xd5\xd5\x11\xaa\x7f|\x847\
\x00\x9f0\xbfW\xb4E\xd0E\x8bT\x12p\t\x82x\xca\x87\xf4K"0\x8dH\xa5KMw\xa7b\
\x8cD*\x08\x87\xef\xdb\xcao\x13C\xf0\xb6\x82\xc4t\x02\x95\x1a\xf7\t4>+\xf0K&\
0\x03\x99\xca\xec\xe4\xcc \xa6\xa2-^\xb6\x02\xfcLR\xbc\xaf\x97\x92\xfb+\xaf\
\xff\x01iXg\x97r\xae{9\x00\x00\x00\x00IEND\xaeB`\x82\x15m\xac\x0e'
