// ------------------------------------------------------------------------- //
// $Id: x11win.cpp,v 1.75 2004/01/15 22:49:11 weismann Exp $
// ------------------------------------------------------------------------- //

/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

// ------------------------------------------------------------------------- //

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if STDC_HEADERS
# include <stdio.h>
#endif

#if HAVE_STDARG_H
# include <stdarg.h>
#endif

#if HAVE_X11_XLIB_H
# include <X11/Xlib.h>
#endif
#if HAVE_X11_XUTIL_H
# include <X11/Xutil.h>
#endif
#if HAVE_X11_XRESOURCE_H
# include <X11/Xresource.h>
#endif
#if HAVE_UNISTD_D
# include <unistd.h>
#endif
#if HAVE_Gl_GLX_H
# include <GL/glx.h>
#endif
#if HAVE_X11_KEYSYM_H
# include <X11/keysym.h>
#endif

#if HAVE_STRING_H
# include <string.h>
#elif HAVE_STRINGS_H
# include <strings.h>
#endif

#include "common.h"
#include "image.h"
#include "node.h"
#include "nodefactory.h"
#include "x11win.h"
#include "../data/pointless.xpm"

// ------------------------------------------------------------------------- //

XWindow::XWindow() : MainWindow()
{
	_width	= 0;
	_height	= 0;
	_fullscreen = false;
	_display = 0;
	_window = 0;
	keymap_init();
	_attr.background_pixel = 0;
	_attr.border_pixel = 0;
	_attr.override_redirect = False;
	_attr.event_mask   = StructureNotifyMask | ExposureMask | KeyPressMask |
		KeyReleaseMask | PointerMotionMask | ButtonPressMask | 
		ButtonReleaseMask;
} 

XWindow::~XWindow()
{
#if HAVE_XVIDMODE
	set_resolution(_original_resolution);
#endif
	// hide a distroy the window
	if (_window) {
		XUnmapWindow(_display, _window);
	}
	if (_window) {
		XDestroyWindow(_display, _window);
	}
}

void XWindow::set_wm_hints(WindowGeometry &geometry) 
{
	assert(_display != 0);
	#define icon_width 80
	#define icon_height 55

	XSizeHints size_hints = { 
		/* PMinSize | PMaxSize | PPosition  | */ PSize | USSize,
		400, 100,		/* x, y */
		_width, _height,/* width, height */
		32, 32,         /* min_width and min_height */
		4096, 4096,     /* max_width and max_height */
		0, 0,           /* width and height increments, not set */
		{0,0},{0,0},    /* aspect ratio, not set */
		0, 0,           /* base_width, base_height added by ICCCM version 1 */
		0,              /* win_gravity added by ICCCM version 1 */
	};
	XWMHints wm_hints = {
		InputHint | StateHint | IconWindowHint,
		True,           /* input model */
		NormalState,    /* starts up in normal state */
		0,              /* icon pixmap - set later */
		0,              /* icon window - created later */
		150, 2,         /* icon position of icon */
		0,              /* icon mask pixmap  - not used */
		0,
	};
	int screen = XDefaultScreen(_display);
	if (geometry._using_placement) {
		if (geometry._x_sign < 0)
			geometry._x = DisplayWidth(_display, screen) +
				geometry._x - size_hints.width;
		size_hints.flags |= USPosition;
		size_hints.x = geometry._x;
		if (geometry._y_sign < 0)
			geometry._y = DisplayHeight(_display, screen) +
				geometry._y - size_hints.height;
		size_hints.flags |= USPosition;
		size_hints.y = geometry._y;
    }

	Window root = RootWindow(_display, screen);
	XIconSize *size_list;
	int count = 0;
	// get available icon sizes from Window manager 
	if (XGetIconSizes(_display, root, &size_list, &count) == 0)
		log_debug("Window manager didn't set icon sizes - using default");
	else {
		;
		/* A real application would search through size_list
		 * here to find an acceptable icon size, and then
		 * create a pixmap of that size.  This requires
		 * that the application have data for several sizes
		 * of icons. */
	}

	Pixmap icon_pixmap;
	icon_pixmap = XCreateBitmapFromData(_display, 
			_window,(const char *)icon_bitmap_bits, icon_bitmap_width, 
			icon_bitmap_height);

	wm_hints.initial_state = NormalState; // normal or icon when first mapped
	wm_hints.input = True;	// does the application need keyboard input
	wm_hints.icon_pixmap = icon_pixmap;
	wm_hints.flags |= StateHint | IconPixmapHint | InputHint;

	char *window_name = (char *) PACKAGE;
	char *icon_name = (char *) PACKAGE;
	XTextProperty windowName, iconName;
	if (XStringListToTextProperty(&window_name, 1, &windowName) == 0) {
		log_abort("Structure allocation for windowName failed");
	}
	if (XStringListToTextProperty(&icon_name, 1, &iconName) == 0) {
		log_abort("Structure allocation for iconName failed");
	}

	XSetWMProperties(_display, _window, &windowName, &iconName, 
			0, 0, &size_hints, &wm_hints, 0);

}


void XWindow::create_window(WindowGeometry &geometry, bool fullscreen)
{ 

	_width  = geometry._width;
	_height = geometry._height;
	_geo_width = geometry._width;
	_geo_height = geometry._height;
	unsigned int display_width, display_height;

	if ( (_display = XOpenDisplay(NULL)) == NULL ) {
		log_abort("Unable to connect to X server %s",
				XDisplayName(NULL));
	}

	int screen = XDefaultScreen(_display);
	display_width = XDisplayWidth(_display, screen);
	display_height = XDisplayHeight(_display, screen);

	if ((_width > display_width) || (_height > display_height)) {
		log_abort("Unable to get a window on %s with (width x height)=%dx%d", 
				XDisplayName(NULL), _width, _height);
	}

	XVisualInfo *visinfo;
	static int attrib[] = { 
		GLX_DOUBLEBUFFER,
		GLX_RGBA,
		GLX_RED_SIZE,   4,
		GLX_GREEN_SIZE, 4,
		GLX_BLUE_SIZE,  4,
		None,
	};

	int dummy;
	if(!glXQueryExtension(_display, &dummy, &dummy)) 
		log_abort("X server must support GLX to run %s", PACKAGE);
	// ask GLX to provde us with a visual
	visinfo = glXChooseVisual(_display, screen, attrib);

#if 0
	// BEGIN: THIS IS JUST FOR TESTING PORTABILITY !!!
	if (!visinfo) {
		// lets drop the RGB display (hack for one of our SPARC stations)
		static int attribMinimal[] = { 
			GLX_DOUBLEBUFFER,
			GLX_RED_SIZE,   4,
			GLX_GREEN_SIZE, 4,
			GLX_BLUE_SIZE,  4,
			None,
		};
		visinfo = glXChooseVisual(_display, screen,attribMinimal);
		if (!visinfo) {
			log_abort("Unable to get visual");
		}
		log_warning("Insufficient OpenGL capabilities");
	}
	// END: THIS IS JUST FOR TESTING PORTABILITY !!!
#endif

	Window root = RootWindow(_display, screen);
	_attr.colormap = XCreateColormap(_display, root, visinfo->visual, AllocNone);
	_attr.override_redirect = True;
	int attrmask = CWEventMask | CWColormap | CWBorderPixel;
	_fullscreen = fullscreen;
	if (fullscreen) {
		_width = XDisplayWidth(_display, screen);
		_height = XDisplayHeight(_display, screen);
		log_debug("Toggle from window mode into fullscreen (%d,%d)", 
				display_width, display_height);
		/* Create a window without frames the hard way (required for KDE/GNOME)
		 * but may cause problems with the focus of other applications) simply
		 * by telling the xserser to bypass the window manager */
	}
	_window = XCreateWindow(_display, root, 0, 0, _width, _height, 0, 
	                        visinfo->depth, InputOutput, visinfo->visual,
	                        attrmask, &_attr); 

	set_wm_hints(geometry);

	if (fullscreen) {
		_attr.override_redirect = True;
		if (!XChangeWindowAttributes(_display, _window, 
					CWOverrideRedirect, &_attr)) {
			log_abort("Unable to change window attributes");
		}
		XMoveResizeWindow(_display, _window, 0, 0, _width, _height);
		XMapWindow(_display, _window);
		XRaiseWindow(_display, _window);
		XReparentWindow(_display, _window, DefaultRootWindow(_display), 0, 0); 
		XSetInputFocus(_display, _window, RevertToParent, CurrentTime);
		if (XGrabKeyboard(_display, _window, False, GrabModeAsync, 
					GrabModeAsync, CurrentTime) != GrabSuccess) {
			log_abort("Unable to grab keyboard");
		} 
		if (XGrabPointer(_display, _window, False, ButtonPressMask, GrabModeAsync, 
					GrabModeAsync, _window, None, CurrentTime) != GrabSuccess) {
			log_abort("Unable to grab keyboard");
		} 
		XMoveWindow(_display, _window, 0, 0);
		XSync(_display, False); 
	}
	else {
		XMapWindow(_display, _window);
		// wait till the window appears on the screen
		XEvent event;
		int ready = 0;
		do {
			XNextEvent(_display, &event);
			if (event.type == Expose && !event.xexpose.count)
				ready = 1;
		} while (!ready);
	}

	GLXContext ctx;
	if ( (ctx = glXCreateContext(_display, visinfo, NULL, True)) == NULL ) {
		XDestroyWindow(_display, _window);
		XCloseDisplay(_display);
		log_abort("Unable to create an OpenGL rendering context");
	}

	if ( glXMakeCurrent(_display, _window, ctx) == False ) {
		XDestroyWindow(_display, _window);
		XCloseDisplay(_display);
		log_abort("Unable to make our rendering context current");
	}

#if HAVE_XVIDMODE
	int j, vm_count;
	XF86VidModeModeInfo **vm_modelines;
	XF86VidModeGetAllModeLines(_display, 0, &vm_count, &vm_modelines);
	for (j=0; j<vm_count; j++) {
		std::string name = to_string(vm_modelines[j]->hdisplay)+"x"+
			to_string(vm_modelines[j]->vdisplay);
		_videomodes[name] = vm_modelines[j];
	}
	_original_resolution = get_resolution();
#endif
}

void XWindow::swap_buffers() 
{ 
	glXSwapBuffers(_display, _window);
}

#if HAVE_XVIDMODE
void XWindow::set_resolution(std::string r)
{
	if (_videomodes.find(r) == _videomodes.end()) {
		log_warning("Unable to switch to mode '%s'", r.c_str());
		return;
	}
	log_debug("Switching to resolution '%s'", r.c_str());
	int screen = XDefaultScreen(_display);
	XF86VidModeSwitchToMode(_display, screen, _videomodes[r]);
	XFlush(_display);
	if (_fullscreen) {
		resize_window(_videomodes[r]->hdisplay, _videomodes[r]->vdisplay);
	}
}
#else
void XWindow::set_resolution(std::string r) 
{
	log_warning("Unable to set resolution without XVIDMODE support");
}
#endif

#if HAVE_XVIDMODE
std::string XWindow::get_resolution() 
{
	int dotclock;
	XF86VidModeModeLine modeline;
	XF86VidModeGetModeLine(_display, 0, &dotclock, &modeline);
	return to_string(modeline.hdisplay)+"x"+to_string(modeline.vdisplay);
}
#else
std::string XWindow::get_resolution() 
{ 
	return "Unable to get resolution without XVIDMODE support"; 
}
#endif

#if HAVE_XVIDMODE
std::string XWindow::get_resolutions()
{
	std::string result = "";
	for (_videomodes_::iterator i = _videomodes.begin();
			i != _videomodes.end(); ++i)
	{
		if (!result.empty()) result += " ";
		result += i->first;
	}
	return result;
}
#else
std::string XWindow::get_resolutions() 
{ 
	return "Unable to get resolutions without XVIDMODE support"; 
}
#endif

void XWindow::toggle_fullscreen()
{ 
	unsigned int display_width, display_height;
	int screen = XDefaultScreen(_display);
	std::string resolution;

	if (_fullscreen) {
		display_width = _geo_width;
		display_height = _geo_height;
		_fullscreen = false;
		_attr.override_redirect = False;
		if (!XChangeWindowAttributes(_display, _window, 
					CWOverrideRedirect, &_attr)) {
			log_abort("Unable to change window attributes");
		};

#if HAVE_XVIDMODE
		resolution = _original_resolution;
#endif
		log_debug("Toggle from fullscreen into window mode (%d,%d)", 
				display_width, display_height);
	}
	else {
		_fullscreen = true;
		display_width = XDisplayWidth(_display, screen);
		display_height = XDisplayHeight(_display, screen);
		resolution = to_string(_geo_width)+"x"+to_string(_geo_height);
		log_debug("Toggle from window mode into fullscreen (%d,%d)", 
				display_width, display_height);
		/* Create a window without frames the hard way (required for KDE/GNOME)
		 * but may cause problems with the focus of other applications) simply
		 * by telling the xserser to bypass the window manager */
		_attr.override_redirect = True;
		if (!XChangeWindowAttributes(_display, _window, 
					CWOverrideRedirect, &_attr)) {
			log_abort("Unable to change window attributes");
		}
	}
#if HAVE_XVIDMODE
	int dotclock;
	XF86VidModeModeLine modeline;
	XF86VidModeGetModeLine(_display, 0, &dotclock, &modeline);
	bool truncated = false;
	if (display_width > modeline.hdisplay) {
		display_width = modeline.hdisplay;
		truncated = true;
	}
	if (display_height > modeline.vdisplay) {
		display_height = modeline.vdisplay;
		truncated = true;
	}
	if (truncated) {
		XF86VidModeSetViewPort(_display, screen, 0, 0);
	} 
#endif
	log_debug("display_height %d, display_width %d", 
			display_height, display_width);
	resize_window(display_width, display_height);
}

void XWindow::resize_window(int width, int height)
{
	XMoveResizeWindow(_display, _window, 0, 0, width, height);
	XMapRaised(_display, _window);
	XRaiseWindow(_display, _window);
	XReparentWindow(_display, _window, DefaultRootWindow(_display), 0, 0); 
	if (_fullscreen) {
		Window dummyw;
		int revert;
		XGetInputFocus(_display, &dummyw, &revert);
		XSetInputFocus(_display, _window, RevertToParent, CurrentTime);
		if (XGrabKeyboard(_display, _window, False, GrabModeAsync, 
					GrabModeAsync, CurrentTime) != GrabSuccess) {
			log_abort("Unable to grab keyboard");
		} 
		if (XGrabPointer(_display, _window, False, ButtonPressMask, GrabModeAsync, 
					GrabModeAsync, _window, None, CurrentTime) != GrabSuccess) {
			log_abort("Unable to grab keyboard");
		} 
		XMoveWindow(_display, _window, 0, 0);
	}
	XSync(_display, False); 
}

void XWindow::make_rasterfont(void) 
{
	XFontStruct *fontInfo;
	Font id;
	unsigned int first, last;

	fontInfo = XLoadQueryFont(XOpenDisplay(NULL), "fixed");
	if (!fontInfo) {
		log_abort("Unable to find fixed Font");
	}

	id    = fontInfo->fid;
	first = fontInfo->min_char_or_byte2;
	last  = fontInfo->max_char_or_byte2;

	m_font_offset = glGenLists((GLuint) last + 1);
	if (!m_font_offset) {
		log_abort("Unable to allocate display lists");
	}
	glXUseXFont(id, first, last - first + 1, m_font_offset + first);
}

bool XWindow::event(bool block)
{
	XEvent event;

	int events = XPending(_display);
	if (block) events++;
	while (events--) {
		XNextEvent(_display, &event);
		switch (event.type) {
			case Expose:
				break;
			case ConfigureNotify:
				{
					uint new_width  = event.xconfigure.width;
					uint new_height = event.xconfigure.height;
					if (new_width == get_width() && new_height == get_height())
						break; 
					Event ev(Event::Resize);
					ResizeEvent& resize_event = ev.resize();
					resize_event.new_width  = new_width;
					resize_event.new_height = new_height;
					_keybuf->insert(ev);
					_width = new_width;
					_height = new_height;
				}
				break; 
			case KeyPress:
				{
					Event ev(Event::Key);
					KeyEvent& key_event = ev.key();
					key_event.type = KeyEvent::DOWN;
					X11_TranslateKey
						(&event.xkey, event.xkey.keycode, key_event);
					_keybuf->insert(ev);
					log_debug("key press");
					break;
		
				}
			case KeyRelease:
				{
					log_debug("key release");
					break;
				}
			case ButtonPress: 
				{
					Event ev(Event::MouseButton);
					MouseButtonEvent& mouse_button_event = ev.mousebutton();
					mouse_button_event.button = event.xbutton.button;
					_keybuf->insert(ev);
					break;
				}

			case ButtonRelease: 
				{
					break;
				}
		} 
	}
	return true; // please come back...
}

// This was grabbed from the SDL sources.
void XWindow::keymap_init()
{
	for (uint i = 0; i < sizeof(ODD_keymap)/sizeof(ODD_keymap[0]); i++)
		ODD_keymap[i] = SDLK_UNKNOWN;

	/* Map the miscellaneous keys */
	for (uint i = 0; i < sizeof(MISC_keymap)/sizeof(MISC_keymap[0]); i++)
		MISC_keymap[i] = SDLK_UNKNOWN;

	/* These X keysyms have 0xFF as the high byte */
	MISC_keymap[XK_BackSpace&0xFF] = SDLK_BACKSPACE;
	MISC_keymap[XK_Tab&0xFF] = SDLK_TAB;
	MISC_keymap[XK_Clear&0xFF] = SDLK_CLEAR;
	MISC_keymap[XK_Return&0xFF] = SDLK_RETURN;
	MISC_keymap[XK_Pause&0xFF] = SDLK_PAUSE;
	MISC_keymap[XK_Escape&0xFF] = SDLK_ESCAPE;
	MISC_keymap[XK_Delete&0xFF] = SDLK_DELETE;

	MISC_keymap[XK_KP_0&0xFF] = SDLK_KP0;		/* Keypad 0-9 */
	MISC_keymap[XK_KP_1&0xFF] = SDLK_KP1;
	MISC_keymap[XK_KP_2&0xFF] = SDLK_KP2;
	MISC_keymap[XK_KP_3&0xFF] = SDLK_KP3;
	MISC_keymap[XK_KP_4&0xFF] = SDLK_KP4;
	MISC_keymap[XK_KP_5&0xFF] = SDLK_KP5;
	MISC_keymap[XK_KP_6&0xFF] = SDLK_KP6;
	MISC_keymap[XK_KP_7&0xFF] = SDLK_KP7;
	MISC_keymap[XK_KP_8&0xFF] = SDLK_KP8;
	MISC_keymap[XK_KP_9&0xFF] = SDLK_KP9;
	MISC_keymap[XK_KP_Insert&0xFF] = SDLK_KP0;
	MISC_keymap[XK_KP_End&0xFF] = SDLK_KP1;	
	MISC_keymap[XK_KP_Down&0xFF] = SDLK_KP2;
	MISC_keymap[XK_KP_Page_Down&0xFF] = SDLK_KP3;
	MISC_keymap[XK_KP_Left&0xFF] = SDLK_KP4;
	MISC_keymap[XK_KP_Begin&0xFF] = SDLK_KP5;
	MISC_keymap[XK_KP_Right&0xFF] = SDLK_KP6;
	MISC_keymap[XK_KP_Home&0xFF] = SDLK_KP7;
	MISC_keymap[XK_KP_Up&0xFF] = SDLK_KP8;
	MISC_keymap[XK_KP_Page_Up&0xFF] = SDLK_KP9;
	MISC_keymap[XK_KP_Delete&0xFF] = SDLK_KP_PERIOD;
	MISC_keymap[XK_KP_Decimal&0xFF] = SDLK_KP_PERIOD;
	MISC_keymap[XK_KP_Divide&0xFF] = SDLK_KP_DIVIDE;
	MISC_keymap[XK_KP_Multiply&0xFF] = SDLK_KP_MULTIPLY;
	MISC_keymap[XK_KP_Subtract&0xFF] = SDLK_KP_MINUS;
	MISC_keymap[XK_KP_Add&0xFF] = SDLK_KP_PLUS;
	MISC_keymap[XK_KP_Enter&0xFF] = SDLK_KP_ENTER;
	MISC_keymap[XK_KP_Equal&0xFF] = SDLK_KP_EQUALS;

	MISC_keymap[XK_Up&0xFF] = SDLK_UP;
	MISC_keymap[XK_Down&0xFF] = SDLK_DOWN;
	MISC_keymap[XK_Right&0xFF] = SDLK_RIGHT;
	MISC_keymap[XK_Left&0xFF] = SDLK_LEFT;
	MISC_keymap[XK_Insert&0xFF] = SDLK_INSERT;
	MISC_keymap[XK_Home&0xFF] = SDLK_HOME;
	MISC_keymap[XK_End&0xFF] = SDLK_END;
	MISC_keymap[XK_Page_Up&0xFF] = SDLK_PAGEUP;
	MISC_keymap[XK_Page_Down&0xFF] = SDLK_PAGEDOWN;

	MISC_keymap[XK_F1&0xFF] = SDLK_F1;
	MISC_keymap[XK_F2&0xFF] = SDLK_F2;
	MISC_keymap[XK_F3&0xFF] = SDLK_F3;
	MISC_keymap[XK_F4&0xFF] = SDLK_F4;
	MISC_keymap[XK_F5&0xFF] = SDLK_F5;
	MISC_keymap[XK_F6&0xFF] = SDLK_F6;
	MISC_keymap[XK_F7&0xFF] = SDLK_F7;
	MISC_keymap[XK_F8&0xFF] = SDLK_F8;
	MISC_keymap[XK_F9&0xFF] = SDLK_F9;
	MISC_keymap[XK_F10&0xFF] = SDLK_F10;
	MISC_keymap[XK_F11&0xFF] = SDLK_F11;
	MISC_keymap[XK_F12&0xFF] = SDLK_F12;
	MISC_keymap[XK_F13&0xFF] = SDLK_F13;
	MISC_keymap[XK_F14&0xFF] = SDLK_F14;
	MISC_keymap[XK_F15&0xFF] = SDLK_F15;

	MISC_keymap[XK_Num_Lock&0xFF] = SDLK_NUMLOCK;
	MISC_keymap[XK_Caps_Lock&0xFF] = SDLK_CAPSLOCK;
	MISC_keymap[XK_Scroll_Lock&0xFF] = SDLK_SCROLLOCK;
	MISC_keymap[XK_Shift_R&0xFF] = SDLK_RSHIFT;
	MISC_keymap[XK_Shift_L&0xFF] = SDLK_LSHIFT;
	MISC_keymap[XK_Control_R&0xFF] = SDLK_RCTRL;
	MISC_keymap[XK_Control_L&0xFF] = SDLK_LCTRL;
	MISC_keymap[XK_Alt_R&0xFF] = SDLK_RALT;
	MISC_keymap[XK_Alt_L&0xFF] = SDLK_LALT;
	MISC_keymap[XK_Meta_R&0xFF] = SDLK_RMETA;
	MISC_keymap[XK_Meta_L&0xFF] = SDLK_LMETA;
	MISC_keymap[XK_Super_L&0xFF] = SDLK_LSUPER; /* Left "Windows" */
	MISC_keymap[XK_Super_R&0xFF] = SDLK_RSUPER; /* Right "Windows */
	MISC_keymap[XK_Mode_switch&0xFF] = SDLK_MODE; /* "Alt Gr" key */
	MISC_keymap[XK_Multi_key&0xFF] = SDLK_COMPOSE; /* Multi-key compose */

	MISC_keymap[XK_Help&0xFF] = SDLK_HELP;
	MISC_keymap[XK_Print&0xFF] = SDLK_PRINT;
	MISC_keymap[XK_Sys_Req&0xFF] = SDLK_SYSREQ;
	MISC_keymap[XK_Break&0xFF] = SDLK_BREAK;
	MISC_keymap[XK_Menu&0xFF] = SDLK_MENU;
	MISC_keymap[XK_Hyper_R&0xFF] = SDLK_MENU;   /* Windows "Menu" key */
}

void XWindow::X11_TranslateKey
	(XKeyEvent *xkey, KeyCode kc, KeyEvent& key_event)
{
	KeySym xsym;

	/* Get the raw keyboard scancode */
	xsym = XKeycodeToKeysym(_display, kc, 0);
#ifdef DEBUG_KEYS
	log_debug("Translating key 0x%.4x (%d)\n", xsym, kc);
#endif
	/* Get the translated SDL virtual keysym */
	key_event.sym = SDLK_UNKNOWN;
	if ( xsym ) {
		switch (xsym>>8) {
			case 0x1005FF:
#ifdef SunXK_F36
				if ( xsym == SunXK_F36 )
					key_event.sym = SDLK_F11;
#endif
#ifdef SunXK_F37
				if ( xsym == SunXK_F37 )
					key_event.sym = SDLK_F12;
#endif
				break;
			case 0x00:	/* Latin 1 */
			case 0x01:	/* Latin 2 */
			case 0x02:	/* Latin 3 */
			case 0x03:	/* Latin 4 */
			case 0x04:	/* Katakana */
			case 0x05:	/* Arabic */
			case 0x06:	/* Cyrillic */
			case 0x07:	/* Greek */
			case 0x08:	/* Technical */
			case 0x0A:	/* Publishing */
			case 0x0C:	/* Hebrew */
			case 0x0D:	/* Thai */
				key_event.sym = (SDLKey)(xsym&0xFF);
				/* Map capital letter syms to lowercase */
				if ((key_event.sym >= 'A')&&(key_event.sym <= 'Z'))
					key_event.sym = SDLKey(key_event.sym + ('a'-'A'));
				break;
			case 0xFE:
				key_event.sym = ODD_keymap[xsym&0xFF];
				break;
			case 0xFF:
				key_event.sym = MISC_keymap[xsym&0xFF];
				break;
			default:
				log_warning("X11: Unknown xsym, sym = 0x%04x\n",
						(unsigned int)xsym);
				break;
		}
	} else {
		/* X11 doesn't know how to translate the key! */
		switch (kc) {
			/* Caution:
			   These keycodes are from the Microsoft Keyboard
			 */
			case 115:
				key_event.sym = SDLK_LSUPER;
				break;
			case 116:
				key_event.sym = SDLK_RSUPER;
				break;
			case 117:
				key_event.sym = SDLK_MENU;
				break;
			default:
				/*
				 * no point in an error message; happens for
				 * several keys when we get a keymap notify
				 */
				break;
		}
	}
	key_event.mod = KMOD_NONE; 
	if (xkey->state & ShiftMask)   key_event.mod |= KMOD_SHIFT;
	if (xkey->state & LockMask)    key_event.mod |= KMOD_CAPS;
	if (xkey->state & ControlMask) key_event.mod |= KMOD_CTRL;
	if (xkey->state & Mod1Mask)    key_event.mod |= KMOD_ALT;
}

// ------------------------------------------------------------------------- //

