/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxmass-polchemdefctxt.h"
#include "polyxmass-self-speak.h"



/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmPolchemdefCtxt *
polyxmass_polchemdefctxt_new (void)
{
  PxmPolchemdefCtxt *pdfctxt = NULL;
  
  
  pdfctxt = g_malloc0 (sizeof (PxmPolchemdefCtxt));
  
  /* The array that will contain the monomer specifications from the
     polymer chemistry definition file.
  */
  pdfctxt->mnmspecGPA = g_ptr_array_new ();
  
  /* The array that will contain the modifivation specifications from
     the polymer chemistry definition file.
  */
  pdfctxt->mdfspecGPA = g_ptr_array_new ();
  
  pdfctxt->refcount = 0;

  /* Initialize the GData keyed datalist for the monomer pixbufs.
   */
  g_datalist_init (&pdfctxt->pixbufGData);
  
  pdfctxt->propGPA = g_ptr_array_new ();
  
  return pdfctxt;
}


gboolean
polyxmass_polchemdefctxt_init_with_type (PxmPolchemdefCtxt *polchemdefctxt,
					 gchar *type,
					 gboolean with_atoms)
{
  PxmPolchemdefSpec *ps = NULL;

  PxmMonomer *mnm = NULL;
  PxmMonomerSpec *mnmspec = NULL;
  
  PxmAtomSpec *as = NULL;

  gchar *dic = NULL;
  
  gint iter = 0;
  
  
  /* We get a polymer chemistry definition context, and a type, and we
     should initialize the relevant members of that 'polchemdefctx' by
     looking in configuration files what the 'type' is and where
     the data are.

     We know that the polyxmass program has global array to help us
     with this. The polyxmass_polchemdefsGPA array contains all the
     polymer chemistry definitions currently available on the system.

     Just get the polchemdefspec object corresponding to the 'type'.
  */
  g_assert (polchemdefctxt != NULL);
  g_assert (type != NULL);

  /*
    At the launch of the GNU polyxmass program, a number of
    configuration files were loaded from disk and their data were
    stored in global arrays.
    
    One of them is polyxmass_polchemdefsGPA, where all the available
    polymer chemistry definitions are stored as PxmPolchemdefSpec
    objects.
    
    Thus, we have to make sure that we can find in that array, an item
    that has the same type as the one we have in the 'type' variable.
    
    Note that 'ps' is not allocated, it is just a reference to a
    PxmPolymerSpec object in an array: do not free it!
  */
  ps = libpolyxmass_polchemdefspec_get_ptr_by_type (polyxmass_polchemdefsGPA,
						    type);

  /* Of course, if that definition type is not found, that means that
     we are not able to find a polymer chemistry definition file
     suitable for the polymer sequence that we are trying to load from
     disk.
     
     That's a real error. Return NULL immediately, after having freed
     anything allocated here.
  */
  if (ps == NULL)
    {
      g_critical (_("%s@%d: could not retrieve polymer definition data"
		    " for type '%s'. Please, make sure that the "
		    "GNU polyxmass software suite is properly installed. "
		    "In particular, is the GNU polyxmass-common package "
		    "installed ?\n"),
		  __FILE__, __LINE__, type);

      return FALSE;
    }
  
  /* At this point we are finally confident that we can try to
     load a polymer chemistry definition file from disk.

     ps now contains three member variables of interest, allocated
     as shown below, before being set to proper values...

     ps->type = g_strdup ("");
     ps->file = g_strdup ("");
     ps->dir = g_strdup ("");

     ps->dir is a string pointing to the directory where the data
     are stored for the polymer chemistry definition. It is in
     this directory that we are going to find the monicons.dic
     file which contains the informations about the different
     monomers of the polymer chemistry definition.

     We will have to do two things to bring the loading of the
     polymer chemistry defintion from file to a successful end:

     - prepare an array of PxmMonomerSpec instances by reading the
     "monicons.dic" file from the directory that is described in
     'ps->dir' above. The same file is used to prepare an array of
     PxmModifSpec instances. These instances will allow the
     polymer sequence editor to know what image to use for each
     monomer code entered at the keyboard... and for the
     modifications' monicons.

     - properly load the polymer definition from its 'ps->file'.
  */

  /* The monicons...
   */
  /* Prepare the 'dic' string so that we have the full filename to the
     monicons dictionary file pertaining to the polymer chemistry
     definition data...
  */
  dic =  g_strdup_printf ("%s/monicons.dic", ps->dir);
  
  /* First work on the monomers.
   */
  if (-1 == 
      libpolyxmass_monomerspec_parse_monicons_dic_file (dic,
							polchemdefctxt->
							mnmspecGPA))
    {
      /* This is really an error, as we cannot go on in any way.
       */
      g_error (_("%s@%d: failed loading monomer monicon "
		 "specs from file: %s\n"),
	       __FILE__, __LINE__, dic);
    }
  
  /* Second work on the modifications.
   */
  if (-1 == 
      libpolyxmass_modifspec_parse_monicons_dic_file (dic,
						      polchemdefctxt->
						      mdfspecGPA))
    {
      /* This is really an error, as we cannot go on in any way.
       */
      g_error (_("%s@%d: failed loading modif monicon specs from file: %s\n"),
	     __FILE__, __LINE__, dic);
    }
  
  g_free (dic);
  
  /* The sounds...
   */
  /* Prepare the 'dic' string so that we have the full filename to the
     sounds dictionary file pertaining to the polymer chemistry
     definition data...
  */
  dic =  g_strdup_printf ("%s/sounds/sounds.dic", ps->dir);
  
  /* First work on the monomers.
   */
  if (-1 == libpolyxmass_monomerspec_parse_sounds_dic_file (dic,
							    polchemdefctxt->
							    mnmspecGPA))
    {
      /* This is really an error, as we cannot go on in any way.
       */
      g_error (_("%s@%d: failed loading monomer sound specs from file: %s\n"),
	     __FILE__, __LINE__, dic);
    }
  
  /* Second work on the modifications.
   */
  if (-1 == libpolyxmass_modifspec_parse_sounds_dic_file (dic,
							  polchemdefctxt->
							  mdfspecGPA))
    {
      /* This is really an error, as we cannot go on in any way.
       */
      g_error (_("%s@%d: failed loading modif sounds specs from file: %s\n"),
	     __FILE__, __LINE__, dic);
    }
  
  g_free (dic);
  

  /* 
     Finally we can render the polymer chemistry definition file into
     the polchemdef member in the polchemdefctxt object. During the
     load, the polchemdef->file member is set to the ps->file string
     that is passed as parameter in the function call below.

     Obviously we should be working with a brand new polymer chemistry
     definition context... so we can assert that the
     polchemdefctxt->polchemdef == NULL.
  */
  g_assert (polchemdefctxt->polchemdef == NULL);

  polchemdefctxt->polchemdef = pxmchem_polchemdef_render_xml_file (ps->file);
  
  if (polchemdefctxt->polchemdef == NULL)
    {
      g_critical (_("%s@%d: Could not render a polchemdef from file '%s'\n"),
	     __FILE__, __LINE__, ps->file);
      
      return FALSE;
    }
  
  /* We want that the polymer chemistry definition file be self-conscious
     of where its data are located in the system.
  */
  /* Set the directory of the polymer definition-related data:
   */
  pxmchem_polchemdef_set_dir (polchemdefctxt->polchemdef, ps->dir);
  
#if 0
  
  debug_printf (("new polymer chemistry definition type is %s\n", 
		 polchemdefctxt->polchemdef->type));
  debug_printf (("new polymer chemistry definition file is %s\n", 
		 polchemdefctxt->polchemdef->file));
  debug_printf (("new polymer chemistry definition dir is %s\n", 
		 polchemdefctxt->polchemdef->dir));
  debug_printf (("new polymer chemistry definition codelen is %d\n", 
		 polchemdefctxt->polchemdef->codelen));
  
#endif
  
  /* Now that the polymer definition has been rendered into memory
     from the xml-formatted file, we should make sure that each
     monomer object in the polchemdef->monomerGPA has a corresponding
     PxmMonomerSpec instance in the array that we had filled before
     (by reading the monicons.dic file). 

     Also, it is of critical importance that both array do contain the
     same number of items, otherwise we would not know where is/are
     the supplementary elements...

     All this is to ensure that there is a good consistency between
     the PxmMonomer objects rendered in memory and the configuration
     file monicons.dic.

     We take advantage of this run through both the arrays to fill in
     the last bit of data in each monomerspec that was lacking: the
     name of the monomer.
  */
      
  g_assert (polchemdefctxt->polchemdef->monomerGPA->len
	    == polchemdefctxt->mnmspecGPA->len);

  /*
  debug_printf (("%d\n%d\n", polchemdefctxt->polchemdef->monomerGPA->len,
		 polchemdefctxt->mnmspecGPA->len));
  */
  for (iter = 0; iter < polchemdefctxt->polchemdef->monomerGPA->len; iter++)
    {
      mnm = g_ptr_array_index (polchemdefctxt->polchemdef->monomerGPA, iter);
      
      g_assert (mnm != NULL);
      
      mnmspec =
	libpolyxmass_monomerspec_get_ptr_by_code (polchemdefctxt->
						  mnmspecGPA, 
						  mnm->code);
      g_assert (mnmspec != NULL);

      libpolyxmass_monomerspec_set_name (mnmspec, mnm->name);
    }
  
  /* We will very often need to make sure a given string, like "Ala"
     for example, does actually correspond to a monomer code. Instead
     of systematically iterating in the array of monomers that is
     member of the polchemdef object, we can ask if we find that code
     in a string (much quicker stuff). For this we need to have a
     string of monomer codes that is initialized using all the codes
     of all the monomers in the polymer chemistry definition.
  */
  pxmchem_polchemdef_update_delim_codes (polchemdefctxt->polchemdef,
					 libpolyxmass_globals_delim);
  
  if (TRUE == with_atoms)
    {
      /* We now have to make sure that the newly created polymer
	 chemistry definition has its specific atomGPA filled with
	 the proper atom definitions !
     
	 We ask to get a pointer to the PxmAtomSpec object from
	 global array in polyxmass, that corresponds to the type
	 of our polymer chemistry definition.
     
	 Note that 'as' is not allocated here, it is just a
	 reference to an object sitting away from here.
      */
      as = polyxmass_globals_get_matching_polchemdef_atomdef 
	(polchemdefctxt->polchemdef->type, 
	 /* gint *polchemdef_idx */NULL,
	 /* gchar *atomdef */NULL, 
	 /* gint *atomdef_idx */NULL);
  
      /* At this point we MUST have a proper 'as' instance,
	 otherwise that is an error, because we could not go on in
	 any imaginable way.
      */
      if (as == NULL)
	{
	  g_critical (_("%s@%d: failed finding a matching atom definition"
		   " for polymer chemistry definition '%s'.\n"),
		 __FILE__, __LINE__, polchemdefctxt->polchemdef->type);
      
	  return FALSE;
	}
  
      /* Because we now have the 'as->file' filename of the atom
	 definition that goes hand in hand with our polymer chemistry
	 definition, we can render that file.
      */
      if (-1 == pxmchem_atom_render_xml_file (as->file, 
					      polchemdefctxt->polchemdef->
					      atomGPA))
	{
	  g_critical (_("%s@%d: failed parsing the atom definition "
		   "file '%s'.\n"),
		 __FILE__, __LINE__, as->file);
      
	  return FALSE;
	}
    }
  /* End of 
     if (TRUE == with_atoms)
  */ 

  return TRUE;
}



  

/*  LOCATING FUNCTIONS
 */
gint
polyxmass_polchemdefctxt_get_index_by_type (GPtrArray *GPA, gchar *type)
{
  gint iter = 0;

  PxmPolchemdefCtxt *pdfctxt = NULL;
  
  g_assert (type != NULL);
  g_assert (GPA != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    {
      pdfctxt = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (pdfctxt->polchemdef->type, type))
	return iter;
    }
  
  return -1;
}


gint
polyxmass_polchemdefctxt_get_index (GPtrArray *GPA, 
				    PxmPolchemdefCtxt *pdfctxt)
{
  gint iter = 0;

  PxmPolchemdefCtxt *pdfctxt_iter = NULL;
  
  g_assert (pdfctxt != NULL);
  g_assert (GPA != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    {
      pdfctxt_iter = g_ptr_array_index (GPA, iter);
      
      if (pdfctxt_iter == pdfctxt)
      return iter;
    }
  
  return -1;
}


PxmPolchemdefCtxt *
polyxmass_polchemdefctxt_get_ptr_by_type (gchar *type, GPtrArray *GPA)
{
  gint iter = 0;

  PxmPolchemdefCtxt *pdfctxt = NULL;
  
  g_assert (type != NULL);
  g_assert (GPA != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    {
      pdfctxt = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (pdfctxt->polchemdef->type, type))
	return pdfctxt;
    }
  
  return NULL;
}





/* REFERENCE COUNTING MANAGEMENT
 */
gint
polyxmass_polchemdefctxt_ref (PxmPolchemdefCtxt *pdfctxt)
{
  g_assert (pdfctxt != NULL);
  
  (pdfctxt->refcount)++;

  return pdfctxt->refcount;
}


gint
polyxmass_polchemdefctxt_unref (PxmPolchemdefCtxt *pdfctxt)
{
  gint count = 0;
  

  g_assert (pdfctxt != NULL);


  (pdfctxt->refcount)--;

  /* Now that we have decremented the refcount variable, we can check
   * if it is now == 0. If so, we'll simply remove this polchemdefctxt
   * from the program's GPtrArray of polchemdefctxt's.
   */
  if (pdfctxt->refcount <= 0)
    {
      g_message (_("%s@%d: closing last polymer definition context of type: "
		   "'%s'\n"),
		 __FILE__, __LINE__, 
		 pdfctxt->polchemdef->type);
      
      /* The function below returns the number of items still in the
	 array, after the removal. The count that is returned is not
	 specific of the type of the polchemdefctxt, but includes all the
	 polchemdefctxt objects that are currently in memory and stored in
	 the program's array polyxmass_polchemdefctxtGPA.
      */
      count = polyxmass_polchemdefctxt_remove_from_GPA 
	(polyxmass_polchemdefctxtGPA,
	 pdfctxt);
      
      polyxmass_polchemdefctxt_free (pdfctxt);
      
      /* Return 0 for the simple reason that there are no more counts for
       * this polchemdefctxt.
       */
      return 0;
    }
  
  /* If here, that means that this polymer definition context is in
     use by at least one object. Return the refcount of this
     polchemdefctxt.
   */
  return pdfctxt->refcount;
}


gint
polyxmass_polchemdefctxt_ref_from_type (GPtrArray *GPA, gchar *type)
{
  PxmPolchemdefCtxt *pdfctxt = NULL;

  g_assert (type != NULL);
  g_assert (GPA != NULL);
  
  pdfctxt = polyxmass_polchemdefctxt_get_ptr_by_type (type, GPA);

  if (pdfctxt != NULL)
    (pdfctxt->refcount)++;
  else
    {
      g_critical (_("%s@%d: polymer definition not found: '%s'\n"),
	     __FILE__, __LINE__, 
	     type);
      
      return -1;
    }

  return pdfctxt->refcount;
}


gint
polyxmass_polchemdefctxt_unref_from_type (GPtrArray *GPA, gchar *type)
{
  PxmPolchemdefCtxt *pdfctxt = NULL;

  g_assert (type != NULL);
  g_assert (GPA != NULL);
  
  pdfctxt = polyxmass_polchemdefctxt_get_ptr_by_type (type, GPA);

  if (pdfctxt != NULL)
    (pdfctxt->refcount)--;
  else
    {
      g_critical (_("%s@%d: polymer definition not found: '%s'\n"),
	     __FILE__, __LINE__, type);
      
      return -1;
    }

  /* Now that we have decremented the refcount variable, we can check
   * if it is now == 0. If so, we'll simply remove this polchemdefctxt
   * from the program's GPtrArray of polchemdefctxt's.
   */
  if (pdfctxt->refcount <= 0)
    {
      g_message (_("%s@%d: closing last polymer definition of type: '%s'\n"),
		 __FILE__, __LINE__, 
		 type);
      
      polyxmass_polchemdefctxt_remove_from_GPA (polyxmass_polchemdefctxtGPA,
					    pdfctxt);
      
      polyxmass_polchemdefctxt_free (pdfctxt);
      
      /* Return 0 for the simple reason that there are no more counts for
       * this polchemdefctxt.
       */
      return 0;
    }
  
  /* If here, that means that this polymer definition context is in
     use by at least one object. Return the refcount of this
     polchemdefctxt.
   */
  return pdfctxt->refcount;
}


gint
polyxmass_polchemdefctxt_remove_from_GPA (GPtrArray *GPA, 
					  PxmPolchemdefCtxt *pdfctxt)
{
  g_assert (pdfctxt != NULL);
  g_assert (GPA != NULL);
  

  if (FALSE == g_ptr_array_remove (GPA, pdfctxt))
    {
      g_warning (_("%s@%d: failed removing polymer definition "
		   "of type: '%s'\n"),
		 __FILE__, __LINE__, 
		 pdfctxt->polchemdef->type);
      
      return -1;
    }

  return GPA->len;
}





/* FREE'ING FUNCTIONS
 */
gboolean
polyxmass_polchemdefctxt_free (PxmPolchemdefCtxt *pdfctxt)
{

  g_assert (pdfctxt != NULL);
  

  /* Start with internal data:
   */
  libpolyxmass_prop_GPA_free (pdfctxt->propGPA);
  
  if (pdfctxt->polchemdef != NULL)
    pxmchem_polchemdef_free (pdfctxt->polchemdef);
  
  libpolyxmass_monomerspec_GPA_free (pdfctxt->mnmspecGPA);
  
  /* Clear the pixbfGData GData list. BUT before doing this we have to
     unref one last time all the GdkPixbuf objects that were made
     during the loading/editing of the polymer sequence.
   */
  g_datalist_foreach (&pdfctxt->pixbufGData,
		      polyxmass_polchemdefctxt_unref_last_pixbuf_from_gdata,
		      NULL);
  
  g_datalist_clear (&pdfctxt->pixbufGData);


  g_free (pdfctxt);
  
  return TRUE;
}


void 
polyxmass_polchemdefctxt_unref_last_pixbuf_from_gdata (GQuark quarkId,
						   gpointer gdata,
						   gpointer data) 
{
  /* We get a pointer corresponding to the data of a GData item in the
   * polchemdef's pixbufGData GData *. We have to unref the GdkPixbuf *
   * that is the data parameter.
   */
  g_object_unref (G_OBJECT (gdata));
}




/* GPtrArray-RELATED FUNCTIONS
 */
gint
polyxmass_polchemdefctxt_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  
  PxmPolchemdefCtxt *pdfctxt = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      pdfctxt = g_ptr_array_remove_index (GPA, 0);
      g_assert (pdfctxt != NULL);
      polyxmass_polchemdefctxt_free (pdfctxt);
      count++;
    }
  
  g_ptr_array_free (GPA, TRUE);

  return count;
}
