/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxmass-ui-filechoosers.h"



GtkWidget *
polyxmass_filechoosers_show_wnd (GtkWidget *parent, 
				 PxmFileChooseConfig config)
{
  GtkWidget *window = NULL;
  
  GMainLoop* loop = NULL;
  
  gchar *help = NULL;
    

  /* We are asked to open a window that will allow the user to select
     a file amongst the available files in the system... The config
     param is an enum, and we'll check for two bits: 

     PXM_FILE_CHOOSE_ATOMDEF and PXM_FILE_CHOOSE_POLCHEMDEF

     Next, in the called functions other bits are going to be tested...

     We have two main arrays were objects are giving indications of
     where the data are located on the running computer (either in
     system directories or in the user's directories).

     We'll have to parse one of these two arrays, depending on the
     'config' parameter, and put data in such a manner that the user
     might select one item as if it were in the normal file selection
     dialog.
  */
  g_assert (parent != NULL);
  
  
  if (config & PXM_FILE_CHOOSE_ATOMDEF)
    {
      window = polyxmass_filechoosers_setup_atomdef_wnd (config);
      
      if (window == NULL)
	{
	  g_critical ("%s@%d: failed setting up the filechooser window\n",
		 __FILE__, __LINE__);
	  
	  return NULL;
	}
    }
  
  if (config & PXM_FILE_CHOOSE_POLCHEMDEF)
    {
      window = polyxmass_filechoosers_setup_polchemdef_polseq_wnd (config);
      
      if (window == NULL)
	{
	  g_critical ("%s@%d: failed setting up the filechooser window\n",
		 __FILE__, __LINE__);
	  
	  return NULL;
	}
    }

  /* Set a pointer to the parent window in the newly created window so
     that the created window can speak with the parent one and transmit
     to it the selection data.
  */
  g_object_set_data (G_OBJECT (window), "parent_window", parent);
  

  /* This window will show up when we call gtk_widget_show_all
     (window); below. What we want is that the program execution be
     stopped while the user answers the modal dialog here. To stop the
     program from proceeding (in the calling function), we have to
     stop the execution until this window is closed. For this we
     create a new loop into which we put ourselves here. This loop is
     associated to *this window in such a manner that when the window
     is closed, then the loop is quit and program execution can
     proceed in the caller function (see g_object_set_data_full
     below). Of course, meanwhile, we will have set the datum of
     interest in the parent (the calling) window, thus giving a raison
     d'\^etre to this whole stuff.
  */
  loop = g_main_loop_new (NULL, FALSE);
  g_assert (loop != NULL);
  help = g_strdup_printf ("loop-%p", loop);
  
  g_object_set_data_full (G_OBJECT (window), help, loop,
			  (GDestroyNotify) g_main_loop_quit);
  g_free (help);

  g_main_loop_run (loop);

  return window;
}



GtkWidget *
polyxmass_filechoosers_setup_atomdef_wnd (PxmFileChooseConfig config)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *sw = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeIter tree_iter;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;

  GladeXML *xml = NULL;
  
  gchar *gui_file = NULL;
  
  gint iter = 0;
  gint col_offset = 0;
  
  PxmAtomSpec *as = NULL;


  /* We now have to setup a window where the user will be able to
     choose one of the available atom definitions.

     Note that if config has the PXM_FILE_CHOOSE_GTK bit set, then
     we'll pack a GtkFileChooserWidget in the window.
  */
  gui_file = 
    g_strdup_printf ("%s/polyxmass-filechoosers.glade", userspec->gladedir);

  xml = glade_xml_new (gui_file, "atomdef_filechooser_wnd", 
		       PACKAGE);

  g_free (gui_file);  

  if (xml == NULL)
    {
      g_error (_("%s@%d: failed loading the interface\n"),
	     __FILE__, __LINE__);
    }
  
  window = glade_xml_get_widget (xml, 
				 "atomdef_filechooser_wnd");

   if (window == NULL)
    {
      g_error (_("%s@%d: failed creating the interface\n"),
	     __FILE__, __LINE__);
    }

  widget = glade_xml_get_widget (xml, "atomdefs_available_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "atomdefs_available_frame", widget);

  widget = glade_xml_get_widget (xml, "atomdef_filechooser_select_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "atomdef_filechooser_select_button", widget);

  widget = glade_xml_get_widget (xml, "atomdef_filechooser_cancel_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "atomdef_filechooser_cancel_button", widget);

  vbox = glade_xml_get_widget (xml, "atomdefs_available_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "atomdefs_available_vbox", vbox);
  
  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  g_object_set_data (G_OBJECT (window),
		     "atomdefs_available_sw", sw);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  

  /* Now create the model of the treeview.
   */
  model = (GtkTreeModel *) gtk_tree_store_new (COLUMN_ATOMSPEC_COL_COUNT,
					       
					       G_TYPE_STRING,
					       G_TYPE_STRING,
					       G_TYPE_STRING,
					       
					       G_TYPE_POINTER,
					       
					       G_TYPE_BOOLEAN);
  
  /* Add data for each polchemdefspec in the array.
   */
  for (iter = 0 ; iter < polyxmass_atomdefsGPA->len ; iter++)
    {
      as = g_ptr_array_index (polyxmass_atomdefsGPA, iter);
      g_assert (as != NULL);
      
      gtk_tree_store_append ((GtkTreeStore *) model, &tree_iter, NULL);

      gtk_tree_store_set ((GtkTreeStore *) model, &tree_iter,
			  
			  COLUMN_ATOMSPEC_POLTYPE, as->poltype,
			  COLUMN_ATOMSPEC_ATOMDEF, as->atomdef,
			  COLUMN_ATOMSPEC_FILE, as->file,
			  
			  COLUMN_ATOMSPEC_POINTER, as,
			  
			  -1);
    }
  
  g_object_set_data (G_OBJECT (window), "atomdefs_treeview_model", model);
  
  /* Now create the treeview.
   */
  treeview = gtk_tree_view_new_with_model (model);
  g_assert (treeview != NULL);
  
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (window), "atomdefs_treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);
  
  /* Atomspec poltype member.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_ATOMSPEC_POLTYPE);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Pol. Def."),
						 renderer, 
						 "text",
						 
						 COLUMN_ATOMSPEC_POLTYPE,
						 
						 NULL);
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  
  /* Atomspec atomdef member.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_ATOMSPEC_ATOMDEF);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Atom. Def."),
						 renderer, 
						 "text",
						 
						 COLUMN_ATOMSPEC_ATOMDEF,
						 
						 NULL);
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  
  /* Atomspec file member.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_ATOMSPEC_FILE);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Atom Def. File"),
						 renderer, 
						 "text",
						 
						 COLUMN_ATOMSPEC_FILE,
						 
						 NULL);
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);





  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  gtk_tree_view_columns_autosize (GTK_TREE_VIEW (treeview));

  
  /* OK, now deal with the buttons. We want that the parent window be
     informed about the selection (we'll select the pointer of the
     item that was selected when the "Select" button was clicked, or we'll
     report a NULL value if "Cancel" was clicked.
  */

  widget = glade_xml_get_widget (xml, "atomdef_filechooser_select_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "atomdef_filechooser_select_button", widget);

  g_signal_connect (G_OBJECT (widget), "clicked", 
		    G_CALLBACK 
		    (polyxmass_filechoosers_atomdef_select_button_clicked),
		    window);

  widget = glade_xml_get_widget (xml, "atomdef_filechooser_cancel_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "atomdef_filechooser_cancel_button", widget);

  g_signal_connect (G_OBJECT (widget), "clicked", 
		    G_CALLBACK 
		    (polyxmass_filechoosers_atomdef_cancel_button_clicked),
		    window);
   

  /* We show the gtk_file_chooser widget frame only if this is required
     by the 'config' parameter. In other words, we show it if the caller
     wants to allow the user to select a file that is not yet registered
     as an available polymer definition.
  */
  if (config & PXM_FILE_CHOOSE_GTK)
    {
      /* By default this widget is invisible. In our present case 
	 we want to make it visible.
      */
      widget = glade_xml_get_widget (xml, "gtk_filechooser_frame");
      g_assert (widget != NULL);
      g_object_set_data (G_OBJECT (window),
			 "gtk_filechooser_frame", widget);

      /* Get a pointer to the vbox in which we'll pack the
	 gtk_file_chooser.
       */
      vbox = glade_xml_get_widget (xml, "gtk_filechooser_vbox");
      g_assert (vbox != NULL);
      g_object_set_data (G_OBJECT (window),
			 "gtk_filechooser_vbox", vbox);

      /* Now create the gtk_file_chooser proper and put it into the 
	 container.
      */
      widget = gtk_file_chooser_widget_new (GTK_FILE_CHOOSER_ACTION_OPEN);
      g_assert (widget != NULL);
      g_object_set_data (G_OBJECT (window),
			 "gtk_filechooser_widget", widget);

      gtk_box_pack_start (GTK_BOX (vbox), widget, TRUE, TRUE, 0);  

      gtk_widget_show_all (vbox);
    }
  else
    {
      /* We hide the Frame into which the gtk filechooser would have been
	 displayed if it were to be displayed (but it's not).
      */
      widget = glade_xml_get_widget (xml, "gtk_filechooser_frame");
      g_assert (widget != NULL);
      g_object_set_data (G_OBJECT (window),
			 "gtk_filechooser_frame", widget);

      gtk_widget_hide (widget);
    }
 

/* WINDOW CLOSING FUNCTIONS
 */
  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK 
		    (polyxmass_filechoosers_atomdef_wnd_delete_event),
		    window);
  
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK 
		    (polyxmass_filechoosers_atomdef_wnd_destroy_event),
		    window);

  
  gtk_widget_show_all (window);
  
  return window;
}



GtkWidget *
polyxmass_filechoosers_setup_polchemdef_polseq_wnd (PxmFileChooseConfig 
						    config)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  GtkWidget *frame = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *sw = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeIter tree_iter;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;

  GladeXML *xml = NULL;
  
  gchar *gui_file = NULL;
  
  gint iter = 0;
  gint col_offset = 0;
  
  PxmPolchemdefSpec *ps = NULL;


  /* We now have to setup a window where the user will be able to
     choose one of the available polchem  definitions.

     Note that if 'config' has the PXM_FILE_CHOOSE_GTK bit set, then
     we'll pack a GtkFileChooserWidget in the window for the user to
     be able to select a random file from disk a the polymer chemistry
     definition.

     Further, if 'config' has the PXM_FILE_CHOOSE_POLSEQDATA_INIT bit
     set, we'll pack a GtkFileChooserWidget in the window for the user
     to be able to select a random file from disk a new polymer
     sequence. Also, some widgets are packed to allow the user to set
     some basic information to initialize the new polymer sequence.
  */
  gui_file = 
    g_strdup_printf ("%s/polyxmass-filechoosers.glade", userspec->gladedir);

  xml = glade_xml_new (gui_file, "polchemdef_polseq_filechooser_wnd", 
		       PACKAGE);

  g_free (gui_file);  

  if (xml == NULL)
    {
      g_error (_("%s@%d: failed loading the interface\n"),
	     __FILE__, __LINE__);
    }
  
  window = glade_xml_get_widget (xml, 
				 "polchemdef_polseq_filechooser_wnd");

   if (window == NULL)
    {
      g_error (_("%s@%d: failed creating the interface\n"),
	     __FILE__, __LINE__);
    }

   /* The timed-out messages entry...
    */
   widget = glade_xml_get_widget (xml, "messages_entry");
   g_assert (widget != NULL);
   g_object_set_data (G_OBJECT (window),
		      "messages_entry", widget);

   /* The paned widget...
    */
   widget = glade_xml_get_widget (xml, "polchemdef_polseq_vpaned");
   g_assert (widget != NULL);
   g_object_set_data (G_OBJECT (window),
		      "polchemdef_polseq_vpaned", widget);
   
   
  frame = glade_xml_get_widget (xml, "polchemdefs_available_frame");
  g_assert (frame != NULL);
  g_object_set_data (G_OBJECT (window),
		     "polchemdefs_available_frame", frame);

  vbox = glade_xml_get_widget (xml, "polchemdefs_available_vbox");
  g_assert (vbox != NULL);
  g_object_set_data (G_OBJECT (window),
		     "polchemdefs_available_vbox", vbox);
  
  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  g_object_set_data (G_OBJECT (window),
		     "polchemdefs_available_sw", sw);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  

  /* Now create the model of the treeview.
   */
  model = (GtkTreeModel *) gtk_tree_store_new (COLUMN_POLCHEMSPEC_COL_COUNT,
					       
					       G_TYPE_STRING,
					       G_TYPE_STRING,
					       G_TYPE_STRING,
					       
					       G_TYPE_POINTER,
					       
					       G_TYPE_BOOLEAN);
  
  /* Add data for each polchemdefspec in the array.
   */
  for (iter = 0 ; iter < polyxmass_polchemdefsGPA->len ; iter++)
    {
      ps = g_ptr_array_index (polyxmass_polchemdefsGPA, iter);
      g_assert (ps != NULL);
      
      gtk_tree_store_append ((GtkTreeStore *) model, &tree_iter, NULL);

      gtk_tree_store_set ((GtkTreeStore *) model, &tree_iter,
			  
			  COLUMN_POLCHEMSPEC_TYPE, ps->type,
			  COLUMN_POLCHEMSPEC_FILE, ps->file,
			  COLUMN_POLCHEMSPEC_DIR, ps->dir,
			  
			  COLUMN_POLCHEMSPEC_POINTER, ps,
			  
			  -1);
    }
  
  g_object_set_data (G_OBJECT (window), "polchemdefs_treeview_model", model);
  
  /* Now create the treeview.
   */
  treeview = gtk_tree_view_new_with_model (model);
  g_assert (treeview != NULL);
  
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (window), "polchemdefs_treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);
  
  /* Polymerspec type member.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_POLCHEMSPEC_TYPE);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Def. Type"),
						 renderer, 
						 "text",
						 
						 COLUMN_POLCHEMSPEC_TYPE,
						 
						 NULL);
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  
  /* Polymerspec file member.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_POLCHEMSPEC_FILE);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Def. File"),
						 renderer, 
						 "text",
						 
						 COLUMN_POLCHEMSPEC_FILE,
						 
						 NULL);
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  
  /* Polymerspec dir member.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_POLCHEMSPEC_DIR);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Def. Dir"),
						 renderer, 
						 "text",
						 
						 COLUMN_POLCHEMSPEC_DIR,
						 
						 NULL);
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_tree_view_columns_autosize (GTK_TREE_VIEW (treeview));


  gtk_widget_show_all (frame);
  


  /* We show the polchemdef gtk_file_chooser widget frame only if this
     is required by the 'config' parameter. In other words, we show it
     if the caller wants to allow the user to select a file that is
     not yet registered as an available polymer chemistry definition.
  */
  if (config & PXM_FILE_CHOOSE_GTK)
    {
      /* We have to pack everything in the following vertical box,
	 so that if we need to hide all this, then it is done easily in
	 one step. Also, it is going to be easier to determine later,
	 when the "Validate" button is clicked what is to be done
	 with these widgets, depending on their being visible or not.
      */
      vbox = glade_xml_get_widget 
	(xml, 
	 "show_hide_vbox_gtk_filechooser_polchemdef");
      g_assert (vbox != NULL);  
      g_object_set_data (G_OBJECT (window),
			 "show_hide_vbox_gtk_filechooser_polchemdef", vbox);
      
      /* That's the decoration frame where the gtk filechooser widget
	 is going to be packed.
      */
      frame = glade_xml_get_widget (xml, "gtk_filechooser_polchemdef_frame");
      g_assert (frame != NULL);
      g_object_set_data (G_OBJECT (window),
			 "gtk_filechooser_polchemdef_frame", frame);

      /* We really need a vbox in the frame, so that we can pack the
	 gtk filechooser widget itself...

       */
      vbox = glade_xml_get_widget (xml, "gtk_filechooser_polchemdef_vbox");
      g_assert (vbox != NULL);
      g_object_set_data (G_OBJECT (window),
			 "gtk_filechooser_polchemdef_vbox", vbox);

      /* Now create the gtk_file_chooser proper and put it into the 
	 container.
      */
      widget = gtk_file_chooser_widget_new (GTK_FILE_CHOOSER_ACTION_OPEN);
      g_assert (widget != NULL);
      g_object_set_data (G_OBJECT (window),
			 "gtk_filechooser_polchemdef_widget", widget);

      gtk_box_pack_start (GTK_BOX (vbox), widget, TRUE, TRUE, 0);  

      gtk_widget_show_all (frame);
    }
  else
    {
      /* We hide the vbox into which the gtk filechooser would have
	 been displayed if it were to be displayed (but it's not). See
	 above for details about this vbox.
      */
      vbox = glade_xml_get_widget 
	(xml, "show_hide_vbox_gtk_filechooser_polchemdef");
      g_assert (vbox != NULL);
      g_object_set_data (G_OBJECT (window),
			 "show_hide_vbox_gtk_filechooser_polchemdef", vbox);
      
      gtk_widget_hide (vbox);
    }



  
  /* But we might also have to show a frame
     gtk_filechooser_polseq_frame we a number of widget (including a
     GtkFileChooserWidget) are packed for the user to fill
     initialization data about a new polymer sequence.
  */
  if (config & PXM_FILE_CHOOSE_POLSEQDATA_INIT)
    {
      /* At this point we can start setting up the
	 GtkFileChooserWidget that is used by the user to select a
	 non-existing file as a new polymer sequence filename.
      */
      vbox = glade_xml_get_widget (xml, "new_polseq_init_data_vbox");
      g_assert (vbox != NULL);
      g_object_set_data (G_OBJECT (window),
			 "new_polseq_init_data_vbox", vbox);      

      widget = glade_xml_get_widget (xml, "gtk_filechooser_new_polseq_frame");
      g_assert (widget != NULL);
      g_object_set_data (G_OBJECT (window),
			 "gtk_filechooser_new_polseq_frame", widget);
      
      vbox = glade_xml_get_widget (xml, "gtk_filechooser_new_polseq_vbox");
      g_assert (vbox != NULL);
      g_object_set_data (G_OBJECT (window),
			 "gtk_filechooser_new_polseq_vbox", vbox);
      
      /*
	widget = glade_xml_get_widget (xml, "");
	g_assert (widget != NULL);
	g_object_set_data (G_OBJECT (window),
	"", widget);
      */

      widget = gtk_file_chooser_widget_new (GTK_FILE_CHOOSER_ACTION_SAVE);
      g_assert (widget != NULL);
      g_object_set_data (G_OBJECT (window),
			 "gtk_filechooser_new_polseq_widget", widget);

      gtk_box_pack_start (GTK_BOX (vbox), widget, TRUE, TRUE, 0);  

      gtk_widget_show_all (vbox);
    }
  else
    {
      vbox = glade_xml_get_widget (xml, "new_polseq_init_data_vbox");
      g_assert (vbox != NULL);
      g_object_set_data (G_OBJECT (window),
			 "new_polseq_init_data_vbox", vbox);      

      /* We hide the whole vbox into which the gtk filechooser would
	 have been displayed if it were to be displayed (but it's not).
      */
      gtk_widget_hide (vbox);
    }
  

      /* And now set up the polymer sequence specific data frame where
	 the user enters data about the new sequence to be created.
      */
      widget = glade_xml_get_widget (xml, "new_polseq_init_frame"); 
      g_assert (widget != NULL);
      g_object_set_data (G_OBJECT (window),
			 "new_polseq_init_frame", widget);

      widget = glade_xml_get_widget (xml, "polseq_name_entry");
      g_assert (widget != NULL);
      g_object_set_data (G_OBJECT (window),
			 "polseq_name_entry", widget);
      
      widget = glade_xml_get_widget (xml, "polseq_code_entry");
      g_assert (widget != NULL);
      g_object_set_data (G_OBJECT (window),
			 "polseq_code_entry", widget);
  
  /* OK, now deal with the buttons. We want that the parent window be
     informed about the selection (we'll select the pointer of the
     item that was selected when the "Select" button was clicked, or we'll
     report a NULL value if "Cancel" was clicked.
  */
  widget = 
    glade_xml_get_widget (xml, 
			  "polchemdef_polseq_filechooser_validate_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "polchemdef_polseq_filechooser_validate_button", widget);

  g_signal_connect 
    (G_OBJECT (widget), "clicked", 
     G_CALLBACK 
     (polyxmass_filechoosers_polchemdef_polseq_validate_button_clicked),
     window);

  widget = 
    glade_xml_get_widget (xml, 
			  "polchemdef_polseq_filechooser_cancel_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "polchemdef_polseq_filechooser_cancel_button", widget);

  g_signal_connect 
    (G_OBJECT (widget), "clicked", 
     G_CALLBACK 
     (polyxmass_filechoosers_polchemdef_polseq_cancel_button_clicked),
     window);

  

/* WINDOW CLOSING FUNCTIONS
 */
  g_signal_connect 
    (G_OBJECT (window),
     "delete_event",
     G_CALLBACK (polyxmass_filechoosers_polchemdef_polseq_wnd_delete_event),
     window);
  
  g_signal_connect 
    (G_OBJECT (window),
     "destroy_event",
     G_CALLBACK (polyxmass_filechoosers_polchemdef_polseq_wnd_destroy_event),
     window);
  
  return window;
}





void
polyxmass_filechoosers_atomdef_select_button_clicked (GtkWidget *widget,
							 gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *parent = NULL;

  GtkTreeView *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeIter tree_iter_sel;
  GtkTreeSelection *tree_selection = NULL;
  
  PxmAtomSpec *as = NULL;

  gboolean result = FALSE;

  gchar *filename = NULL;



  g_assert (window != NULL);

  /* Get the pointer to the parent window, the one that is letting
     the user choose a atomdef ...
  */
  parent = g_object_get_data (G_OBJECT (window), "parent_window");
  g_assert (parent != NULL);
  

  /* There should be one item selected in the treeview.
   */

  model = (GtkTreeModel *) g_object_get_data (G_OBJECT (window), 
					      "atomdefs_treeview_model");
  g_assert (model != NULL);

  treeview= g_object_get_data (G_OBJECT (window), "atomdefs_treeview");
  g_assert (treeview != NULL);

  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);

  if (result == FALSE)
    {
      /* No item is selected in the treeview, so check if the gtk
	 filechooser frame is visible. If so, try to get the filename
	 that might be selected...
      */
      widget = g_object_get_data (G_OBJECT (window),
				  "gtk_filechooser_frame");
      g_assert (widget != NULL);
      
      if (FALSE == GTK_WIDGET_VISIBLE (widget))
	{
	  /* Just set a datum to the parent window that's a NULL pointer.
	   */
	  g_object_set_data (G_OBJECT (parent), 
			     "atomdefs_filename", NULL);
	  
	  return;
	}
      
      /* At this point we know that the gtk filechooser widget is currently
	 visible, so check if the user has selected a filename in it.
      */
      widget = g_object_get_data (G_OBJECT (window),
				  "gtk_filechooser_widget");
      
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (widget));

      /* Just set a datum to the parent window that's a NULL pointer.
       */
      g_object_set_data (G_OBJECT (parent), 
			 "atomdefs_filename", filename);
      
      gtk_widget_destroy (window);
      
      return;
    }

  /* At this point we know that one item was selected in the treeview,
     so we should extract from it the filename and send it as a datum
     to the parent window.
  */
  gtk_tree_model_get (model,
		      &tree_iter_sel,
		      COLUMN_ATOMSPEC_POINTER, &as,
		      -1);
  
  /* We duplicate it because the caller window will free that filename
     automatically (g_object_set_data_full () _full variant).
   */
  filename = g_strdup (as->file);
  
  g_object_set_data_full (G_OBJECT (parent), 
			  "atomdefs_filename", filename,
			  (GDestroyNotify) g_free);
  
  gtk_widget_destroy (window);
  
  return;
}


void
polyxmass_filechoosers_atomdef_cancel_button_clicked (GtkWidget *widget,
							 gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *parent = NULL;


  g_assert (window != NULL);

  /* Get the pointer to the parent window, the one that is letting
     the user choose a atomdef ...
  */
  parent = g_object_get_data (G_OBJECT (window), "parent_window");
  g_assert (parent != NULL);
  
  /* Just set a datum to the parent window that's a NULL pointer.
   */
  g_object_set_data (G_OBJECT (parent), 
		     "atomdefs_filename", NULL);

  gtk_widget_destroy (window);
  
  return;
}



void
polyxmass_filechoosers_polchemdef_polseq_validate_button_clicked (GtkWidget *
								  widget,
								  gpointer 
								  data)
{
  GtkWidget *window = data;
  GtkWidget *parent = NULL;
  GtkWidget *frame = NULL;
  GtkWidget *messages_entry = NULL;

  GtkTreeView *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreeIter tree_iter_sel;
  GtkTreeSelection *tree_selection = NULL;

  PxmPolchemdefSpec *ps = NULL;

  gboolean is_item_selected = FALSE;

  gchar *filename = NULL;
  gchar *type = NULL;
  gchar *help = NULL;
  


  g_assert (window != NULL);

  /* Get the pointer to the parent window, the one that is letting the
     user choose a polchemdef or initializing a new polymer
     sequence file...
  */
  parent = g_object_get_data (G_OBJECT (window), "parent_window");
  g_assert (parent != NULL);
  
  messages_entry = g_object_get_data (G_OBJECT (window), "messages_entry");
  g_assert (messages_entry != NULL);
  
  /* One of the reasons why this window was used is to let the user
     select a polymer chemistry definition either only from a treeview
     listing the ones available on the system or also from a
     GtkFileChooser widget packed and show onto it.

     Another reason why this window might be displayed is because the
     user is asked to initialize the creation of a new polymer
     sequence file.

     Such functionalities are packed neatly in distinct GtkFrame
     widgets thare are either visible or hidden depending on what the
     window was created for in the first place.

     Each time one of the reasons above is true, its related GtkFrame
     is visible, so we use this criterium to know where to look for
     significant data to check here.

     IN ANY CASE, the treeview of available polchemdefs is shown as it
     is the primary destination of this window.
  */

  /* Set the size requisition for the big vpaned widget where everything
     is packed.
  */
  widget = g_object_get_data (G_OBJECT (window),
			      "polchemdef_polseq_vpaned");
  g_assert (widget != NULL);
  
  gtk_widget_set_size_request (widget, -1, -1);

  /* 1. *****************************************************************
     
  The treeview displaying all the polymer chemistry definitions
  available on the system.
  */
  
  /* There should be one item selected in the treeview. If not, then
     that means that the user might have choosen a file in the file
     chooser widget, if it was allowed to do so (ie the frame
     containing that file choosing widget is not hidden).
   */
  
  model = (GtkTreeModel *) g_object_get_data (G_OBJECT (window), 
					      "polchemdefs_treeview_model");
  g_assert (model != NULL);

  treeview= g_object_get_data (G_OBJECT (window), "polchemdefs_treeview");
  g_assert (treeview != NULL);

  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  is_item_selected = gtk_tree_selection_get_selected (tree_selection,
						   &model,
						   &tree_iter_sel);
  
  if (is_item_selected == FALSE)
    {
      /* No item is selected in the treeview, so check if the gtk
	 filechooser-containing vbox is visible. If so, try to get the
	 filename that might be selected...
      */
      widget = g_object_get_data 
	(G_OBJECT (window),
	 "show_hide_vbox_gtk_filechooser_polchemdef");
      
      g_assert (widget != NULL);
      
      if (FALSE == GTK_WIDGET_VISIBLE (widget))
	{
	  /* Just set a datum to the parent window that's a NULL
	     pointer.
	   */
	  g_object_set_data (G_OBJECT (frame), 
			     "polchemdefs_filename", NULL);
	}
      else
	{
	  /* At this point we know that the gtk filechooser widget is
	     currently visible, so check if the user has selected a
	     filename in it.
	  */
	  widget = g_object_get_data (G_OBJECT (window),
				      "gtk_filechooser_polchemdef_widget");
	  
	  filename = 
	    gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (widget));
	  
	  if (NULL == filename)
	    {
	      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
						"Choose a filename or select"
						" an item in the treeview",
						POLYXMASS_LONG_MSG_TIMEOUT);
	      
	      return;
	    }
	  else
	    {
	      /* Just set a datum to the parent window that's the
		 filename. When the parent window is destroyed, the allocated
		 filename is freed. Or the parent window is free to set a
		 datum by the same name to a NULL pointer, to trigger freeing
		 the filename immediately.
	      */
	      g_object_set_data_full (G_OBJECT (parent), 
				      "polchemdefs_filename", filename,
				      (GDestroyNotify) g_free);
	    }
	}
    }
  else
    /* 
       converse of if (is_item_selected == FALSE)
     */
    {
      /* At this point we know that one item was selected in the treeview,
	 so we should extract from it the filename and send data from it
	 as full data elements to the parent window (see above for details
	 about using the g_object_set_data_full function on the 'parent'
	 window). This is not true for the 'ps'-pointed-to object that is
	 not allocated here, 'ps' being only a reference to an item in an
	 array.
      */
      gtk_tree_model_get (model,
			  &tree_iter_sel,
			  COLUMN_POLCHEMSPEC_POINTER, &ps,
			  -1);
  
      g_object_set_data (G_OBJECT (parent), 
			 "polchemdefspec", ps);


      filename = g_strdup (ps->file);
  
      g_object_set_data_full (G_OBJECT (parent), 
			      "polchemdefs_filename", filename,
			      (GDestroyNotify) g_free);

  
      type = g_strdup (ps->type);

      g_object_set_data_full (G_OBJECT (parent), 
			      "polchemdefs_type", type,
			      (GDestroyNotify) g_free);
    }
  

  /* 2. *****************************************************************
     
  But we might not have finished with our work. It is possible that
  this window was set up also to let the user select a new filename
  for a new polymer sequence file and enter some data about the new
  polymer sequence, like the name of the polymer sequence, its code...

  When we did set up this window, we did either show
  (PXM_FILE_CHOOSE_POLSEQDATA_INIT bit was set in 'config' param of
  setup function) the "new_polseq_init_data_vbox" or hide it, in case
  this window was not set up for the user to initialize a new polymer
  sequence.

  So lets immediately see if that vbox is hidden or not.
  */
  widget = g_object_get_data (G_OBJECT (window),
			      "new_polseq_init_data_vbox");
  g_assert (widget != NULL);
  
  if (FALSE == GTK_WIDGET_VISIBLE (widget))
    {
      /* No, apparently this window was not set up to let the user
	 initialize a new polymer sequence, so we can just return now,
	 after having destroyed *this window.
      */

      /* It is crucial that we first remove all the pending timeout
	 messages, because we set very long-lasting error messages, and
	 before they are normally ended, this window might be closed by
	 the user validating the window!
      */
      polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
      
      gtk_widget_destroy (window);
      
      return;
    }
  

  /* Oh YES, apparently we still have to work on this window: get the
     filename for the new polymer sequence file.
  */
  widget = g_object_get_data (G_OBJECT (window),
			      "gtk_filechooser_new_polseq_widget");
  
  filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (widget));
  
  if (NULL == filename)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					"Choose a filename for the new"
					" polymer sequence file",
					POLYXMASS_LONG_MSG_TIMEOUT);
      
      return;
    }
  else
    {
      /* Just set a datum to the parent window that's the
	 filename. When the parent window is destroyed, the allocated
	 filename is freed. Or the parent window is free to set a
	 datum by the same name to a NULL pointer, to trigger freeing
	 the filename immediately.
      */
      g_object_set_data_full (G_OBJECT (parent), 
			      "new_polseq_filename", filename,
			      (GDestroyNotify) g_free);
    }
  

  /* Now let's work with the data in the polymer sequence-specific
     entries...
  */
  widget = g_object_get_data (G_OBJECT (window),
			      "polseq_name_entry");
  g_assert (widget != NULL);
  
  help = gtk_editable_get_chars (GTK_EDITABLE (widget),
				 0, -1);
  g_assert (help != NULL);
  
  g_object_set_data_full (G_OBJECT (parent), 
			  "polseq_name_entry_data", help,
			  (GDestroyNotify) g_free);
  

  widget = g_object_get_data (G_OBJECT (window),
			      "polseq_code_entry");
  g_assert (widget != NULL);
  
  help = gtk_editable_get_chars (GTK_EDITABLE (widget),
				 0, -1);
  g_assert (help != NULL);
  
  g_object_set_data_full (G_OBJECT (parent), 
			  "polseq_code_entry_data", help,
			  (GDestroyNotify) g_free);
  

  /* OK, finally all is done...
   */  
  /* It is crucial that we first remove all the pending timeout
     messages, because we set very long-lasting error messages, and
     before they are normally ended, this window might be closed by
     the user validating the window!
   */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  gtk_widget_destroy (window);
  
  return;
}


void
polyxmass_filechoosers_polchemdef_polseq_cancel_button_clicked (GtkWidget *
								widget,
								gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *parent = NULL;


  g_assert (window != NULL);

  /* Get the pointer to the parent window, the one that is letting
     the user choose a polchemdef ...
  */
  parent = g_object_get_data (G_OBJECT (window), "parent_window");
  g_assert (parent != NULL);
  
  /* Just set a datum to the parent window that's a NULL pointer.
   */
  g_object_set_data (G_OBJECT (parent), 
		     "polchemdefs_filename", NULL);

  /* It is crucial that we first remove all the pending timeout
     messages, because we set very long-lasting error messages, and
     before they are normally ended, this window might be closed by
     the user validating the window!
   */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  gtk_widget_destroy (window);
  
  return;
}





/* WINDOW CLOSING FUNCTIONS
 */
gboolean
polyxmass_filechoosers_atomdef_wnd_delete_event (GtkWidget *window,
						     GdkEvent *event,
						     gpointer data)
{

  return FALSE;
}

gboolean
polyxmass_filechoosers_atomdef_wnd_destroy_event (GtkWidget *window,
						      GdkEvent *event,
						      gpointer data)
{
  return FALSE;
}



/* WINDOW CLOSING FUNCTIONS
 */
gboolean
polyxmass_filechoosers_polchemdef_polseq_wnd_delete_event (GtkWidget *window,
							   GdkEvent *event,
							   gpointer data)
{
  /* It is crucial that we first remove all the pending timeout
     messages, because we set very long-lasting error messages, and
     before they are normally ended, this window might be closed by
     the user validating the window!
   */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  return FALSE;
}

gboolean
polyxmass_filechoosers_polchemdef_polseq_wnd_destroy_event (GtkWidget *window,
							    GdkEvent *event,
							    gpointer data)
{
  return FALSE;
}

