/*
 * ratPGPprog.c --
 *
 *	This file contains compatibility functions.
 *
 * TkRat software and its included text is Copyright 1996,1997,1998
 * by Martin Forssn
 *
 * Postilion software and its included text and images
 * Copyright (C) 1998 Nic Bernstein
 *
 * The full text of the legal notices is contained in the files called
 * COPYING and COPYRIGHT.TkRat, included with this distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "ratFolder.h"
#include "ratPGP.h"

/*
 * The contents of teh first bodypart of nested bodyparts
 */
#define ENCFIRST "Version: 1\r\n"

static int RatRunPGP(Tcl_Interp *interp, int interactive, char *args,
		     int *toPGP, int *fromPGP, int *errPGP);
static Tcl_DString *RatPGPRunOld(Tcl_Interp *interp, BodyInfo *bodyInfoPtr,
				 char *text, char *start, char *end);


/*
 *----------------------------------------------------------------------
 *
 * RatRunPGP --
 *
 *      Run the pgp command
 *
 * Results:
 *	Returns the pid of the pgp program on success and a negative
 *	value on failure. The toPGP and fromPGP integers will be modified.
 *
 * Side effects:
 *	forks.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatRunPGP(Tcl_Interp *interp, int interactive, char *args, int *toPGP,
	int *fromPGP, int *errPGP)
{
    int toPipe[2], fromPipe[2], errPipe[2], argc, pid;
    Tcl_DString cmd;
    char *pgp, **argv, *cargs;

    /*
     * Setup command arrays
     */
    pgp = Tcl_GetVar2(interp, "option", "pgp_prog", TCL_GLOBAL_ONLY);
    cargs = Tcl_GetVar2(interp, "option", "pgp_args", TCL_GLOBAL_ONLY);
    Tcl_DStringInit(&cmd);
    Tcl_DStringAppend(&cmd, pgp, -1);
    if (strlen(cargs)) {
	Tcl_DStringAppend(&cmd, " ", 1);
	Tcl_DStringAppend(&cmd, cargs, -1);
    }
    Tcl_DStringAppend(&cmd, " ", 1);
    Tcl_DStringAppend(&cmd, args, -1);
    Tcl_SplitList(interp, Tcl_DStringValue(&cmd), &argc, &argv);
    Tcl_DStringFree(&cmd);

    /*
     * Create the pgp subprocess and create a handler on the from pipe.
     */
    pipe(toPipe);
    pipe(fromPipe);
    pipe(errPipe);
    if (0 == (pid = fork())) {
	close(toPipe[1]);
	close(fromPipe[0]);
	close(errPipe[0]);
	dup2(toPipe[0], 0);
	dup2(fromPipe[1], 1);
	dup2(errPipe[1], 2);
	fcntl(0, F_SETFD, 0);
	fcntl(1, F_SETFD, 0);
	fcntl(2, F_SETFD, 0);
	if (!interactive) {
	    putenv("PGPPASSFD=0");
	}
	execv(pgp, argv);
	{
	    char buf[1024];
	    sprintf(buf, "ERROR executing '%s %s': %s\n", pgp, args,
							  strerror(errno));
	    write(1, buf, strlen(buf));
	}
	exit(-1);
	/* notreached */
    }
    close(toPipe[0]);
    close(fromPipe[1]);
    close(errPipe[1]);
    free(argv);
    *toPGP = toPipe[1];
    *fromPGP = fromPipe[0];
    *errPGP = errPipe[0];
    return pid;
}

/*
 *----------------------------------------------------------------------
 *
 * RatPGPEncrypt --
 *
 *      Encrypt a bodypart. Optionally also sign it.
 *
 * Results:
 *	A multipart/encrypted body
 *
 * Side effects:
 *	Will call pgp.
 *
 *
 *----------------------------------------------------------------------
 */

BODY*
RatPGPEncrypt(Tcl_Interp *interp, ENVELOPE *env, BODY *body, int sign)
{
    int toPGP, fromPGP, errPGP, length, retry, pid, result, status;
    char *passPhrase = NULL, *hdrPtr;
    char buf[MAILTMPLEN];
    ADDRESS *adrPtr;
    Tcl_DString cmdDS, encDS;
    BODY *multiPtr;
    PARAMETER *parmPtr;
    PART *partPtr;

    Tcl_DStringInit(&cmdDS);
    Tcl_DStringInit(&encDS);

    rfc822_encode_body_8bit(env, body);

    /*
     * Create command to run
     */
    Tcl_DStringAppend(&cmdDS, "+BATCHMODE +VERBOSE=0 -eaf", -1);
    if (sign) {
	Tcl_DStringAppend(&cmdDS, "s", 1);
    }
    for (adrPtr = env->to; adrPtr; adrPtr = adrPtr->next) {
	buf[0] = '\0';
	rfc822_address(buf, adrPtr);
	Tcl_DStringAppend(&cmdDS, " ", 1);
	Tcl_DStringAppend(&cmdDS, buf, -1);
    }
    for (adrPtr = env->cc; adrPtr; adrPtr = adrPtr->next) {
	buf[0] = '\0';
	rfc822_address(buf, adrPtr);
	Tcl_DStringAppend(&cmdDS, " ", 1);
	Tcl_DStringAppend(&cmdDS, buf, -1);
    }
    for (adrPtr = env->bcc; adrPtr; adrPtr = adrPtr->next) {
	buf[0] = '\0';
	rfc822_address(buf, adrPtr);
	Tcl_DStringAppend(&cmdDS, " ", 1);
	Tcl_DStringAppend(&cmdDS, buf, -1);
    }

    /*
     * Run command
     */
    do {
	if (sign) {
	    passPhrase = RatSenderPGPPhrase(interp);
	    if (NULL == passPhrase) {
		Tcl_DStringFree(&encDS);
		Tcl_DStringFree(&cmdDS);
		return NULL;
	    }
	}
	pid = RatRunPGP(interp, 0, Tcl_DStringValue(&cmdDS), &toPGP,
			&fromPGP, &errPGP);
	if (sign) {
	    write(toPGP, passPhrase, strlen(passPhrase));
	    memset(passPhrase, '\0', strlen(passPhrase));
	}
	write(toPGP, "\n", 1);
	buf[0] = '\0';
	hdrPtr = buf;
	rfc822_write_body_header(&hdrPtr, body);
	strcat(hdrPtr, "\015\012");
	write(toPGP, buf, strlen(buf));
	RatInitDelaySoutr();
	rfc822_output_body(body, RatDelaySoutr, (void*)toPGP);
	close(toPGP);

	/*
	 * Read result
	 */
	Tcl_DStringSetLength(&encDS, 0);
	do {
	    length = read(fromPGP, buf, sizeof(buf));
	    if (length) {
		Tcl_DStringAppend(&encDS, buf, length);
	    }
	} while (length > 0);
	close(fromPGP);

	/*
	 * Check for errors
	 */
	do {
	    result = waitpid(pid, &status, 0);
	} while(-1 == result && EINTR == errno);
	if (pid != result || WEXITSTATUS(status)) {
	    Tcl_DString error;
	    char *reply;

	    Tcl_DStringInit(&error);
	    Tcl_DStringAppendElement(&error, "PGP");
	    Tcl_DStringAppendElement(&error, "error");
	    Tcl_DStringStartSublist(&error);
	    do {
		length = read(errPGP, buf, sizeof(buf));
		if (length) {
		    Tcl_DStringAppend(&error, buf, length);
		}
	    } while (length > 0);
	    Tcl_DStringEndSublist(&error);

	    reply = RatSendPGPCommand(Tcl_DStringValue(&error));
	    Tcl_DStringFree(&error);
	    if (!strncmp("ABORT", reply, 5)) {
		close(errPGP);
		Tcl_DStringFree(&encDS);
		Tcl_DStringFree(&cmdDS);
		return NULL;
	    }
	    retry = 1;
	} else {
	    retry = 0;
	}
	close(errPGP);
    } while(0 != retry);
    Tcl_DStringFree(&cmdDS);
    mail_free_body(&body);

    /*
     * Build encrypted multipart
     */
    multiPtr = mail_newbody();
    multiPtr->type = TYPEMULTIPART;
    multiPtr->subtype = cpystr("encrypted");
    multiPtr->parameter = parmPtr = mail_newbody_parameter();
    parmPtr->attribute = cpystr("protocol");
    parmPtr->value = cpystr("application/pgp-encrypted");
    parmPtr->next = NULL;
    multiPtr->encoding = ENC7BIT;
    multiPtr->id = NULL;
    multiPtr->description = NULL;
    multiPtr->nested.part = partPtr = mail_newbody_part();
    partPtr->body.type = TYPEAPPLICATION;
    partPtr->body.subtype = cpystr("pgp-encrypted");
    partPtr->body.encoding = ENC7BIT;
    partPtr->body.contents.text.data = cpystr(ENCFIRST);
    partPtr->body.size.bytes = strlen(ENCFIRST);
    partPtr->next = mail_newbody_part();
    partPtr = partPtr->next;
    partPtr->body.type = TYPEAPPLICATION;
    partPtr->body.subtype = cpystr("octet-stream");
    partPtr->body.encoding = ENC7BIT;
    partPtr->body.contents.text.data = cpystr(Tcl_DStringValue(&encDS));
    partPtr->body.size.bytes = Tcl_DStringLength(&encDS);
    Tcl_DStringFree(&encDS);
    partPtr->next = NULL;

    return multiPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * RatPGPSign --
 *
 *      Sign a bodypart.
 *
 * Results:
 *	A multipart/signed body
 *
 * Side effects:
 *	Will call pgp.
 *
 *
 *----------------------------------------------------------------------
 */

BODY*
RatPGPSign(Tcl_Interp *interp, BODY *body)
{
    int toPGP, fromPGP, errPGP, length, retry, pid, status, result;
    char *passPhrase, *hdrPtr;
    char buf[MAILTMPLEN];
    Tcl_DString sigDS;
    BODY *multiPtr;
    PARAMETER *parmPtr;
    PART *partPtr;

    Tcl_DStringInit(&sigDS);
    do {
	/*
	 * Run command
	 */
        passPhrase = RatSenderPGPPhrase(interp);
	if (NULL == passPhrase) {
	    return NULL;
	}
	rfc822_encode_body_7bit(NIL, body);
	pid = RatRunPGP(interp, 0, "+BATCHMODE +VERBOSE=0 -satbf",
		  	&toPGP, &fromPGP, &errPGP);
	write(toPGP, passPhrase, strlen(passPhrase));
	memset(passPhrase, '\0', strlen(passPhrase));
	write(toPGP, "\n", 1);
	buf[0] = '\0';
	hdrPtr = buf;
	rfc822_write_body_header(&hdrPtr, body);
	strcat(hdrPtr, "\015\012");
	write(toPGP, buf, strlen(buf));
	RatInitDelaySoutr();
	rfc822_output_body(body, RatDelaySoutr, (void*)toPGP);
	close(toPGP);

	/*
	 * Read result
	 */
	Tcl_DStringSetLength(&sigDS, 0);
	do {
	    length = read(fromPGP, buf, sizeof(buf));
	    if (length) {
		Tcl_DStringAppend(&sigDS, buf, length);
	    }
	} while (length > 0);
	close(fromPGP);

	/*
	 * Check for errors
	 */
	do {
	    result = waitpid(pid, &status, 0);
	} while(-1 == result && EINTR == errno);
	if (pid != result || WEXITSTATUS(status)) {
	    Tcl_DString error;
	    char *reply;

	    Tcl_DStringInit(&error);
	    Tcl_DStringAppendElement(&error, "PGP");
	    Tcl_DStringAppendElement(&error, "error");
	    Tcl_DStringStartSublist(&error);
	    do {
		length = read(errPGP, buf, sizeof(buf));
		if (length) {
		    Tcl_DStringAppend(&error, buf, length);
		}
	    } while (length > 0);
	    Tcl_DStringEndSublist(&error);

	    reply = RatSendPGPCommand(Tcl_DStringValue(&error));
	    Tcl_DStringFree(&error);
	    if (!strncmp("ABORT", reply, 5)) {
		close(errPGP);
		Tcl_DStringFree(&sigDS);
		return NULL;
	    }
	    retry = 1;
	} else {
	    retry = 0;
	}
	close(errPGP);
    } while(0 != retry);

    /*
     * Build signature multipart
     */
    multiPtr = mail_newbody();
    multiPtr->type = TYPEMULTIPART;
    multiPtr->subtype = cpystr("signed");
    multiPtr->parameter = parmPtr = mail_newbody_parameter();
    parmPtr->attribute = cpystr("micalg");
    parmPtr->value = cpystr("pgp-md5");
    parmPtr->next = mail_newbody_parameter();
    parmPtr = parmPtr->next;
    parmPtr->attribute = cpystr("protocol");
    parmPtr->value = cpystr("application/pgp-signature");
    parmPtr->next = NULL;
    multiPtr->encoding = ENC7BIT;
    multiPtr->id = NULL;
    multiPtr->description = NULL;
    multiPtr->nested.part = partPtr = mail_newbody_part();
    partPtr->body = *body;
    partPtr->next = mail_newbody_part();
    partPtr = partPtr->next;
    partPtr->body.type = TYPEAPPLICATION;
    partPtr->body.subtype = cpystr("pgp-signature");
    partPtr->body.encoding = ENC7BIT;
    partPtr->body.contents.text.data = cpystr(Tcl_DStringValue(&sigDS));
    partPtr->body.size.bytes = Tcl_DStringLength(&sigDS);
    Tcl_DStringFree(&sigDS);
    partPtr->next = NULL;

    return multiPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * FindBoundary --
 *
 *      Find the boundary in a message string.
 *
 * Results:
 *	Pointer to the start of the boundary.
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */

static char*
FindBoundary(char *text, char*boundary)
{
    char *cPtr = text;
    int l = strlen(boundary);

    do {
	if ('-' == cPtr[0] && '-' == cPtr[1] && !strncmp(cPtr+2, boundary, l)) {
	    return cPtr;
	}
	cPtr = strchr(cPtr, '\n');
    } while (cPtr++);
    return NULL;
}


/*
 *----------------------------------------------------------------------
 *
 * RatPGPChecksig --
 *
 *      Check the signature of a bodypart
 *
 * Results:
 *	None
 *
 * Side effects:
 *	Updates the bodyInfoPtr->sigStatus
 *
 *
 *----------------------------------------------------------------------
 */

void
RatPGPChecksig(Tcl_Interp *interp, MessageProcInfo* procInfo,
	BodyInfo *bodyInfoPtr)
{
    unsigned char *text;
    unsigned long length;

    /*
     * Check if PGP/MIME message or old style PGP.
     * The algorithms kind of differ:-)
     */
    if (bodyInfoPtr->secPtr) {
	char *boundary, *start, *end, *tmp;
	char buf[2048], textfile[1024], sigfile[1024];
	int fd, toPGP, fromPGP, errPGP, pid, status, result;
	PARAMETER *parPtr;
	Tcl_DString *resultDS = (Tcl_DString*)malloc(sizeof(Tcl_DString));

	/*
	 * Generate filenames
	 */
	tmp = Tcl_GetVar2(interp, "option", "tmp", TCL_GLOBAL_ONLY);
	RatGenId(NULL, interp, 0, NULL);
	sprintf(textfile, "%s/rat.%s", tmp, interp->result);
	strcpy(sigfile, textfile);
	strcat(sigfile, ".sig");

	/*
	 * Save text and signature in files
	 */
	boundary = NULL;
	text = (unsigned char*)(*procInfo[bodyInfoPtr->type].fetchBodyProc)
		(bodyInfoPtr->secPtr, &length);
	for (parPtr = bodyInfoPtr->secPtr->bodyPtr->parameter; parPtr;
		parPtr = parPtr->next) {
	    if (!strcasecmp(parPtr->attribute, "boundary")) {
		boundary = parPtr->value;
		break;
	    }
	}
	if (!boundary || NULL == (start = FindBoundary((char*)text, boundary))){
	    bodyInfoPtr->sigStatus = RAT_SIG_BAD;
	    return;
	}
	start += strlen(boundary) + 4;
	if (NULL == (end = FindBoundary(start, boundary))) {
	    bodyInfoPtr->sigStatus = RAT_SIG_BAD;
	    return;
	}
	end -= 2;
	fd = open(textfile, O_CREAT | O_TRUNC | O_WRONLY, 0666);
	write(fd, start, end-start);
	close(fd);
	text = (unsigned char*)(*procInfo[bodyInfoPtr->type].fetchBodyProc)
		(bodyInfoPtr->secPtr->firstbornPtr->nextPtr, &length);
	fd = open(sigfile, O_CREAT | O_TRUNC | O_WRONLY, 0666);
	write(fd, text, length);
	close(fd);

	/*
	 * Run PGP command
	 */
	sprintf(buf, "+batchmode +verbose=0 %s %s", sigfile, textfile);
	pid = RatRunPGP(interp, 0, buf, &toPGP, &fromPGP, &errPGP);
	close(toPGP);
	Tcl_DStringInit(resultDS);
	while ((length = read(errPGP, buf, sizeof(buf)))) {
	    Tcl_DStringAppend(resultDS, buf, length);
	}
	while ((length = read(fromPGP, buf, sizeof(buf)))) {
	    Tcl_DStringAppend(resultDS, buf, length);
	}
	close(fromPGP);
	close(errPGP);
	do {
	    result = waitpid(pid, &status, 0);
	} while(-1 == result && EINTR == errno);
	if (pid != result || WEXITSTATUS(status)) {
	    bodyInfoPtr->sigStatus = RAT_SIG_BAD;
	} else {
	    bodyInfoPtr->sigStatus = RAT_SIG_GOOD;
	}
	bodyInfoPtr->pgpOutput = resultDS;

	/*
	 * Clean up
	 */
	unlink(textfile);
	unlink(sigfile);
    } else {
	Tcl_DString *bodyDSPtr;
	char *start, *end;

	text = (unsigned char*)(*procInfo[bodyInfoPtr->type].fetchBodyProc)
		(bodyInfoPtr, &length);
	start = RatPGPStrFind((char*)text, length, "BEGIN PGP SIGNED", 1);
	end = RatPGPStrFind(start, length - (start-(char*)text), "END PGP ", 1);
	bodyDSPtr = RatPGPRunOld(interp, bodyInfoPtr, (char*)text, start,end+1);
	Tcl_DStringFree(bodyDSPtr);
	free(bodyDSPtr);
    }
    if (bodyInfoPtr->pgpOutput && 1<Tcl_DStringLength(bodyInfoPtr->pgpOutput)){
	Tcl_SetResult(interp, Tcl_DStringValue(bodyInfoPtr->pgpOutput),
		TCL_VOLATILE);
    } else {
	Tcl_ResetResult(interp);
    }
}


/*
 *----------------------------------------------------------------------
 *
 * RatPGPDecrypt --
 *
 *      Decryt a bodypart.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	The BodyInfo structure will be modified if the decryption is ok.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatPGPDecrypt(Tcl_Interp *interp, MessageProcInfo *procInfo,
	      BodyInfo **bodyInfoPtrPtr)
{
    int toPGP, fromPGP, errPGP, result, pid, retry, status;
    char *text, *passPhrase, *msg, buf[1024];
    BodyInfo *origPtr = *bodyInfoPtrPtr;
    MessageInfo *msgPtr;
    unsigned long length;
    Tcl_DString bodyDS, *errDSPtr = (Tcl_DString*)malloc(sizeof(Tcl_DString));

    msg = Tcl_GetVar2(interp, "t", "decrypting", TCL_GLOBAL_ONLY);
    RatLog(interp, RAT_PARSE, msg, 1);

    /*
     * Decode the bodypart
     */
    Tcl_DStringInit(&bodyDS);
    (*procInfo[(*bodyInfoPtrPtr)->type].makeChildrenProc)
	    (interp, *bodyInfoPtrPtr);
    text = (*procInfo[(*bodyInfoPtrPtr)->type].fetchBodyProc)
	    ((*bodyInfoPtrPtr)->firstbornPtr->nextPtr, &length);
    do {
	passPhrase = RatPGPPhrase(interp);
	if (NULL == passPhrase) {
	    break;
	}
	pid = RatRunPGP(interp, 0, "+BATCHMODE +VERBOSE=0 -f", &toPGP, &fromPGP,
							    &errPGP);
	write(toPGP, passPhrase, strlen(passPhrase));
	memset(passPhrase, '\0', strlen(passPhrase));
	free(passPhrase);
	write(toPGP, "\n", 1);
	write(toPGP, text, length);
	close(toPGP);

	/*
	 * Read result
	 */
	Tcl_DStringSetLength(&bodyDS, 0);
	Tcl_DStringAppend(&bodyDS, "MIME-Version: 1.0\r\n", -1);
	while ((length = read(fromPGP, buf, sizeof(buf)))) {
	    Tcl_DStringAppend(&bodyDS, buf, length);
	}
	close(fromPGP);
	Tcl_DStringInit(errDSPtr);
	while ((length = read(errPGP, buf, sizeof(buf)))) {
	    Tcl_DStringAppend(errDSPtr, buf, length);
	}
	close(errPGP);

	/*
	 * Check for errors?
	 */
	do {
	    result = waitpid(pid, &status, 0);
	} while(-1 == result && EINTR == errno);
	if (pid != result
		|| (WEXITSTATUS(status) != 0 && WEXITSTATUS(status) != 1)) {
	    Tcl_DString error;

	    ClearPGPPass(NULL);
	    Tcl_DStringInit(&error);
	    Tcl_DStringAppend(&error, "RatPGPError", -1);
	    Tcl_DStringAppendElement(&error, Tcl_DStringValue(errDSPtr));
	    if (TCL_OK != Tcl_Eval(interp, Tcl_DStringValue(&error))
		    || !strcmp("ABORT", interp->result)) {
		close(errPGP);
		Tcl_DStringFree(&error);
		Tcl_DStringFree(&bodyDS);
		Tcl_DStringFree(errDSPtr);
		free(errDSPtr);
		RatLog(interp, RAT_PARSE, "", 1);
		return;
	    } else {
		retry = 1;
	    }
	} else {
	    retry = 0;
	}
    } while (0 != retry);

    /*
     * Now parse the bodypart
     */
    Tcl_DeleteCommand(interp, (*bodyInfoPtrPtr)->cmdName);
    (*bodyInfoPtrPtr)->containedEntity = RatFrMessageCreate(interp,
	    Tcl_DStringValue(&bodyDS), Tcl_DStringLength(&bodyDS),
	    &msgPtr);
    Tcl_DStringFree(&bodyDS);
    *bodyInfoPtrPtr = Fr_CreateBodyProc(interp, msgPtr);
    msgPtr->bodyInfoPtr = NULL;
    if (WEXITSTATUS(status)) {
	(*bodyInfoPtrPtr)->sigStatus = RAT_UNSIGNED;
    } else {
	(*bodyInfoPtrPtr)->sigStatus = RAT_SIG_GOOD;
    }
    (*bodyInfoPtrPtr)->pgpOutput = errDSPtr;
    (*bodyInfoPtrPtr)->altPtr = origPtr;
    RatLog(interp, RAT_PARSE, "", 1);
}

/*
 *----------------------------------------------------------------------
 *
 * RatPGPListKeys --
 *
 *      Lists the keys on a keyring
 *
 * Results:
 *	See ../doc/interface.
 *
 * Side effects:
 *	Runs the pgp command
 *
 *
 *----------------------------------------------------------------------
 */

int
RatPGPListKeys(Tcl_Interp *interp, char *keyring)
{
    int toPGP, fromPGP, errPGP, pid, i, status, length, ret;
    Tcl_DString cmd, result;
    Tcl_RegExp exp;
    char buf[1024], buf2[1024], *start, *end;
    FILE *fp;

    Tcl_DStringInit(&cmd);
    Tcl_DStringInit(&result);

    Tcl_DStringAppend(&cmd, "-kv +BATCHMODE +VERBOSE=0 ", -1);
    if (keyring) {
	Tcl_DStringAppend(&cmd, keyring, -1);
    }
    pid = RatRunPGP(interp, 0, Tcl_DStringValue(&cmd), &toPGP,&fromPGP,&errPGP);
    Tcl_DStringFree(&cmd);
    fp = fdopen(fromPGP, "r");
    close(toPGP);

    /*
     * Read output
     */
    buf[sizeof(buf)-1] = '\0';
    exp = Tcl_RegExpCompile(interp,
	    "^([a-z]+)[ \t]+([0-9]+)/([A-F0-9]+)[ \t]+([0-9/]+)[ \t]+(.+)$");
    while (fgets(buf, sizeof(buf)-1, fp), !feof(fp)) {
	buf[strlen(buf)-1] = '\0';
	if (Tcl_RegExpExec(interp, exp, buf, buf)) {
	    Tcl_DStringStartSublist(&result);
	    for (i=1; i<=5; i++) {
		Tcl_RegExpRange(exp, i, &start, &end);
		strncpy(buf2, start, end-start);
		buf2[end-start] = '\0';
		Tcl_DStringAppendElement(&result, buf2);
	    }
	    Tcl_DStringEndSublist(&result);
	}
    }
    fclose(fp);

    /*
     * Check for errors?
     */
    do {
	ret = waitpid(pid, &status, 0);
    } while(-1 == ret && EINTR == errno);
    if (pid != ret
	    || (WEXITSTATUS(status) != 0 && WEXITSTATUS(status) != 1)) {
	Tcl_DStringSetLength(&result, 0);
	do {
	    length = read(errPGP, buf, sizeof(buf));
	    if (length) {
		Tcl_DStringAppend(&result, buf, length);
	    }
	} while (length > 0);
	close(errPGP);
	Tcl_DStringResult(interp, &result);
	return TCL_ERROR;
    }
    close(errPGP);
    Tcl_DStringResult(interp, &result);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatPGPExtractKey --
 *
 *      Extracts a key from a keyring
 *
 * Results:
 *	See ../doc/interface.
 *
 * Side effects:
 *	Runs the pgp command
 *
 *
 *----------------------------------------------------------------------
 */

int
RatPGPExtractKey(Tcl_Interp *interp, char *id, char *keyring)
{
    int toPGP, fromPGP, errPGP, pid, status, length, ret;
    Tcl_DString cmd, result;
    char buf[1024];
    char *cPtr;

    Tcl_DStringInit(&cmd);
    Tcl_DStringInit(&result);

    Tcl_DStringAppend(&cmd, "-kxaf +BATCHMODE +VERBOSE=0 ", -1);
    Tcl_DStringAppend(&cmd, "\"", 1);
    for (cPtr = id; *cPtr; cPtr++) {
      if ('"' == *cPtr) {
          Tcl_DStringAppend(&cmd, "\\\"", 2);
      } else {
          Tcl_DStringAppend(&cmd, cPtr, 1);
      }
    }
    Tcl_DStringAppend(&cmd, "\"", 1);
    if (keyring) {
	Tcl_DStringAppend(&cmd, keyring, -1);
    }
    pid = RatRunPGP(interp, 0, Tcl_DStringValue(&cmd), &toPGP,&fromPGP,&errPGP);
    Tcl_DStringFree(&cmd);
    close(toPGP);

    /*
     * Read output
     */
    do {
	length = read(fromPGP, buf, sizeof(buf));
	if (length) {
	    Tcl_DStringAppend(&result, buf, length);
	}
    } while (length > 0);
    close(fromPGP);

    /*
     * Check for errors?
     */
    do {
	ret = waitpid(pid, &status, 0);
    } while(-1 == ret && EINTR == errno);
    if (pid != ret
	    || (WEXITSTATUS(status) != 0 && WEXITSTATUS(status) != 1)) {
	Tcl_DStringSetLength(&result, 0);
	do {
	    length = read(errPGP, buf, sizeof(buf));
	    if (length) {
		Tcl_DStringAppend(&result, buf, length);
	    }
	} while (length > 0);
	close(errPGP);
	Tcl_DStringResult(interp, &result);
	return TCL_ERROR;
    }
    close(errPGP);
    Tcl_DStringResult(interp, &result);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatPGPAddKeys --
 *
 *      Adds keys to a keyring
 *
 * Results:
 *	See ../doc/interface.
 *
 * Side effects:
 *	Runs the pgp command
 *
 *
 *----------------------------------------------------------------------
 */

int
RatPGPAddKeys(Tcl_Interp *interp, char *keys, char *keyring)
{
    Tcl_DString cmd;
    int result;

    /*
     * Setup and execute command
     */
    Tcl_DStringInit(&cmd);
    Tcl_DStringAppendElement(&cmd, "RatPGPAddKeys");
    Tcl_DStringAppendElement(&cmd, keys);
    if (keyring) {
	Tcl_DStringAppendElement(&cmd, keyring);
    }
    result = Tcl_Eval(interp, Tcl_DStringValue(&cmd));
    Tcl_DStringFree(&cmd);
    return result;
}


/*
 *----------------------------------------------------------------------
 *
 * RatPGPRunOld --
 *
 *      Handle an bodypart generated by pgp (which is not PGP/MIME).
 *
 * Results:
 *	None
 *
 * Side effects:
 *	The BodyInfo structure will be modified.
 *
 *
 *----------------------------------------------------------------------
 */

static Tcl_DString*
RatPGPRunOld(Tcl_Interp *interp, BodyInfo *bodyInfoPtr, char *text,
		char *start, char *end)
{
    Tcl_DString *errDSPtr = (Tcl_DString*)malloc(sizeof(Tcl_DString)),
		*bodyDSPtr = (Tcl_DString*)malloc(sizeof(Tcl_DString));
    int needPhrase, toPGP, fromPGP, errPGP, length, preamble, pid, status,
	result, retry;
    char *cPtr, *passPhrase, buf[1024];

    /*
     * Prepare text part
     */
    Tcl_DStringInit(bodyDSPtr);
    Tcl_DStringAppend(bodyDSPtr, text, start-text);

    /*
     * Setup and run pgp command
     */
    if (!(cPtr = strchr(end, '\n'))) {
	cPtr = end + strlen(end);
    }
    needPhrase = strncmp(start, "-----BEGIN PGP SIGNED", 21);
    preamble = Tcl_DStringLength(bodyDSPtr);
    do {
	if (needPhrase) {
	    passPhrase = RatPGPPhrase(interp);
	    if (NULL == passPhrase) {
		Tcl_DStringAppend(bodyDSPtr, start, cPtr-start);
		free(errDSPtr);
		bodyInfoPtr->pgpOutput = NULL;
		return bodyDSPtr;
	    }
	} else {
	    passPhrase = "";
	}
	pid = RatRunPGP(interp, 0, "+BATCHMODE +VERBOSE=0 -f", &toPGP, &fromPGP,
							    &errPGP);
	write(toPGP, passPhrase, strlen(passPhrase));
	memset(passPhrase, '\0', strlen(passPhrase));
	free(passPhrase);
	write(toPGP, "\n", 1);
	write(toPGP, start, cPtr-start);
	close(toPGP);

	/*
	 * Read result
	 */
	Tcl_DStringSetLength(bodyDSPtr, preamble);
	while ((length = read(fromPGP, buf, sizeof(buf)))) {
	    Tcl_DStringAppend(bodyDSPtr, buf, length);
	}
	close(fromPGP);
	Tcl_DStringInit(errDSPtr);
	while ((length = read(errPGP, buf, sizeof(buf)))) {
	    Tcl_DStringAppend(errDSPtr, buf, length);
	}
	close(errPGP);

	/*
	 * Check for errors?
	 */
	do {
	    result = waitpid(pid, &status, 0);
	} while(-1 == result && EINTR == errno);
	if (pid != result
		|| (WEXITSTATUS(status) != 0 && WEXITSTATUS(status) != 1)) {
	    Tcl_DString error;

	    ClearPGPPass(NULL);
	    Tcl_DStringInit(&error);
	    Tcl_DStringAppend(&error, "RatPGPError", -1);
	    Tcl_DStringAppendElement(&error, Tcl_DStringValue(errDSPtr));
	    if (TCL_OK != Tcl_Eval(interp, Tcl_DStringValue(&error))
		    || !strcmp("ABORT", interp->result)) {
		close(errPGP);
		Tcl_DStringFree(&error);
		Tcl_DStringFree(errDSPtr);
		free(errDSPtr);
		RatLog(interp, RAT_PARSE, "", 1);
		Tcl_DStringAppend(bodyDSPtr, start, cPtr-start);
		bodyInfoPtr->pgpOutput = NULL;
		return bodyDSPtr;
	    } else {
		retry = 1;
	    }
	} else {
	    retry = 0;
	}
    } while (0 != retry);
    if (WEXITSTATUS(status)) {
	if (needPhrase) {
	    bodyInfoPtr->sigStatus = RAT_UNSIGNED;
	} else {
	    bodyInfoPtr->sigStatus = RAT_SIG_BAD;
	}
    } else {
	bodyInfoPtr->sigStatus = RAT_SIG_GOOD;
    }
    bodyInfoPtr->pgpOutput = errDSPtr;

    return bodyDSPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * RatPGPHandleOld --
 *
 *      Handle an bodypart generated by pgp (which is not PGP/MIME).
 *
 * Results:
 *	None
 *
 * Side effects:
 *	The BodyInfo structure will be modified.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatPGPHandleOld(Tcl_Interp *interp, BodyInfo *bodyInfoPtr, char *text,
		char *start, char *end)
{
    if (strncmp(start, "-----BEGIN PGP SIGNED", 21)) {
	char *cPtr, *t;

	bodyInfoPtr->decodedTextPtr = RatPGPRunOld(interp, bodyInfoPtr, text,
						start, end);
	if (!(cPtr = strchr(end, '\n'))) {
	    cPtr = end + strlen(end);
	}
	if (*cPtr) {
	    Tcl_DStringAppend(bodyInfoPtr->decodedTextPtr, cPtr, -1);
	}
	if (bodyInfoPtr->pgpOutput
		&& 1 < Tcl_DStringLength(bodyInfoPtr->pgpOutput)) {
	    Tcl_DString cmd;

	    Tcl_DStringInit(&cmd);
	    Tcl_DStringAppendElement(&cmd, "RatText");
	    t = Tcl_GetVar2(interp, "t", "pgp_output", TCL_GLOBAL_ONLY);
	    Tcl_DStringAppendElement(&cmd, t);
	    Tcl_DStringAppendElement(&cmd,
		    Tcl_DStringValue(bodyInfoPtr->pgpOutput));
	    Tcl_Eval(interp, Tcl_DStringValue(&cmd));
	    Tcl_DStringFree(&cmd);
	}

    } else {
	bodyInfoPtr->sigStatus = RAT_UNCHECKED;
    }
}
