/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef CPUFREQ_MANAGEMENT_H
#define CPUFREQ_MANAGEMENT_H

#include <list>

#include "config_pm.h"
#include "event_management.h"

/** @brief class caring about all cpus */
class CpufreqManagement {
public:
	/** @brief constructor */
	CpufreqManagement();

	/** @brief destructor */
	~CpufreqManagement();

	/** @brief This can be called outside after calling createCPUFreqObject()
	 *  
	 *  @return true if supported
	 */
	bool isSupported();

	/** @brief set cpufreq policy for all CPUs
	 *
	 * @param mode the new mode to set
	 * @param eM object to throw apporpriate events or NULL
	 */
	bool setModes(CPUFREQ_MODE mode, EventManagement *eM);

	/** @brief get the current cpu policy
	 *
	 * @return the current cpu policy
	 */
	CPUFREQ_MODE getMode();

	/** @brief get the current control mode used
	 *
	 * @return @ref CPUFREQ_CONTROL_MODE
	 */
	CPUFREQ_CONTROL_MODE controlMode();

	/** @brief enables a CPU
	 *
	 * @param cpu CPU to disable starting from 0
	 * @return @ref CPUFreq_Interface::enable()
	 */
	int enableCPU(int cpu);

	/** @brief disables a CPU
	 *
	 * @param cpu CPU to disable starting from 0
	 * @return @ref CPUFreq_Interface::enable()
	 */	
	int disableCPU(int cpu);

private:
	/** @brief checks CPU hotplugging depending on the scheme */
	void checkCPUHotplug();

	/** fetches the current active governor from Hal
	 *
	 * @return the governor, the string doesn't have to be freed
	 */
	char *getGovernor();

	/** @brief sets a specific governor
	 *
	 * @param governor the governor to set
	 *
	 * @return true on success, false otherwise
	 */
	bool setGovernor(const char *governor);

	/** @brief sets the performance for a dynamic scaling mechanism
	 *
	 * @param performance the performance to set between 1 and 100
	 *
	 * @return true on success, false otherwise
	 */
	bool setPerformance(int performance);

	/** @brief sets if niced processes should be considered on cpu
	 *         load calculation
	 *
	 * @param consider boolean
	 *
	 * @return true on success, false otherwise
	 */
	bool setConsiderNice(int consider);

	/** @brief number of CPUs on startup */
	int _initial_cpu_count;

	/** @brief if CPUFreq is supported at all */
	bool _is_supported;

	/** @brief the hal device udi which exposes the cpufreq methods */
	std::string _hal_device;
};

#endif // CPUFREQ_MANAGEMENT_H
