/* 
 * Prospect: a developer's system profiler.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Author: Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: incache.c,v 1.12 2004/01/09 20:29:28 type2 Exp $ */

/*
********************************************************************************
**
**                          PROSPECT PROJECT
**                           INCACHE MODULE
**
********************************************************************************
**
**      This  module  is  for   support  of  the   instruction   profile
**      disassembly feature of Prospect.  It has an initialization entry
**      that allocates memory for the instructions and hit counts.  Then
**      an entry that is called to put the  instructions  in memory, and
**      finally  when  the  last  data  is  input  (it  knows  from  the
**      initialization) it decides what data to output.
**
********************************************************************************
*/

#ifndef __LINT__
static const char gRCSid[] = "@(#) $Id: incache.c,v 1.12 2004/01/09 20:29:28 type2 Exp $";
#endif

/*
 * System Header Files
 */
#include <sys/types.h>
#include <sys/param.h>
#include <string.h>

/*
 * InCache Module Files
 */
#include "incache.h"
#include "malloc_debug.h"
#include "f_sym_name.h"
#include "linux_model.h"
#include "prospect.h"
#include "../dass_gdb/dass.h"

/*
 * InCache prototypes
 */
static void flush_cache(void);
static void pinstr(unsigned long PC1, unsigned long PC2, unsigned int Hits);
static void *das_get(char *name);

/*
 * InCache Instruction Type:
 */
typedef struct _instr_asm
{
        unsigned long  ia_PC;          /* Real address of the hit */
        unsigned int   ia_Instr;       /* Instruction at that address */
        unsigned int   ia_Hits;        /* Number hits at that address */
} instr_asm_t, *Pinstr_asm_t, tINSTRASM, *tPINSTRASM;


/*
 * InCache File Static Variables.
 */
static  unsigned long   gInPregVaddr;        /* Absolute Pregion address     */
static  unsigned long   gInPoffValue;        /* Symbol offset into Pregion   */
static  unsigned long   gOldSymsValue;       /* Local Symbol offset into Preg*/
static  unsigned int    gInSymCnt;           /* Instr hits to symbol name    */
static  unsigned int    gInNumInstr;         /* Unique PC's instr in symbol  */
static  syms_key_t     *gInPsysms_key;       /* Pointer to current symb info */
static  unsigned int    gInCIcptr;           /* Next element gInPinstr_asm   */

/*
 * InCache Memory for PC hits:
 */
static  tPINSTRASM     gInPinstr_asm = (tPINSTRASM)0;

/*
 * InCache list of opened disassemblers.  We keep a list of pipes open
 * to gdb's.  The number is configurable via the -g option with default
 * of 8.  Each struct in the list corrolates a gdb with a file.  This is
 * a doubly linked list and we traverse it every time through the 
 * init_incache function.  When an open gdb is used, it is moved to the head
 * of the list.  Pipes that fall off the end of the list are closed.
 */
typedef struct _dl_head {
    struct _dl_node   *dlh_head;
    struct _dl_node   *dlh_tail;
    unsigned int       dlh_num_entries;
} Dl_head_t;

typedef struct _dl_node {
    struct _dl_node   *dln_next;
    struct _dl_node   *dln_prev;
    char              *dln_aout_name;
    void              *dln_handle;
} Dl_node_t;

static Dl_head_t *gDl_head=NULL;
static void      *gDasHandle=NULL;

/*
 * Statistics for debug array.
 */
enum {
    DIS_OPENS,
    DIS_CLOSES,
    DIS_SHIFTS,
    DIS_DROPS,
    DIS_OPEN_FAILS,
    DIS_HITS,
    DIS_HITS_ON_FIRST,
    DIS_MISSES,
    DIS_CALLS,
    DIS_CALL_FAILS,
    DIS_MAX_STATS
};
static unsigned long gDstats[DIS_MAX_STATS]={0,};

/*
 * Initialize the Instruction Cache routine for single symbol.
 */
void
init_incache(unsigned long     PregVaddr,      /* Absolute Pregion address     */
             unsigned long     PoffValue,      /* Symbol offset into Pregion   */
             unsigned int      SymCnt,         /* Instr hits to symbol name    */
             unsigned int      NumInstr,       /* Unique PC's instr in symbol  */
             syms_key_t       *P_sym_key       /* Needed for iDasm() routine   */
            )
{
    gInPregVaddr = PregVaddr;
    gOldSymsValue = gInPoffValue   = PoffValue;
    gInSymCnt = SymCnt;
    gInNumInstr = NumInstr;
    gInPsysms_key = P_sym_key;

    if (gConf.flags.dis_debug) {
        if (gConf.bug_level) {
            ferr("init_incache:\n");
            ferr("\tgInPregVaddr = 0x%lx\n", gInPregVaddr);
            ferr("\tgOldSymsValue = 0x%lx\n", gOldSymsValue);
            ferr("\tgInSymCnt = %d\n", gInSymCnt);
            ferr("\tgInNumInstr = %d\n", gInNumInstr);
            ferr("\tgInPsysms_key = 0x%lx\n", gInPsysms_key);
            ferr("\tgAoutName = %s\n", gInPsysms_key->sk_Path);
        }
    }

    if (gInPinstr_asm != (tPINSTRASM)0)
    {
        /* Get rid of old malloc'ed memory */
        FREE(gInPinstr_asm);
    }

    /* Allocate new memory to save the instructions and counts.  */
    gInPinstr_asm = (tPINSTRASM)MALLOC(NumInstr * sizeof(tINSTRASM));

    /* Next available element in 'gInPinstr_asm' */
    gInCIcptr = (unsigned int)0;

    /* Initialize the disassembly list if necessary */
    if (gDl_head==NULL) {
        gDl_head = (Dl_head_t*)CALLOC(1,sizeof(Dl_head_t));
        if(!gDl_head) {
            ferr("\nDisaster: Calloc returns nill for gDl_head!!!\n");
            prospect_exit(1);
        }
    }
    /* get open pipe, or open new if neccessary */
    gDasHandle = das_get(gInPsysms_key->sk_Path);

    return;
} /* init_incache() */

/*
 * void close_incache(void)
 *
 * Clean up routine, should be called before main exits.
 * Close all child gdbs if any.  Freeing memory is optional,
 * I suppose.
 */
void
close_incache(void)
{
    if (gDl_head) {
        Dl_node_t *n = gDl_head->dlh_head;
        while (n) {
            if (gConf.flags.dis_debug) 
                ferr("dis: closing pipe for: %s\n", n->dln_aout_name);
            dass_close(n->dln_handle);
            gDstats[DIS_CLOSES]++;
            n = n->dln_next;
        }

        if (gConf.flags.dis_debug) {
           ferr("dis: Final disassembly stats:\n");
           ferr("     Max slots:             %d\n", gConf.dis_list_size);
           ferr("     Max slots used:        %ld\n",gDl_head->dlh_num_entries);
           ferr("     Number of opens:       %ld\n",gDstats[DIS_OPENS]);
           ferr("     Number of closes:      %ld\n",gDstats[DIS_CLOSES]);
           ferr("     Number of drops:       %ld\n",gDstats[DIS_DROPS]);
           ferr("     Number of hits on 1st: %ld\n",gDstats[DIS_HITS_ON_FIRST]);
           ferr("     Number of shifts:      %ld\n",gDstats[DIS_SHIFTS]);
           ferr("     Number of list hits:   %ld\n",gDstats[DIS_HITS]);
           ferr("     Number of list misses: %ld\n",gDstats[DIS_MISSES]);
           ferr("     Number of open fails:  %ld\n",gDstats[DIS_OPEN_FAILS]);
           ferr("     Number of dass calls:  %ld\n",gDstats[DIS_CALLS]);
           ferr("     Number of dass fails:  %ld\n",gDstats[DIS_CALL_FAILS]);
           ferr("\n");
        }
    }
    else
        if (gConf.flags.dis_debug) ferr("dis: No disassembly ever happened.\n");
} /* close_incache() */

/*
 * Acquire all the PC hits for current symbol, return -1
 * when got all them.  This spills them out when done.
 */
int
stuff_incache(unsigned long  PC,     /* Real address of the hit      */
              unsigned int   Hits    /* Number hits at that address  */
             )
{
    /* Collect all the hits to the current Symbol */
    gInPinstr_asm[gInCIcptr].ia_PC = PC;
    gInPinstr_asm[gInCIcptr].ia_Hits = Hits;

    /* Bump to next available element for next time */
    gInCIcptr++;

    /* Done?  */
    if (gInCIcptr >= gInNumInstr) 
    {
        flush_cache();
        return(-1);
    }
    return(0);
} /* stuff_incache() */

/* --------------------------- statics follow -------------------------- */

/*
 *  Now decide what disassembly is meaningful to output
 */
static void
flush_cache(void)
{
    unsigned int   kk;
    unsigned long  PC;     /* Real address of the hit */
    unsigned long  nextPC; /* Next address of the hits */
    unsigned int   hits;   /* Number hits at that first address  */


    /* Print marker if we don't start at start of symbol */
    if ((gInPoffValue + gInPregVaddr) != gInPinstr_asm[0].ia_PC)
        pscr( "                         ...\n");

    /* Now print all instruction hit plus some gaps */
    for (kk = 0; kk < gInNumInstr; kk++) 
    {
        unsigned long    diffPC;

        PC = gInPinstr_asm[kk].ia_PC;
        hits = gInPinstr_asm[kk].ia_Hits;
        nextPC = (kk<gInNumInstr-1) ? gInPinstr_asm[kk+1].ia_PC : PC;

        diffPC = nextPC - PC;

        /* print out instructions between the hitters */
#ifdef __ia64__
        if (diffPC > 0L && 
                ((gConf.flags.disassemble > 1 && (diffPC <= 128L)) 
                 || diffPC <= 64L) )
#else
        if (diffPC > 0L && diffPC < 21L)
#endif
        {
            /* Note that gdb will disassemble exclusive of the second
             * address.  That means that pc, pc+1 will only print the 
             * first instruction even if the next is only one byte long.
             * These PCs always have hits but nextPC will not get printed
             * now but next time around.
             */
            pinstr(PC, nextPC, hits);
           
        }
        else if (diffPC) {
            /* this is more space between than we want to print */
#ifdef __ia64__
            /* print triplets for ia64, print the issue group (2 instr) */
            pinstr(PC, PC+3, hits);
            pinstr(PC+16, PC+19, hits);
#else
            pinstr(PC, PC+1, hits);
#endif
            pscr( "                         ...\n");
        }
        else {
            /* otherwise, it's the same instruction: start special case */
#ifdef __ia64__
            pinstr(PC, PC+3, hits);
#else
            pinstr(PC, PC+1, hits);
#endif
        }
    }
    pscr("\n");

    return;
} /* flush_cache() */

/*
 * Print the disassembled Instruction and hit count.
 */
static void
pinstr(unsigned long   PC1,  /* Absolute beginning PC value */
       unsigned long   PC2,  /* Absolute (exclusive) ending PC value */
       unsigned int    Hits  /* Number of clock hits at that PC1 location */
      )
{
    char **str_array, **ptr, *eip1, *eip2;
    int ii;
    
    if (Hits)
    {
        /* If there were hits, print the number and times */
        pscr("            %4d  %5.2f  ",
               Hits,
               (double)(Hits)/gOp.samplehz
              ); 
    }
    else
    {
        /* No hits, skip times and number */
        pscr("                         ");
    }

    /* disassemble instruction, use offset for shlibs  */
    if (gInPsysms_key->sk_isReloc) {
        eip1 = (char*) (PC1 - gInPregVaddr);
        eip2 = (char*) (PC2 - gInPregVaddr);
    }
    else {
        eip1 = (char*)PC1;
        eip2 = (char*)PC2;
    }
    str_array = dass(gDasHandle, eip1, eip2);
    gDstats[DIS_CALLS]++;
    if (str_array && *str_array) {
        ptr = str_array;
        ii = 0;
        while (ptr && *ptr) {
            /* Note that if we're printing multiple lines, then by our
             * current definition, there's no hits to them.
             */
            if (ii) pscr( "                         ");
            pscr("%s\n", *ptr);
            ptr++; 
            ii++;
        }
        dass_free(str_array);
    }
    else {
        gDstats[DIS_CALL_FAILS]++;
        if (gConf.flags.dis_debug) {
            ferr("dis: dass call fail dass(0x%lx, 0x%lx, 0x%lx)\n",
                 gDasHandle, eip1, eip2);
        }
        pscr("0x%08x    offset: +0x%.4x\n",
               (void*)PC1,
               PC1 - gInPregVaddr - gOldSymsValue
              );
    }

    return;
} /* pinstr() */

/*
 * void *das_get(char *name)
 *
 * Get the handle to the named file from list.  If not open, 
 * then open.   Move to head of list.  If something falls of end
 * of list, close it.
 */
static void*
das_get(char *name)
{
    Dl_node_t *node=NULL;
    void *handle=NULL;
    int ii=0;

    if (!name) return NULL;

    /* check for first ever open */
    if (!gDl_head->dlh_num_entries ) {
        /* open pipe to gdb */
        handle = dass_open(name);
        if (handle == NULL) {
            /* open failed, return without update */
            gDstats[DIS_OPEN_FAILS]++;
            if (gConf.flags.dis_debug)
                ferr("dis: open fails of: %s\n", name);
            return NULL;
        }
        /* create new node */
        node = (Dl_node_t*)CALLOC(1,sizeof(Dl_node_t));
        if (!node) {
            ferr("Disaster: Calloc fails for new dis node!\n");
            prospect_exit(1);
        }
        node->dln_aout_name = strdup(name);
        node->dln_handle = handle;

        /* link in new node */
        gDl_head->dlh_head = node;
        gDl_head->dlh_tail = node;
        node->dln_next = NULL;
        node->dln_prev = NULL;

        /* update stats and pass back handle */
        gDl_head->dlh_num_entries++;
        gDstats[DIS_OPENS]++;
        return node->dln_handle;
    }

    /* otherwise, check if it's already open */
    node = gDl_head->dlh_head;
    ii=0;
    while(ii<gDl_head->dlh_num_entries) {
        if (strcmp(name, node->dln_aout_name)==0) {
            /* found it, move to head and return handle */
            if (node->dln_prev == NULL) {
                /* already at head */
                gDstats[DIS_HITS_ON_FIRST]++;
            }
            else {
                Dl_node_t *tmp;
                /* not the first already, so we move to head */
                gDstats[DIS_SHIFTS]++;
                /* first pluck out node */
                tmp = node->dln_prev;
                tmp->dln_next = node->dln_next;
                if (node->dln_next) {
                    /* if not the tail, link next prev to prev */
                    node->dln_next->dln_prev = tmp;
                }
                else {
                    /* this is the tail being moved so reset tail ptr */
                    gDl_head->dlh_tail = tmp;
                }
                /* node removed, now insert at head */
                tmp = gDl_head->dlh_head;
                gDl_head->dlh_head = node;
                node->dln_prev = NULL;
                node->dln_next = tmp;
                tmp->dln_prev = node;
            }
            gDstats[DIS_HITS]++;
            return node->dln_handle;
        }
        node = node->dln_next;
        ii++;
        /* check for end of list = not found */
        if (!node) break;
    }

    /* If we're here, that means it wasn't open so open and 
     * insert at head.
     */
    gDstats[DIS_MISSES]++;
    gDstats[DIS_OPENS]++;
    /* open pipe to gdb */
    handle = dass_open(name);
    if (handle == NULL) {
        /* open failed, return without update */
        gDstats[DIS_OPEN_FAILS]++;
        if (gConf.flags.dis_debug)
            ferr("dis: open fails of: %s\n", name);
        return NULL;
    }
    node = (Dl_node_t*) CALLOC(1, sizeof(Dl_node_t));
    if (!node) {
        ferr("Disaster: Calloc fails for new dis node!\n");
        prospect_exit(1);
    }
    node->dln_aout_name = strdup(name);
    node->dln_handle = handle;
    /* insert at head of list */
    node->dln_prev = NULL;
    node->dln_next = gDl_head->dlh_head;
    node->dln_next->dln_prev = node;
    gDl_head->dlh_head = node;
    gDl_head->dlh_num_entries++;
      
    /* the tail drops of the end of the world */
    if (gDl_head->dlh_num_entries > gConf.dis_list_size) {
        Dl_node_t *tmp;
        tmp = gDl_head->dlh_tail;
        tmp->dln_prev->dln_next = NULL;
        gDl_head->dlh_tail = tmp->dln_prev;
        dass_close(tmp->dln_handle);
        FREE(tmp->dln_aout_name);
        FREE(tmp);
        gDstats[DIS_DROPS]++;
        gDl_head->dlh_num_entries--;
    }

    return node->dln_handle;
} /* das_get() */

