/****************************************************************************
** jabio.cpp - handles JabTasks
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"jabio.h"
#include"common.h"


/****************************************************************************
  JabTask
****************************************************************************/
JabTask::JabTask(JabTask *_parent)
:QObject(_parent)
{
	v_parent = _parent;
	v_io = parent()->io();
	v_isRoot = FALSE;
	v_isDone = FALSE;

	v_id = io()->getId();
}

JabTask::JabTask(JabIO *_parent)
:QObject(_parent)
{
	v_parent = 0;
	v_io = _parent;
	v_isRoot = TRUE;
	v_isDone = FALSE;
}

JabTask *JabTask::parent()
{
	return v_parent;
}

QDomDocument *JabTask::doc()
{
	return &io()->doc;
}

JabTask *JabTask::find(const char *s)
{
	const QObjectList *p = children();
	if(!p)
		return 0;

	QObjectListIt it(*p);
	JabTask *j;
	for(; it.current(); ++it) {
		j = static_cast<JabTask*>(it.current());
		if(j->isA(s))
			return j;
	}

	return 0;
}

JabTask *JabTask::findById(const QString &s)
{
	const QObjectList *p = children();
	if(!p)
		return 0;

	QObjectListIt it(*p);
	JabTask *j;
	for(; it.current(); ++it) {
		j = static_cast<JabTask*>(it.current());
		if(j->id() == s)
			return j;
	}

	return 0;
}

bool JabTask::stamp2TS(const QString &ts, QDateTime *d)
{
	if(ts.length() != 17)
		return FALSE;

	int year  = ts.mid(0,4).toInt();
	int month = ts.mid(4,2).toInt();
	int day   = ts.mid(6,2).toInt();

	int hour  = ts.mid(9,2).toInt();
	int min   = ts.mid(12,2).toInt();
	int sec   = ts.mid(15,2).toInt();

	QDate xd;
	xd.setYMD(year, month, day);
	if(!xd.isValid())
		return FALSE;

	QTime xt;
	xt.setHMS(hour, min, sec);
	if(!xt.isValid())
		return FALSE;

	d->setDate(xd);
	d->setTime(xt);

	return TRUE;
}

QString JabTask::TS2stamp(const QDateTime &d)
{
	QString str;

	str.sprintf("%04d%02d%02dT%02d:%02d:%02d",
		d.date().year(),
		d.date().month(),
		d.date().day(),
		d.time().hour(),
		d.time().minute(),
		d.time().second());

	return str;
}

QDomElement JabTask::textTag(const QString &name, const QString &content)
{
	QDomElement tag = doc()->createElement(name);
	QDomText text = doc()->createTextNode(content);
	tag.appendChild(text);

	return tag;
}

QString JabTask::tagContent(const QDomElement &e)
{
	// look for some tag content
	for(QDomNode n = e.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomText i = n.toText();
		if(i.isNull())
			continue;
		return i.data();
	}

	return "";
}

QDomElement JabTask::findSubTag(const QDomElement &e, const QString &name, bool *found)
{
	if(found)
		*found = FALSE;

	for(QDomNode n = e.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomElement i = n.toElement();
		if(i.isNull())
			continue;
		if(i.tagName() == name) {
			if(found)
				*found = TRUE;
			return i;
		}
	}

	QDomElement tmp;
	return tmp;
}

QDomElement JabTask::createIQ(const QString &type, const QString &to, const QString &id)
{
	QDomElement iq = doc()->createElement("iq");
	if(!type.isEmpty())
		iq.setAttribute("type", type);
	if(!to.isEmpty())
		iq.setAttribute("to", to);
	if(!id.isEmpty())
		iq.setAttribute("id", id);

	return iq;
}

QDomElement JabTask::queryTag(const QDomElement &e)
{
	bool found;
	QDomElement q = findSubTag(e, "query", &found);
	return q;
}

QString JabTask::queryNS(const QDomElement &e)
{
	bool found;
	QDomElement q = findSubTag(e, "query", &found);
	if(found)
		return q.attribute("xmlns");

	return "";
}

JabRoster JabTask::xmlReadRoster(const QDomElement &q, bool push)
{
	JabRoster r;

	for(QDomNode n = q.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomElement i = n.toElement();
		if(i.isNull())
			continue;

		if(i.tagName() == "item") {
			JabRosterEntry entry;
			bool found;

			entry.jid = i.attribute("jid");
			entry.sub = i.attribute("subscription");
			entry.nick = i.attribute("name");

			// ask?
			found = i.hasAttribute("ask");
			if(found && i.attribute("ask") == "subscribe")
				entry.ask = 1;
			else
				entry.ask = 0;

			if(push)
				entry.push = TRUE;

			// grab the groups
			for(QDomNode n = i.firstChild(); !n.isNull(); n = n.nextSibling()) {
				QDomElement tag = n.toElement();
				if(tag.isNull())
					continue;

				if(tag.tagName() == "group")
					entry.groups.append(tagContent(tag));
			}

			JabRosterEntry *tmp = new JabRosterEntry(entry);
			r.add(tmp);
		}
	}

	return r;
}

QString JabTask::getErrorString(const QDomElement &e)
{
	bool found;
	QDomElement tag = findSubTag(e, "error", &found);
	if(!found)
		return "";

	return tagContent(tag);
}

void JabTask::done(JabTask *j)
{
	io()->doneJT(j);
	childFinished(j);
}

QString JabTask::toString()
{
	return "";
}

bool JabTask::fromString(const QString &)
{
	return FALSE;
}

void JabTask::setSuccess(bool b)
{
	v_success = b;
	v_isDone = TRUE;
	finished(this);
	JabTask *p = parent();
	if(p)
		p->done(this);
}

void JabTask::setError(const QString &s)
{
	v_errorString = s;
}

void JabTask::send(const QDomElement &x)
{
	io()->send(x);
}

bool JabTask::take(const QDomElement &x)
{
	const QObjectList *p = children();
	if(!p)
		return FALSE;

	// pass along the xml
	QObjectListIt it(*p);
	JabTask *j;
	for(; it.current(); ++it) {
		j = static_cast<JabTask*>(it.current());
		if(j->take(x))
			return TRUE;
	}

	return FALSE;
}

void JabTask::childEvent(QChildEvent *e)
{
	if(e->inserted())
		childAdded((JabTask *)e->child());
	else if(e->removed())
		childRemoved((JabTask *)e->child());
}


/****************************************************************************
  JabIO
****************************************************************************/
JabIO::JabIO()
{
	id_seed = 0xabcd;
	v_root = new JabTask(this);
}

JabIO::~JabIO()
{
}

QString JabIO::getId()
{
	QString s;
	s.sprintf("a%x", id_seed);
	id_seed += 0x10;
	return s;
}

JabTask *JabIO::root()
{
	return v_root;
}

void JabIO::reset()
{
	clear();
}

void JabIO::clear()
{
	const QObjectList *p = root()->children();
	if(!p)
		return;
	QObjectListIt it(*p);
	for(; it.current(); ++it) {
		JabTask *j = (JabTask *)it.current();
		if(j->isClearable())
			j->deleteLater();
	}
}

void JabIO::doneJT(JabTask *p)
{
	if(p->parent()->isRoot())
		done(p);
	else
		anyDone(p);
}

void JabIO::distribute(const QDomElement &x)
{
	if(!root()->take(x)) {
		//printf("JabIO: packet was ignored.\n");
	}
}

void JabIO::send(const QDomElement &x)
{
	QString out;
	QTextStream ts(&out, IO_WriteOnly);
	x.save(ts, 0);

	pdb(DEBUG_JABBER, QString("JabIO: outgoing: [\n%1]\n").arg(out));
	outgoingPacket(x);
}

void JabIO::incomingPacket(const QDomElement &x)
{
	QString out;
	QTextStream ts(&out, IO_WriteOnly);
	x.save(ts, 0);

	pdb(DEBUG_JABXML, QString("JabIO: incoming: [\n%1]\n").arg(out));
	distribute(x);
}

JabTask *JabIO::find(const char *s)
{
	return root()->find(s);
}

JabTask *JabIO::findById(const QString &s)
{
	return root()->findById(s);
}
