/****************************************************************************
** searchdlg.cpp - a dialog for searching for Jabber contacts
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"searchdlg.h"
#include"common.h"
#include<qlayout.h>
#include<qpushbutton.h>
#include<qlistview.h>
#include<qgroupbox.h>


SearchDlg::SearchDlg(const QString &jid, const JabForm &_form, int _localStatus, QWidget *parent, const char *name)
:SearchUI(parent, name, WDestructiveClose), UniqueWindow(TRUE, "SearchDlg", this, cleanJid(jid))
{
	v_jid = jid;
	form = _form;
	localStatus = _localStatus;

	setCaption(QString(tr("Search: %1")).arg(jid));
	//setIcon(transport2icon("", STATUS_ONLINE));

	isBusy = FALSE;

	QWidget *w;
	QLabel *lb;

	w = new QWidget(this);
	busy = new BusyWidget(w);
	QHBoxLayout *hb1 = new QHBoxLayout(w);
	hb1->addStretch(1);
	hb1->addWidget(busy);

	int i = SearchUILayout->findWidget(line);
	SearchUILayout->insertWidget(i, w);

	// instructions
	lb = new QLabel(QString("<qt>%1</qt>").arg(form.instructions), gb_search);
	lb->setFrameStyle( QFrame::Panel | QFrame::Sunken );
	gb_searchLayout->insertWidget(0, lb);

	// form
	w = new QWidget(gb_search);
	QGridLayout *gr = new QGridLayout(w, 2, 1, 4, 8);
	int at = 0;
	lb_field.setAutoDelete(TRUE);
	le_field.setAutoDelete(TRUE);
	QPtrListIterator<JabFormField> it(form);
	JabFormField *f;
	for(; (f = it.current()); ++it) {
		QLabel *lb = new QLabel(f->fieldName(), w);
		QLineEdit *le = new QLineEdit(w);
		le->setMinimumWidth(120);
		if(f->type() == JabFormField::password)
			le->setEchoMode(QLineEdit::Password);
		le->setText(f->value());

		lb_field.append(lb);
		le_field.append(le);

		gr->addWidget(lb, at, 0);
		gr->addWidget(le, at, 1);
		++at;

		connect(le, SIGNAL(returnPressed()), this, SLOT(doSubmitForm()));
	}
	gb_searchLayout->insertWidget(1, w);


	pb_add->setEnabled(FALSE);
	pb_info->setEnabled(FALSE);
	pb_stop->setEnabled(FALSE);

	if(localStatus == STATUS_OFFLINE)
		pb_search->setEnabled(FALSE);

	connect(pb_close, SIGNAL(clicked()), SLOT(close()));
	connect(pb_search, SIGNAL(clicked()), SLOT(doSubmitForm()));
	connect(pb_stop, SIGNAL(clicked()), SLOT(doStop()));
	connect(pb_add, SIGNAL(clicked()), SLOT(doAdd()));
	connect(pb_info, SIGNAL(clicked()), SLOT(doInfo()));
	connect(lv_results, SIGNAL(selectionChanged()), SLOT(selectionChanged()));
}

SearchDlg::~SearchDlg()
{
}

/* static */ SearchDlg *SearchDlg::find(const QString &xjid)
{
	return (SearchDlg *)UniqueWindowBank::find("SearchDlg", cleanJid(xjid));
}

void SearchDlg::closeEvent(QCloseEvent *e)
{
	// cancel active transaction
	if(isBusy) {
		signalCancelTransaction(id);
	}

	e->accept();
}

void SearchDlg::localUpdate(const JabRosterEntry &e)
{
	int oldstate = localStatus;
	localStatus = e.localStatus();

	if(localStatus == STATUS_OFFLINE) {
		if(isBusy) {
			busy->stop();
			isBusy = FALSE;
		}

		pb_search->setEnabled(FALSE);
		pb_stop->setEnabled(FALSE);
		clear();
	}
	else {
		// coming online?
		if(oldstate == STATUS_OFFLINE) {
			pb_search->setEnabled(TRUE);
		}
	}
}

void SearchDlg::addEntry(const QString &jid, const QString &nick, const QString &first, const QString &last, const QString &email)
{
	QListViewItem *lvi = new QListViewItem(lv_results);
	lvi->setText(0, nick);
	//lvi->setPixmap(0, *pix_online);
	lvi->setText(1, first);
	lvi->setText(2, last);
	lvi->setText(3, email);
	lvi->setText(4, jid);
}

void SearchDlg::clear()
{
	lv_results->clear();

	pb_add->setEnabled(FALSE);
	pb_info->setEnabled(FALSE);
}

void SearchDlg::doSubmitForm()
{
	if(isBusy || !pb_search->isEnabled())
		return;

	clear();

	JabForm submitform = form;

	// import the changes back into the form.
	// the QPtrList of QLineEdits should be in the same order
	QPtrListIterator<JabFormField> it(submitform);
	QPtrListIterator<QLineEdit> lit(le_field);
	JabFormField *f;
	for(; (f = it.current());) {
		QLineEdit *le = lit.current();
		f->setValue(le->text());
		++it;
		++lit;
	}

	pb_search->setEnabled(FALSE);
	pb_stop->setEnabled(TRUE);

	isBusy = TRUE;
	busy->start();

	signalSubmitForm(submitform, &id);
}

void SearchDlg::doStop()
{
	if(!isBusy || !pb_stop->isEnabled())
		return;

	signalCancelTransaction(id);
	isBusy = FALSE;
	busy->stop();

	pb_search->setEnabled(TRUE);
	pb_stop->setEnabled(FALSE);
}

void SearchDlg::putSearchFormResponse(JabRoster *p)
{
	if(isBusy) {
		isBusy = FALSE;
		busy->stop();
	}
	pb_search->setEnabled(TRUE);
	pb_stop->setEnabled(FALSE);

	// success
	if(p) {
		// fill out the results
		for(JabRosterEntry *entry = p->first(); entry; entry = p->next())
			addEntry(entry->jid, entry->nick, entry->first, entry->last, entry->email);
	}
	else {
		QMessageBox::information(this, tr("Error"), tr("There was an error retrieving search results."));
	}
}

void SearchDlg::selectionChanged()
{
	QListViewItem *i = lv_results->selectedItem();
	if(!i) {
		pb_add->setEnabled(FALSE);
		pb_info->setEnabled(FALSE);
	}

	pb_add->setEnabled(TRUE);
	pb_info->setEnabled(TRUE);
}

void SearchDlg::doAdd()
{
	QListViewItem *i = lv_results->selectedItem();
	if(!i)
		return;

	aAdd(i->text(4), i->text(0), "");

	QMessageBox::information(this, tr("Add User"), QString(tr("%1 has been added to your contact list.")).arg(i->text(0)));
}

void SearchDlg::doInfo()
{
	QListViewItem *i = lv_results->selectedItem();
	if(!i)
		return;

	aInfo(i->text(4));
}

