#include "opt_appearance.h"
#include "common.h"
#include "iconwidget.h"

#include <qwhatsthis.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qcolordialog.h>
#include <qfontdialog.h>
#include <qlineedit.h>

#include "opt_appearance-ui.h"

//----------------------------------------------------------------------------
// FontLabel
//----------------------------------------------------------------------------

FontLabel::FontLabel(QWidget *parent, const char *name)
	: QLineEdit(parent, name)
{
	setReadOnly(true);
	setPaletteBackgroundColor(parent->paletteBackgroundColor());

	m_defaultHeight = QLineEdit::sizeHint().height();
}

void FontLabel::setFont(QString fontName)
{
	QFont f;
	f.fromString(fontName);
	m_font = fontName;
	setText( tr("%1 %2").arg( f.family() ).arg( f.pointSize() ) );
	QLineEdit::setFont(f);
}

QString FontLabel::fontName() const
{
	return m_font;
}

QSize FontLabel::sizeHint() const
{
	return QSize(QLineEdit::sizeHint().width(), m_defaultHeight);
}

//----------------------------------------------------------------------------
// OptionsTabAppearance
//----------------------------------------------------------------------------

OptionsTabAppearance::OptionsTabAppearance(QObject *parent)
: OptionsTab(parent, "appearance", "", tr("Appearance"), tr("Psi's Appearance"), "psi/ok")
{
	w = 0;
	bg_font = 0;
	bg_color = 0;
	o = new Options;
}

OptionsTabAppearance::~OptionsTabAppearance()
{
	if ( bg_font )
		delete bg_font;
	if ( bg_color )
		delete bg_color;
	delete o;
}

static QPixmap name2color(QString name) // taken from opt_general.cpp
{
	QColor c(name);
	QPixmap pix(16, 16);
	QPainter p(&pix);

	p.fillRect(0, 0, pix.width(), pix.height(), QBrush(c));
	p.setPen( QColor(0, 0, 0) );
	p.drawRect(0, 0, pix.width(), pix.height());
	p.end();

	return pix;
}

QWidget *OptionsTabAppearance::widget()
{
	if ( w )
		return 0;

	w = new OptAppearanceUI();
	OptAppearanceUI *d = (OptAppearanceUI *)w;

	le_font[0] = d->le_fRoster;
	le_font[1] = d->le_fMessage;
	le_font[2] = d->le_fChat;
	le_font[3] = d->le_fPopup;

	bg_font = new QButtonGroup;
	bg_font->insert(d->pb_fRoster, 0);
	bg_font->insert(d->pb_fMessage, 1);
	bg_font->insert(d->pb_fChat, 2);
	bg_font->insert(d->pb_fPopup, 3);
	connect(bg_font, SIGNAL(clicked(int)), SLOT(chooseFont(int)));

	QWhatsThis::add(le_font[0],
		tr("Specifies the font style for the main window."));
	QWhatsThis::add(le_font[1],
		tr("Specifies the font style for message windows."));
	QWhatsThis::add(le_font[2],
		tr("Specifies the font style for chat windows."));
	QWhatsThis::add(le_font[3],
		tr("Specifies the font style for popup windows."));
	QWhatsThis::add(d->pb_fRoster,
		tr("Selects a font for the a roster window using the font selection dialog."));
	QWhatsThis::add(d->pb_fMessage,
		tr("Selects a font for message windows using the font selection dialog."));
	QWhatsThis::add(d->pb_fChat,
		tr("Selects a font for chat windows using the font selection dialog."));

	bg_color = new QButtonGroup;
	bg_color->insert(d->pb_cOnline, 0);
	bg_color->insert(d->pb_cOffline, 1);
	bg_color->insert(d->pb_cAway, 2);
	bg_color->insert(d->pb_cDND, 3);
	bg_color->insert(d->pb_cGroupFore, 4);
	bg_color->insert(d->pb_cGroupBack, 5);
	bg_color->insert(d->pb_cListBack, 6);
	bg_color->insert(d->pb_cAnimFront, 7);
	bg_color->insert(d->pb_cAnimBack, 8);
	connect(bg_color, SIGNAL(clicked(int)), SLOT(chooseColor(int)));

	QString s = tr("Specifies the text color for a contact name in the main window when that user is \"%1\".");
	QWhatsThis::add(d->pb_cOnline,
		s.arg(tr("online")));
	QWhatsThis::add(d->pb_cOffline,
		s.arg(tr("offline")));
	QWhatsThis::add(d->pb_cAway,
		s.arg(tr("away")));
	QWhatsThis::add(d->pb_cDND,
		s.arg(tr("do not disturb")));
	QWhatsThis::add(d->pb_cGroupBack,
		tr("Specifies the background color for a group name in the main window."));
	QWhatsThis::add(d->pb_cListBack,
		tr("Specifies the background color for the main window."));
	QWhatsThis::add(d->pb_cAnimFront,
		tr("Specifies the foreground animation color for nicks."));
	QWhatsThis::add(d->pb_cAnimBack,
		tr("Specifies the background animation color for nicks."));

	// Avatars
	//QWhatsThis::add(d->ck_avatarsChatdlg,
	//	tr("Toggles displaying of avatars in the chat dialog"));

	return w;
}

void OptionsTabAppearance::applyOptions(Options *opt)
{
	if ( !w )
		return;

	//OptAppearanceUI *d = (OptAppearanceUI *)w;
	//opt->avatarsChatdlgEnabled = d->ck_avatarsChatdlg->isChecked(); // Avatars

	int n;
	for (n = 0; n < 4; ++n)
		opt->font[n] = le_font[n]->fontName();

	for (n = 0; n < 9; ++n)
		opt->color[n] = o->color[n];
}

void OptionsTabAppearance::restoreOptions(const Options *opt)
{
	if ( !w )
		return;

	//OptAppearanceUI *d = (OptAppearanceUI *)w;
	//d->ck_avatarsChatdlg->setChecked( opt->avatarsChatdlgEnabled ); // Avatars

	int n;
	for (n = 0; n < 4; ++n)
		le_font[n]->setFont(opt->font[n]);

	for (n = 0; n < 9; ++n) {
		o->color[n] = opt->color[n];
		((QPushButton *)bg_color->find(n))->setPixmap(name2color(opt->color[n].name()));
	}
}

void OptionsTabAppearance::setData(PsiCon *, QWidget *parentDialog)
{
	parentWidget = parentDialog;
}

void OptionsTabAppearance::chooseFont(int x)
{
	bool ok;
	QFont font;
	font.fromString( le_font[x]->fontName() );

	QString fnt = QFontDialog::getFont(&ok, font, parentWidget).toString();
	le_font[x]->setFont(fnt);

	if(ok)
		emit dataChanged();
}

void OptionsTabAppearance::chooseColor(int x)
{
	QColor c = o->color[x];

	c = QColorDialog::getColor(c, parentWidget);
	if(c.isValid()) {
		o->color[x] = c;
		((QPushButton *)bg_color->find(x))->setPixmap(name2color(o->color[x].name()));

		emit dataChanged();
	}
}
