package PSP::Parser::PreScan;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: PreScan.pm,v 1.1 2000/11/23 23:36:20 muaddib Exp $

use strict;

use PSP::Parser;
@PSP::Parser::PreScan::ISA = qw(PSP::Parser);

use Error qw(:try);

use vars qw($STATIC_GROUP);
@PSP::Parser::PreScan::EXPORT = qw($STATIC_GROUP);
BEGIN { $STATIC_GROUP = '_main'; }

use vars qw(@handled @handled_no_end @propagatable);
BEGIN {
  @handled  = qw(dynamic_display);
  @handled_no_end = qw(input display alias hidden);
  @propagatable  = qw(data form_count current_fsuse stack_gname);
};

sub register {
  my ($this,$pkg) = @_;

  $this->SUPER::register($pkg);

  $this->{form_count} = 0;
  $this->{stack_fsname} = "";
  $this->{stack_gname} = [];
}

=head1 PRIVATE TAG HANDLING METHODS

=head2 begin_form

 [private] instance
 () begin_form (str $tag, \%attr)

DESCRIPTION:

Counts the forms encountered and remembers the current fieldspace for
later use.

=cut

sub begin_form {
  my ($this, $tag, $attr) = @_;

  $this->{form_count}++;

  if (my $fsname = $attr->{fieldspace}) {
    $fsname and $this->{current_fsuse} = $fsname;
  }
}

=head2 end_form

 [private] instance
 () end_form ()

DESCRIPTION:

Clears the current fieldspace.

=cut

sub end_form {
  my ($this) = @_;
  delete $this->{current_fsuse};
}

=head2 begin_pspinput

 [private] instance
 () begin_pspinput (str $tag, \%attr)

DESCRIPTION:

Notes if the field is changed flagged. This and other HTML-control
related tag handlers also notes that the static group should be
requested unless we are in a DYNAMICDISPLAY, which will be used
later in B<PSP::Piler> when the start form tag is encountered.

=cut

sub begin_pspinput {
  my ($this, $tag, $attr) = @_;

  my $name = $attr->{name};
  if (my $changeflag = $attr->{changeflag}) {
    $this->{preparsed}->{change_flag_data}->{$name} = $changeflag;
  }

  @{$this->{stack_gname}} or
    $this->add_requested_group($STATIC_GROUP);
}

=head2 begin_pspdisplay

 [private] instance
 () begin_pspdisplay ()

DESCRIPTION:

Notes that the static groups should be requested when the form is begun later
when the pile is constructed. Has no effect if we are in a DYNAMICDISPLAY.

=cut

sub begin_pspdisplay {
  my ($this) = @_;
  @{$this->{stack_gname}} or
    $this->add_requested_group($STATIC_GROUP);
}

=head2 begin_pspalias

 [private] instance
 () begin_pspalias ()

DESCRIPTION:

Notes that the static groups should be requested when the form is begun later
when the pile is constructed. Has no effect if we are in a DYNAMICDISPLAY.

=cut

sub begin_pspalias {
  my ($this) = @_;
  @{$this->{stack_gname}} or
    $this->add_requested_group($STATIC_GROUP);
}

=head2 begin_psphidden

 [private] instance
 () begin_psphidden ()

DESCRIPTION:

As C<begin_pspdisplay>.

=cut

sub begin_psphidden {
  my ($this) = @_;
  @{$this->{stack_gname}} or
    $this->add_requested_group($STATIC_GROUP);
}

=head2 begin_pspdynamicdisplay

 [private] instance
 () begin_pspdynamic_display

DESCRIPTION:

Notes the group to request and that a DYNAMICDISPLAY is being
requested. Both these facts will be used later in B<PSP::Piler> at a
point before the DYNAMICDISPLAY tag.

=cut

sub begin_pspdynamicdisplay {
  my ($this, $tag, $attr) = @_;
  my $gname = $attr->{name};
  my $numvar = $attr->{numvar};
  my $fsname = $this->{current_fsuse};

  push @{$this->{stack_gname}}, $gname;

  $this->{preparsed}->{dynamic_displays}->{$gname}->{fsname} = $fsname;
  $this->{preparsed}->{dynamic_displays}->{$gname}->{numvar} = $numvar;

  $this->add_requested_group($gname);
}

=head2 end_pspdynamicdisplay

 [private] instance
 () end_pspdynamicdisplay ()

DESCRIPTION:

Keeps internal state information correct so that we can identify what
state we are in, you know.

=cut

sub end_pspdynamicdisplay {
  my ($this) = @_;
  pop @{$this->{stack_gname}};
}


=head2 add_requested_group

 [private] instance
 () add_requested_group (str $group, str $fieldspace)

DESCRIPTION:

Notes that specified group is needed on the code being processed for
the form that we are currently in.

BUG: We should be using C<$fieldspace> here and and the
C<requested_groups()> function so that we don't get confused whon they
use multiple fieldspaces on the same page with the same group
names. The use of C<$form_count> in C<requested_groups()> may have the
side effect of protecting us, but we should verify and then remove the
fieldspace constraint.'

=cut 

sub add_requested_group {
  my ($this, $group, $fieldspace) = @_;

  # create a new lexical array, and transfer contents into it.
  my @groups = @{$this->{stack_gname}};

  my $form_count = $this->{form_count};
  $this->{preparsed}->{requested_groups}->[$form_count]->{$group} = \@groups;
}

=head2 requested_groups

 instance
 (\%data) requested_groups (int $form_count)

DESCRIPTION:

Will return the approprate data hash for the form within the code
processed indicated by C<$form_count> (which starts with 1 being the
first form).

C<%data> is keyed by the a group name, the value of which is a
reference to a list of groups which are ordered so that the last in
the array is the key and each succeeding, if any, is a successive
parent. It may also be an empty hash.

=cut

sub requested_groups {
  my ($this, $form_count) = @_;
  return ($this->{preparsed}->{requested_groups}->[$form_count] || {});
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
