"""Provides functions for logging error and other messages to one or more
files and/or the console, using python's own logging module. Some warning
messages and error messages are generated by PsychoPy itself. The user can
generate more using the functions in this module.

There are various levels for logged messages; ranging from ERROR, through
WARNING, to INFO and DEBUG. When setting the level for a particular log
(file or console) the  user can set the minimum level that is required 
for messages to enter the log. For example, setting a level of INFO will
result in INFO, WARNING and ERROR messages to be recorded but not DEBUG 
nessages.

By default, PsychoPy will record messages of WARNING level and above to
the console and INFO upwards to a psychopy.log in the current directory. The user
can silence both by setting them to receive only CRITICAL messages, which 
(PsychoPy doesn't use) using the commands; 
::

    from psychopy import log
    log.console.setLevel(log.CRITICAL)

"""
# Part of the PsychoPy library
# Copyright (C) 2010 Jonathan Peirce
# Distributed under the terms of the GNU General Public License (GPL).

from os import path
import logging
_packagePath = path.split(__file__)[0]

CRITICAL = logging.CRITICAL
ERROR = logging.ERROR#40
DATA = 35#will be a custom level to be included with every level above WARNING
WARNING = logging.WARNING#30
INFO = logging.INFO#20
DEBUG = logging.DEBUG#10

#create links to the logging functions
error = logging.error
"""log.error(message) Send the message to any receiver of logging info (e.g. a LogFile) of level `log.ERROR` or higher
"""
warning = logging.warning
"""log.warning(message) Send the message to any receiver of logging info (e.g. a LogFile) of level `log.WARNING` or higher
"""
info = logging.info
"""log.info(message) Send the message to any receiver of logging info (e.g. a LogFile) of level `log.INFO` or higher
"""
debug = logging.debug
"""log.debug(message) Send the message to any receiver of logging info (e.g. a LogFile) of level `log.DEBUG` or higher
"""
#NB we add our own function for data() below

console = logging.StreamHandler() #create a handler for the console
console.setFormatter(logging.Formatter('%(asctime)-s %(levelname)-8s %(message)s', '%y-%m-%d %H:%M'))
console.setLevel(WARNING)
#the default 'origin' of the log messages
_rootLogger = logging.getLogger('')
_rootLogger.addHandler(console)# add the console logger to receive all root logs
_rootLogger.setLevel(logging.DEBUG) #the minimum to be sent

#add DATA as a custom level
logging.addLevelName(DATA, 'DATA')
def data(msg ,*args, **kwargs):
    """log.data(message) Send the message to any receiver of logging info (e.g. a LogFile) of level `log.DATA` or higher
    """
    logging.root.log(DATA, msg, *args, **kwargs)
    
class LogFile:
    """Creates an object to help with logging events to a file"""
    def __init__(self, filename, 
                       filemode = 'a', level=INFO,
                       format = '%(asctime)-s %(levelname)-8s %(message)s', 
                       dateFormat='%y-%m-%d %H:%M'):
        """**Arguments:**
            - filename
            - filemode = 'a'(ppend) or 'w'(rite). The latter will remove the previous file
            - level = the minimum level of the messages to be entered into the log
            - format = a string defining the format of messages
            - datefmt = a string specifying just the date part of the message
        """
        self._log = logging.FileHandler(filename, filemode)
        #~ self._log = logging.basicConfig(level=level,
                    #~ format=format, datefmt=datefmt,
                    #~ filename=filename, filemode=filemode)
        self._log.setLevel(level)
        formatter = logging.Formatter(format, dateFormat)
        self._log.setFormatter(formatter)
        _rootLogger.addHandler(self._log)
    def setLevel(self, level):
        """Sets the current level for this log (numerically)
        The values correspond to:
        
            - 40:Error
            - 35 Data
            - 30:Warning
            - 20:Info
            - 10:Debug"""
        self._log.setLevel(level)
    def getLevel(self):
        """Returns the current level for this log (numerically)
        The values correspond to:
        
            - 40:Error
            - 35 Data
            - 30:Warning
            - 20:Info
            - 10:Debug
            """
        return self._log.level        
    def write(self, text):
        """Write arbitrary text to the logfile (and no other file).
        Consider using functions like `psychopy.log.info` instead, to write 
        the message to all logfiles of a given level.
        """
        self._log.stream.write(text)
    def writeline(self, text):
        """As `LogFile.write` but adds a \n at the end of the text
        """
        self._log.stream.write(text+'\n')
#psychopyLog = LogFile('psychopy.log', 'a', level=ERROR)
