#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import sys
import gtk
import locale
import consts
import msgBox
import gettext
import os.path
import dlgAbout
import dlgOutput
import dlgSavePar2
import preferences
import prefsManager
import treeviewCreate

from prefsManager import prefs
from gettext import gettext as _

# Pages in advanced settings
(
    PAGE_MEMORY,
    PAGE_BLOCKS,
    PAGE_REDUNDANCY,
    PAGE_RECOVERY_FILES,
    PAGE_FIRST_BLOCK,
) = range(5)

class Main :
    def __init__(self, argv) :
        self.wTree  = gtk.glade.XML(os.path.join(consts.dirRes, 'win-main.glade'), domain=consts.appNameShort)
        self.window = self.wTree.get_widget('win-main')

        for widget in self.wTree.get_widget_prefix('') :
            self.__dict__[gtk.glade.get_widget_name(widget)] = widget

        self.TreeViewCreate = treeviewCreate.TreeViewCreate(self.window, self.treeview) 

        self.wTree.signal_autoconnect(self)

        # Widgets that are enabled/disabled when a radio button is clicked
        # The first list in the tuple contains the widgets that must be enabled
        # The second one contains the widgets that must be disabled
        self.radioActions = {}
        self.radioActions[self.advBlocksSettingSize]           = ([self.advBlocksSize], [self.advBlocksCount])
        self.radioActions[self.advBlocksSettingCount]          = ([self.advBlocksCount], [self.advBlocksSize])
        self.radioActions[self.advBlocksSettingDynamic]        = ([], [self.advBlocksCount, self.advBlocksSize])
        self.radioActions[self.advRedundancySettingCount]      = ([self.advRedundancyCount], [self.advRedundancyLvl])
        self.radioActions[self.advRedundancySettingRedundancy] = ([self.advRedundancyLvl], [self.advRedundancyCount])
        self.radioActions[self.advRecoverySettingSize]         = ([], [self.advRecoveryNumber, self.advRecoveryUniform])
        self.radioActions[self.advRecoverySettingNumber]       = ([self.advRecoveryNumber, self.advRecoveryUniform], [])

        # All settings chosen thanks to radio buttons
        # The list is composed of 2-tuples: the first one is the name of the radio button, the second one is the associated value
        self.advRadioSettings = {}
        self.advRadioSettings['advBlocksSetting'] = [('advBlocksSettingDynamic', preferences.BLOCKS_DYNAMIC),
                                                     ('advBlocksSettingCount', preferences.BLOCKS_COUNT),
                                                     ('advBlocksSettingSize', preferences.BLOCKS_SIZE)
                                                    ]
        self.advRadioSettings['advRedundancySetting'] = [('advRedundancySettingRedundancy', preferences.REDUNDANCY_REDUNDANCY),
                                                         ('advRedundancySettingCount', preferences.REDUNDANCY_BLOCK_COUNT)
                                                        ]
        self.advRadioSettings['advRecoverySetting'] = [('advRecoverySettingSize', preferences.RECOVERY_LIMIT_SIZE),
                                                       ('advRecoverySettingNumber', preferences.RECOVERY_NUMBER_OF_FILES)
                                                      ]

        # Widgets used for advanced settings
        self.advSpinButtons = ['advMemoryUsage', 'advBlocksCount', 'advBlocksSize', 'advRedundancyLvl', 'advRedundancyCount', 'advRecoveryNumber', 'advBeginningFirst']
        self.advCheckButtons = ['advRecoveryUniform']

        # Prepare the file chooser
        self.fcOpenPar2.set_current_folder(consts.dirUsr)
        filter = gtk.FileFilter()
        filter.add_pattern('*.par2')
        filter.add_pattern('*.PAR2')
        self.fcOpenPar2.set_filter(filter)

        # Prepare widgets
        self.prefsToWidgets()

        # If a file was given on the command line, select it for checking
        if len(argv) > 1 :
            self.fcOpenPar2.set_filename(argv[1])
            self.notebook.set_current_page(preferences.PAGE_CHECK)

        # Prepare the window and show it
        self.window.set_title(consts.appName)
        self.window.show()
        gtk.main()


    def getAdvancedSettings(self, creationSettings) :
        if not self.chkAdv.get_active() :
            return []

        # "Memory" group, always used
        list = ['-m' + str(int(self.advMemoryUsage.get_value()))]
        if creationSettings :
            # "Blocks" group
            if self.advBlocksSettingCount.get_active() :
                list += ['-b' + str(int(self.advBlocksCount.get_value()))]
            elif self.advBlocksSettingSize.get_active() :
                list += ['-s' + str(int(self.advBlocksSize.get_value())*1024)]
            # "Redundancy" group :
            if self.advRedundancySettingRedundancy.get_active() :
                list += ['-r' + str(int(self.advRedundancyLvl.get_value()))]
            else :
                list += ['-c' + str(int(self.advRedundancyCount.get_value()))]
            # "Recovery files" group
            if self.advRecoverySettingSize.get_active() :
                list += ['-l']
            elif self.advRecoverySettingNumber.get_active() :
                list += ['-n' + str(int(self.advRecoveryNumber.get_value()))]
                if self.advRecoveryUniform.get_active() :
                    list += ['-u']
            # "Beginning" group
            list += ['-f' + str(int(self.advBeginningFirst.get_value()))]
        return list


    def showCreationPages(self, visible) :
        if visible :
            self.advNotebook.get_nth_page(PAGE_BLOCKS).show()
            self.advNotebook.get_nth_page(PAGE_REDUNDANCY).show()
            self.advNotebook.get_nth_page(PAGE_RECOVERY_FILES).show()
            self.advNotebook.get_nth_page(PAGE_FIRST_BLOCK).show()
        else :
            self.advNotebook.get_nth_page(PAGE_BLOCKS).hide()
            self.advNotebook.get_nth_page(PAGE_REDUNDANCY).hide()
            self.advNotebook.get_nth_page(PAGE_RECOVERY_FILES).hide()
            self.advNotebook.get_nth_page(PAGE_FIRST_BLOCK).hide()


    def checkPar2(self) :
        # Ensure that a file has been selected
        if self.fcOpenPar2.get_filename() is None :
            msgBox.error(self.window, _('You must first select a par2 file!'))
            return
        # Check action
        if self.radRepair.get_active() :
            action = 'r'
        elif self.radVerif.get_active() :
            action = 'v'
        else :
            msgBox.error(self.window, _('No action selected!'))
            return
        # Constructs the command line
        args = ['par2', action] + self.getAdvancedSettings(False) + ['--', self.fcOpenPar2.get_filename()]
        # Execute this command
        dlgOutput.instance(self.window).show('par2', args)


    def createPar2(self) :
        # Check that the user chosed some files to protect
        selectedFiles = self.TreeViewCreate.getAllFilenames()
        if len(selectedFiles) == 0 :
            msgBox.error(self.window, _('You must first add some files to protect!'))
            return
        # Ask him to provide a name for the par2 file
        # By default, use the first filename with a .par2 extension appended
        filename = dlgSavePar2.instance(self.window).show(selectedFiles[0] + '.par2')
        if filename is None :
            return
        # Constructs the command line
        args = ['par2', 'c'] + self.getAdvancedSettings(True) + ['--', filename] + selectedFiles
        # Execute this command
        dlgOutput.instance(self.window).show('par2', args)


    def advPrefsToWidgets(self) :
        # Set the value of the spin buttons
        for widget in self.advSpinButtons :
            self.__dict__[widget].set_value(prefs.__dict__[widget])
        # Set the value of the check buttons
        for widget in self.advCheckButtons :
            self.__dict__[widget].set_active(prefs.__dict__[widget])
        # Activate the correct radio button for each preferences
        for prefName in self.advRadioSettings.keys() :
            for (radio,value) in self.advRadioSettings[prefName] :
                if prefs.__dict__[prefName] == value :
                    self.__dict__[radio].set_active(True)
                    self.onRadioButton(self.__dict__[radio])


    def prefsToWidgets(self) :
        # Selected page in the notebook
        self.notebook.set_current_page(prefs.mainPage)
        # Default action when checking files
        if prefs.checkAction == preferences.CHECK_VERIFY :
            self.radVerif.set_active(True)
        elif prefs.checkAction == preferences.CHECK_REPAIR :
            self.radRepair.set_active(True)
        # Show advanced options ?
        self.chkAdv.set_active(prefs.showAdvSettings)
        # Advanced settings
        self.advPrefsToWidgets()


    def widgetsToPrefs(self) :
        # Selected page in the notebook
        prefs.mainPage = self.notebook.get_current_page()
        # Default action when checking files
        if self.radVerif.get_active() : 
            prefs.checkAction = preferences.CHECK_VERIFY
        elif self.radRepair.get_active() :
            prefs.checkAction = preferences.CHECK_REPAIR
        # Show advanced options ?
        prefs.showAdvSettings = self.chkAdv.get_active()
        # Retrieve values from the spin buttons
        for widget in self.advSpinButtons :
            prefs.__dict__[widget] = int(self.__dict__[widget].get_value())
        # Retrieve values from the check buttons
        for widget in self.advCheckButtons :
            prefs.__dict__[widget] = self.__dict__[widget].get_active()
        # Retrieve values from the radio buttons
        for prefName in self.advRadioSettings.keys() :
            for (radio, value) in self.advRadioSettings[prefName] :
                if self.__dict__[radio].get_active() :
                    prefs.__dict__[prefName] = value

#==========================================================
#
# Events
#
#==========================================================

    def onRadioButton(self, widget, data=None) :
        if widget.get_active() :
            toBeEnabled, toBeDisabled = self.radioActions[widget]
            for w in toBeEnabled :
                w.set_sensitive(True)
            for w in toBeDisabled :
                w.set_sensitive(False)


    def onQuit(self, widget) :
        self.widgetsToPrefs()
        prefsManager.save()
        gtk.main_quit()


    def onMnuAbout(self, widget) :
        dlgAbout.show(self.window)


    def onBtnDefault(self, widget) :
        prefs.resetAdvancedSettings()
        self.advPrefsToWidgets()


    def onChkAdvanced(self, widget) :
        if self.chkAdv.get_active() :
            self.boxAdv.show()
            self.showCreationPages(self.notebook.get_current_page() == preferences.PAGE_CREATE)
        else :
            self.boxAdv.hide()


    def onSwitchPage(self, notebook, page, pageNum) :
        if self.chkAdv.get_active() :
            self.showCreationPages(pageNum == preferences.PAGE_CREATE)


    def onBtnGo(self, widget) :
        if self.notebook.get_current_page() == preferences.PAGE_CHECK :
            self.checkPar2()
        else :
            self.createPar2()


    def onTreeviewMouseButton(self, tree, event) :
        self.TreeViewCreate.onMouseButton(tree, event)


    def onTreeviewKey(self, tree, event) :
        self.TreeViewCreate.onKey(tree, event)


if __name__ != '__main__' :
    print 'Error : Not a module'
    sys.exit()

# i18n
locale.setlocale(locale.LC_ALL, '')
gettext.textdomain(consts.appNameShort)
gettext.bindtextdomain(consts.appNameShort, consts.dirLocale)
gtk.glade.textdomain(consts.appNameShort)
gtk.glade.bindtextdomain(consts.appNameShort, consts.dirLocale)

Main(sys.argv)
