#include "Python.h"
#ifdef MS_WIN32
#include <windows.h>
#endif
#include <GL/gl.h>
#include <math.h>

#ifdef NUMERIC
static int typecode2gltype[] =
    { -1, GL_UNSIGNED_BYTE, GL_BYTE, GL_SHORT, GL_INT, -1, GL_FLOAT, -1,
    -1, -1, -1, -1
};
#endif

static PyObject *py_gl_Error;

static PyObject *ErrorReturn(char *message)
{
    PyErr_SetString(py_gl_Error, message);
    return NULL;
}

#define ASSERT(E,M) if(!(E)) return ErrorReturn(M)

#define TRY(E) if(! (E)) return NULL

#ifdef NUMERIC

#if   defined(HAVE_ARRAYOBJECT_H)
#include "arrayobject.h"
#elif defined(HAVE_EXTENSIONS_ARRAYOBJECT_H)
#include "Extensions/arrayobject.h"
#elif defined(HAVE_NUMERIC_ARRAYOBJECT_H)
#include "Numeric/arrayobject.h"
#elif defined(HAVE_NUMERICAL_ARRAYOBJECT_H)
#include "numerical/arrayobject.h"
#else
/*#error "Don't know where to find file 'arrayobject.h'" */
#include "arrayobject.h"
#endif

#define PyArray_ClearMemory(op, pitems) Py_DECREF((op))

static int PyArray_AsUByteArray(PyObject ** op, GLubyte ** pitems, int *pn)
{
    PyArrayObject *mp;
    mp =
	(PyArrayObject *) PyArray_ContiguousFromObject(*op, PyArray_UBYTE,
						       0, 1);
    if (!mp)
	return 0;
    *pitems = (GLubyte *) (mp->data);
    *pn = mp->dimensions[0];
    *op = (PyObject *) mp;
    return 1;
}

static int PyArray_AsDoubleArray(PyObject ** op, GLdouble ** pitems, int *pn)
{
    PyArrayObject *mp;
    mp =
	(PyArrayObject *) PyArray_ContiguousFromObject(*op,
						       PyArray_DOUBLE, 0, 1);
    if (!mp)
	return 0;
    *pitems = (GLdouble *) (mp->data);
    *pn = mp->dimensions[0];
    *op = (PyObject *) mp;
    return 1;
}

static int PyArray_AsFloatArray(PyObject ** op, GLfloat ** pitems, int *pn)
{
    PyArrayObject *mp;
    mp =
	(PyArrayObject *) PyArray_ContiguousFromObject(*op, PyArray_FLOAT,
						       0, 1);
    if (!mp)
	return 0;
    *pitems = (GLfloat *) (mp->data);
    *pn = mp->dimensions[0];
    *op = (PyObject *) mp;
    return 1;
}

static int PyArray_AsIntArray(PyObject ** op, GLint ** pitems, int *pn)
{
    PyArrayObject *mp;
    mp = (PyArrayObject *) PyArray_ContiguousFromObject(*op, PyArray_INT, 0, 1);
    if (!mp)
	return 0;
    *pitems = (GLint *) (mp->data);
    *pn = mp->dimensions[0];
    *op = (PyObject *) mp;
    return 1;
}

static int PyArray_AsShortArray(PyObject ** op, GLshort ** pitems, int *pn)
{
    PyArrayObject *mp;
    mp =
	(PyArrayObject *) PyArray_ContiguousFromObject(*op, PyArray_SHORT,
						       0, 1);
    if (!mp)
	return 0;
    *pitems = (GLshort *) (mp->data);
    *pn = mp->dimensions[0];
    *op = (PyObject *) mp;
    return 1;
}

#else

#include "abstract.h"

#define PyArray_ClearMemory(op, pitems) PyMem_DEL(pitems)

static int PyArray_AsDoubleArray(PyObject ** op, GLdouble ** pitems, int *pn)
{
    GLdouble *items;
    PyObject *item;
    int n, i;
    if (!PySequence_Check(*op))
	return 0;
    n = PySequence_Length(*op);
    items = PyMem_NEW(GLdouble, n);
    if (items == NULL) {
	PyErr_NoMemory();
	return 0;
    }
    for (i = 0; i < n; i++) {
	if ((item = PySequence_GetItem(*op, i))) {
	    items[i] = PyFloat_AsDouble(item);
	    Py_DECREF(item);
	}
	if (PyErr_Occurred())
	    return 0;
    }
    *pitems = items;
    *pn = n;
    return 1;
}

static int PyArray_AsFloatArray(PyObject ** op, GLfloat ** pitems, int *pn)
{
    GLfloat *items;
    PyObject *item;
    int n, i;
    if (!PySequence_Check(*op))
	return 0;
    n = PySequence_Length(*op);
    items = PyMem_NEW(GLfloat, n);
    if (items == NULL) {
	PyErr_NoMemory();
	return 0;
    }
    for (i = 0; i < n; i++) {
	if ((item = PySequence_GetItem(*op, i))) {
	    items[i] = PyFloat_AsDouble(item);
	    Py_DECREF(item);
	}
	if (PyErr_Occurred())
	    return 0;
    }
    *pitems = items;
    *pn = n;
    return 1;
}

static int PyArray_AsIntArray(PyObject ** op, GLint ** pitems, int *pn)
{
    GLint *items;
    PyObject *item;
    int n, i;
    if (!PySequence_Check(*op))
	return 0;
    n = PySequence_Length(*op);
    items = PyMem_NEW(GLint, n);
    if (items == NULL) {
	PyErr_NoMemory();
	return 0;
    }
    for (i = 0; i < n; i++) {
	if ((item = PySequence_GetItem(*op, i))) {
	    items[i] = PyInt_AsLong(item);
	    Py_DECREF(item);
	}
	if (PyErr_Occurred())
	    return 0;
    }
    *pitems = items;
    *pn = n;
    return 1;
}

static int PyArray_AsShortArray(PyObject ** op, GLshort ** pitems, int *pn)
{
    GLshort *items;
    PyObject *item;
    int n, i;
    if (!PySequence_Check(*op))
	return 0;
    n = PySequence_Length(*op);
    items = PyMem_NEW(GLshort, n);
    if (items == NULL) {
	PyErr_NoMemory();
	return 0;
    }
    for (i = 0; i < n; i++) {
	if ((item = PySequence_GetItem(*op, i))) {
	    items[i] = PyInt_AsLong(item);
	    Py_DECREF(item);
	}
	if (PyErr_Occurred())
	    return 0;
    }
    *pitems = items;
    *pn = n;
    return 1;

}

#endif				/* Not NUMERIC */


/* State management */

static PyObject *py_gl_EnableClientState(PyObject * self, PyObject * args)
{
    GLenum arg1;
    TRY(PyArg_ParseTuple(args, "i", &arg1));
    glEnableClientState(arg1);
    Py_INCREF(Py_None);
    return Py_None;
}
static PyObject *py_gl_DisableClientState(PyObject * self, PyObject * args)
{
    GLenum arg1;
    TRY(PyArg_ParseTuple(args, "i", &arg1));
    glDisableClientState(arg1);
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *py_gl_PushClientAttrib(PyObject * self, PyObject * args)
{
    GLbitfield arg1;
    TRY(PyArg_ParseTuple(args, "i", &arg1));
    glPushClientAttrib(arg1);
    Py_INCREF(Py_None);
    return Py_None;
}
static PyObject *py_gl_PopClientAttrib(PyObject * self, PyObject * args)
{
    TRY(PyArg_ParseTuple(args, ""));
    glPopClientAttrib();
    Py_INCREF(Py_None);
    return Py_None;
}

/* State Querying */
static PyObject *py_gl_GetLight(PyObject * self, PyObject * args)
{
    GLenum light, pname;
    GLfloat oneparam; /* store a single-value parameter */
    int dims[1]; /* store the dimensions of the parameter array */
#ifdef NUMERIC
    PyArrayObject *params;
#else				/* NUMERIC */
    int i, nitems;
    GLfloat *items;
    PyObject *params;
#endif				/* NUMERIC */
    if (!PyArg_ParseTuple(args, "ii", &light, &pname))
		return NULL;
    switch (pname) {
		/* four items */
		case GL_AMBIENT:
		case GL_DIFFUSE:
		case GL_SPECULAR:
		case GL_POSITION:
			dims[0] = 4;
			break;
		/* three items */
		case GL_SPOT_DIRECTION:
			dims[0] = 3;
			break;
		/* one item */
		case GL_SPOT_EXPONENT:
		case GL_SPOT_CUTOFF:
		case GL_CONSTANT_ATTENUATION:
		case GL_LINEAR_ATTENUATION:
		case GL_QUADRATIC_ATTENUATION:
		default:
			glGetLightfv(light, pname, &oneparam);
			return Py_BuildValue("d", oneparam);
	}
		
#ifdef NUMERIC
    TRY(params = (PyArrayObject *) PyArray_FromDims(1, dims, PyArray_FLOAT));
    glGetLightfv(light, pname, (float *) params->data);
#else				/* NUMERIC */
    nitems = dims[0];
    TRY(items = PyMem_NEW(GLfloat, nitems));
    glGetLightfv(light, pname, items);
    TRY(params = PyTuple_New(nitems));
    for (i = 0; i < nitems; i++) {
	PyTuple_SET_ITEM(params, i, PyFloat_FromFloat(items[i]));
    }
    PyMem_DEL(items);
#endif				/* NUMERIC */
    return (PyObject *) params;
}

static PyObject *py_gl_GetMaterial(PyObject * self, PyObject * args)
{
    GLenum face, pname;
    GLfloat oneparam; /* store a single-value parameter */
    int dims[1]; /* store the dimensions of the parameter array */
#ifdef NUMERIC
    PyArrayObject *params;
#else				/* NUMERIC */
    int i, nitems;
    GLfloat *items;
    PyObject *params;
#endif				/* NUMERIC */
    if (!PyArg_ParseTuple(args, "ii", &face, &pname))
		return NULL;
    switch (pname) {
		/* four items */
		case GL_AMBIENT:
		case GL_DIFFUSE:
		case GL_SPECULAR:
		case GL_EMISSION:
			dims[0] = 4;
			break;
		/* three items */
		case GL_COLOR_INDEXES:
			dims[0] = 3;
			break;
		/* one item */
		case GL_SHININESS:
		default:
			glGetMaterialfv(face, pname, &oneparam);
			return Py_BuildValue("d", oneparam);
	}
		
#ifdef NUMERIC
    TRY(params = (PyArrayObject *) PyArray_FromDims(1, dims, PyArray_FLOAT));
    glGetMaterialfv(face, pname, (float *) params->data);
#else				/* NUMERIC */
    nitems = dims[0];
    TRY(items = PyMem_NEW(GLfloat, nitems));
    glGetMaterialfv (face, pname, items);
    TRY(params = PyTuple_New(nitems));
    for (i = 0; i < nitems; i++) {
	PyTuple_SET_ITEM(params, i, PyFloat_FromFloat(items[i]));
    }
    PyMem_DEL(items);
#endif				/* NUMERIC */
    return (PyObject *) params;
}

static PyObject *py_gl_GetPixelMap(PyObject * self, PyObject * args)
{
    GLenum pname;
	GLint tempsize;
#ifdef NUMERIC
    PyArrayObject *params;
#else				/* NUMERIC */
	GLsizei * size[1];
    int i, nitems;
    GLfloat *items;
    PyObject *params;
#endif				/* NUMERIC */
    if (!PyArg_ParseTuple(args, "i", &pname))
		return NULL;
	/* clear error flag so we'll know if glGet fails */
	glGetError();
/*	TRY(tempsize = PyMem_NEW(GLint, 1)); */
    switch (pname) {
		/* four items */
		case GL_PIXEL_MAP_I_TO_I:
			glGetIntegerv(
				GL_PIXEL_MAP_I_TO_I_SIZE,
				&tempsize
			);
			break;
		case GL_PIXEL_MAP_S_TO_S:
			glGetIntegerv(
				GL_PIXEL_MAP_S_TO_S_SIZE,
				&tempsize
			);
			break;
		case GL_PIXEL_MAP_I_TO_R:
			glGetIntegerv(
				GL_PIXEL_MAP_I_TO_R_SIZE,
				&tempsize
			);
			break;
		case GL_PIXEL_MAP_I_TO_G:
			glGetIntegerv(
				GL_PIXEL_MAP_I_TO_G_SIZE,
				&tempsize
			);
			break;
		case GL_PIXEL_MAP_I_TO_B:
			glGetIntegerv(
				GL_PIXEL_MAP_I_TO_B_SIZE,
				&tempsize
			);
			break;
		case GL_PIXEL_MAP_I_TO_A:
			glGetIntegerv(
				GL_PIXEL_MAP_I_TO_A_SIZE,
				&tempsize
			);
			break;
		case GL_PIXEL_MAP_R_TO_R:
			glGetIntegerv(
				GL_PIXEL_MAP_R_TO_R_SIZE,
				&tempsize
			);
			break;
		case GL_PIXEL_MAP_G_TO_G:
			glGetIntegerv(
				GL_PIXEL_MAP_G_TO_G_SIZE,
				&tempsize
			);
			break;
		case GL_PIXEL_MAP_B_TO_B:
			glGetIntegerv(
				GL_PIXEL_MAP_B_TO_B_SIZE,
				&tempsize
			);
			break;
		case GL_PIXEL_MAP_A_TO_A:
			glGetIntegerv(
				GL_PIXEL_MAP_A_TO_A_SIZE,
				&tempsize
			);
			break;
		default:
			/* shoudl really raise some sort of error, no? */
			return NULL;
	}
	if (! glGetError() == GL_NO_ERROR) {
		return NULL;
	}
		
#ifdef NUMERIC
    TRY(params = (PyArrayObject *) PyArray_FromDims(1, (int *) & tempsize, PyArray_FLOAT));
    glGetPixelMapfv(pname, (float *) params->data);
#else				/* NUMERIC */
    nitems = tempsize;
    TRY(items = PyMem_NEW(GLfloat, nitems));
    glGetPixelMapfv (face, pname, items);
    TRY(params = PyTuple_New(nitems));
    for (i = 0; i < nitems; i++) {
	PyTuple_SET_ITEM(params, i, PyFloat_FromFloat(items[i]));
    }
    PyMem_DEL(items);
#endif				/* NUMERIC */
    return (PyObject *) params;
}

static PyObject *py_gl_GetPolygonStipple(PyObject * self, PyObject * args)
{
#ifdef NUMERIC
	PyArrayObject * params;
	int dims[2] = {32,4};
#else /* numeric */
	int nitems = 32*4;
	GLubyte *items;
#endif

    if (!PyArg_ParseTuple(args, ""))
		return NULL;
#ifdef NUMERIC
    TRY(params = (PyArrayObject *) PyArray_FromDims(2, dims, PyArray_UBYTE));
    glGetPolygonStipple((GLubyte *) params->data);
#else				/* NUMERIC */
    TRY(items = PyMem_NEW(GLubyte, nitems));
    glGetPolygonStipple(items);
    TRY(params = PyTuple_New(nitems));
    for (i = 0; i < nitems; i++) {
		PyTuple_SET_ITEM(params, i, PyInt_FromInt((int)(items[i])));
    }
    PyMem_DEL(items);
#endif				/* NUMERIC */
    return (PyObject *) params;
}



/* Array specification */

static int type2gl[] =
    { -1, GL_UNSIGNED_BYTE, GL_BYTE, GL_SHORT, GL_INT, -1, GL_FLOAT,
    GL_DOUBLE, -1, -1, -1, -1
};

static PyObject *py_gl_VertexPointer(PyObject * self, PyObject * args)
{
    GLint size;
    GLenum type;
    GLsizei stride;
    PyObject *op;
    PyArrayObject *ap;
    TRY(PyArg_ParseTuple(args, "iiO", &size, &stride, &op));
    if (PyArray_Check(op))
	ap = (PyArrayObject *) op;
    else {
	TRY(ap =
	    (PyArrayObject *) PyArray_ContiguousFromObject(op,
							   PyArray_DOUBLE,
							   1, 0));
    }
    type = type2gl[ap->descr->type_num];
    ASSERT(type != -1, "Can't convert this type of array!");
    glVertexPointer(size, type, stride, ap->data);
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *py_gl_ColorPointer(PyObject * self, PyObject * args)
{
    GLint size;
    GLenum type;
    GLsizei stride;
    PyObject *op;
    PyArrayObject *ap;
    TRY(PyArg_ParseTuple(args, "iiO", &size, &stride, &op));
    if (PyArray_Check(op))
	ap = (PyArrayObject *) op;
    else {
	TRY(ap =
	    (PyArrayObject *) PyArray_ContiguousFromObject(op,
							   PyArray_DOUBLE,
							   1, 0));
    }
    type = type2gl[ap->descr->type_num];
    ASSERT(type != -1, "Can't convert this type of array!");
    glColorPointer(size, type, stride, ap->data);
    Py_INCREF(Py_None);
    return Py_None;
}


static PyObject *py_gl_NormalPointer(PyObject * self, PyObject * args)
{
    GLint size;
    GLenum type;
    GLsizei stride;
    PyObject *op;
    PyArrayObject *ap;
    TRY(PyArg_ParseTuple(args, "iiO", &size, &stride, &op));
    if (PyArray_Check(op))
	ap = (PyArrayObject *) op;
    else {
	TRY(ap =
	    (PyArrayObject *) PyArray_ContiguousFromObject(op,
							   PyArray_DOUBLE,
							   1, 0));
    }
    type = type2gl[ap->descr->type_num];
    ASSERT(type != -1, "Can't convert this type of array!");
    glNormalPointer(type, stride, ap->data);
    Py_INCREF(Py_None);
    return Py_None;
}
static PyObject *py_gl_TexCoordPointer(PyObject * self, PyObject * args)
{
    GLint size;
    GLenum type;
    GLsizei stride;
    PyObject *op;
    PyArrayObject *ap;
    TRY(PyArg_ParseTuple(args, "iiO", &size, &stride, &op));
    if (PyArray_Check(op))
	ap = (PyArrayObject *) op;
    else {
	TRY(ap =
	    (PyArrayObject *) PyArray_ContiguousFromObject(op,
							   PyArray_DOUBLE,
							   1, 0));
    }
    type = type2gl[ap->descr->type_num];
    ASSERT(type != -1, "Can't convert this type of array!");
    glTexCoordPointer(size, type, stride, ap->data);
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *py_gl_IndexPointer(PyObject * self, PyObject * args)
{
    GLenum type;
    GLsizei stride;
    PyObject *op;
    PyArrayObject *ap;
    TRY(PyArg_ParseTuple(args, "iO", &stride, &op));
    if (PyArray_Check(op))
	ap = (PyArrayObject *) op;
    else {
	TRY(ap =
	    (PyArrayObject *) PyArray_ContiguousFromObject(op,
							   PyArray_INT,
							   1, 0));
    }
    type = type2gl[ap->descr->type_num];
    ASSERT(type != -1, "Can't convert this type of array!");
	glIndexPointer( type, stride, ap->data );
    Py_INCREF(Py_None);
    return Py_None;
}




static PyObject *py_gl_InterleavedArrays(PyObject * self, PyObject * args)
{
	/* for some reason this function just won't
	work properly.  No errors raised, just doesn't
	seem to transfer any data. */
    GLenum format;
    GLint stride;
    GLenum type;
    PyObject *op;
    PyArrayObject *ap;
    TRY(PyArg_ParseTuple(args, "iiO", &format, &stride, &op));
    if (PyArray_Check(op))
	ap = (PyArrayObject *) op;
    else {
	TRY(ap =
	    (PyArrayObject *) PyArray_ContiguousFromObject(op,
							   PyArray_FLOAT,
							   1, 0));
    }
    type = type2gl[ap->descr->type_num];
    ASSERT(type != -1, "Can't convert this type of array!");
    glInterleavedArrays(format, stride, ap->data);
    Py_INCREF(Py_None);
    return Py_None;
}



/* Array Use (drawing) */

static PyObject *py_gl_ArrayElement(PyObject * self, PyObject * args)
{
    GLint arg1;
    TRY(PyArg_ParseTuple(args, "i", &arg1));
    glArrayElement(arg1);
    Py_INCREF(Py_None);
    return Py_None;
}

static char py_gl_DrawArrays__doc__[] = 
"glDrawArrays(GLenum mode, GLint first, GLsizei count)\n\
  mode -- The kind of primitives to render. It can \n\
    assume one of the following symbolic values:\n\
		GL_POINTS,\n\
		GL_LINE_STRIP,\n\
		GL_LINE_LOOP,\n\
		GL_LINES, \n\
		GL_TRIANGLE_STRIP, \n\
		GL_TRIANGLE_FAN, \n\
		GL_TRIANGLES, \n\
		GL_QUAD_STRIP, \n\
		GL_QUADS, and \n\
		GL_POLYGON. \n\
  first -- The array index from which to start\n\
  count -- The number of verticies to be drawn";


static PyObject *py_gl_DrawArrays(PyObject * self, PyObject * args)
{
    GLenum mode;
    GLint first;
    GLsizei count;
    TRY(PyArg_ParseTuple(args, "iii", &mode, &first, &count));
    glDrawArrays(mode, first, count);
    Py_INCREF(Py_None);
    return Py_None;
}


static char py_gl_DrawElements__doc__[] =
"glDrawElements(GLenum mode, array indices)\n\
  mode -- The kind of primitives to render. It can \n\
    assume one of the following symbolic values:\n\
      GL_POINTS,\n\
      GL_LINE_STRIP,\n\
	  GL_LINE_LOOP,\n\
	  GL_LINES, \n\
	  GL_TRIANGLE_STRIP, \n\
	  GL_TRIANGLE_FAN, \n\
	  GL_TRIANGLES, \n\
	  GL_QUAD_STRIP, \n\
	  GL_QUADS, and \n\
	  GL_POLYGON. \n\
  indices -- (Numeric) array of indices into the\n\
    active arrays.  Indicies should be integer values.";


static PyObject *py_gl_DrawElements(PyObject * self, PyObject * args)
{
    GLenum mode;
    GLint count;
/*    GLenum type; */
    PyObject *op;
    PyArrayObject *ap;
    TRY(PyArg_ParseTuple(args, "iO", &mode, &op));
    if (PyArray_Check(op))
	ap = (PyArrayObject *) op;
    else {
	TRY(ap =
	    (PyArrayObject *) PyArray_ContiguousFromObject(op,
							   PyArray_INT,
							   1, 0));
    }
	/* do I not need to do an explicit cast for the data
	from INT to unsigned Int??? */
/*    type = type2gl[ap->descr->type_num]; 
    ASSERT(type != -1, "Can't convert this type of array!"); */
	count = PyArray_SIZE(ap);
    glDrawElements(mode, count, GL_UNSIGNED_INT, ap->data);
    Py_INCREF(Py_None);
    return Py_None;
}

static PyMethodDef py_gl_methods[] = {
    {"glEnableClientState", py_gl_EnableClientState, 1},
    {"glDisableClientState", py_gl_DisableClientState, 1},
    {"glPushClientAttrib", py_gl_PushClientAttrib, 1},
    {"glPopClientAttrib", py_gl_PopClientAttrib, 1},

    {"glGetLightfv", py_gl_GetLight, 1},
    {"glGetLight", py_gl_GetLight, 1},
    {"glGetMaterial", py_gl_GetMaterial, 1},
    {"glGetMaterialfv", py_gl_GetMaterial, 1},
    {"glGetPixelMap", py_gl_GetPixelMap, 1},
    {"glGetPixelMapfv", py_gl_GetPixelMap, 1},
    {"glGetPolygonStipple", py_gl_GetPolygonStipple, 1},

    {"glVertexPointer", py_gl_VertexPointer, 1},
    {"glColorPointer", py_gl_ColorPointer, 1},
    {"glIndexPointer", py_gl_IndexPointer, 1},
    {"glNormalPointer", py_gl_NormalPointer, 1},
    {"glTexCoordPointer", py_gl_TexCoordPointer, 1},
	{"glInterleavedArrays", py_gl_InterleavedArrays, 1},

    {"glArrayElement", py_gl_ArrayElement, 1},
    {"glDrawArrays", py_gl_DrawArrays, 1, py_gl_DrawArrays__doc__},
	{"glDrawElements", py_gl_DrawElements, 1, py_gl_DrawElements__doc__},
    {NULL, NULL}
};

#ifdef WIN32
__declspec(dllexport)
#endif
void initarraystuff(void)
{
    PyObject *m;
    m = Py_InitModule("arraystuff", py_gl_methods);
    import_array();
    if (PyErr_Occurred())
	Py_FatalError("can't initialize module arraystuff");
}
