// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "gldevice.h"
#include "prim.h"
#include "exceptions.h"
#include "frame.h"
#include <iostream>
#include <stdexcept>

/***************** GLDevice *******************/

namespace visual {

glFont::~glFont()
{
}
	
GLDevice::GLDevice()
	: mode(HIDE),
	mouse_mode(INITIALIZE),
	last_mouse_mode(NOBUTTONS),
 	initstereo(NONE),
 	stereodepth( 0.0f),
	active(false),
	manual_scale(1.0),
	initx(-1), inity(-1),
	initw(430), inith(430),
	initfull(0),
	newmouse_enabled(1),
	quitOnClose(0),
	oldButtons(0),
	clickDelta(clickTolerance),
	last_wct( tmatrix() )
{
	tmatrix z; 
	z.x_column(0,0,0);
	z.y_column(0,0,0);
	z.z_column(0,0,0);
	z.w_column(0,0,0);
	z.w_row(0,0,0);
	last_wct = z;
}

GLDevice::~GLDevice() 
{
	hide();
	join();
}

bool 
GLDevice::show() 
{
	if (active && mode==DISPLAY) 
		return true;
	
	mode = SHOW;
	addCallback();
	join();
	if (mode == DISPLAY) {
		addCallback();
		return true;
	} 
	else {
		PySys_WriteStderr("OpenGL initialization failed.\n"
			"  %s\n", error_message.c_str());
		return false;
	}
	
	return false;
}

std::string 
GLDevice::info() 
{
	if (closed()) {
		return std::string( "Renderer inactive.\n");
	} 
	else {
		std::string s;
		s += "OpenGL renderer active.\n  Vendor: "
		  + vendor
		  + "\n  Version: " + version
		  + "\n  Renderer: " + renderer
		  + "\n  Extensions: ";
		
		// this->extensions is a whitespace-separated list of extensions.
		std::istringstream buffer(extensions);
		std::istream_iterator<std::string> parser(buffer);
		std::istream_iterator<std::string> end;
		while (parser != end) {
			s += *parser + "\n";
			++parser;
		}
		return s;
	}
}

void 
GLDevice::hide() 
{
	mode = HIDE;
}

bool 
GLDevice::closed() 
{
	return !active;
}

void 
GLDevice::join() 
{
	while (active) {
		threaded_sleep(0.100);
	}
}

void 
GLDevice::frame() 
{
	if (closed()) {
		show();
		return;
	}
	mode = FRAME;
	join();
	if (mode == DISPLAY)
		addCallback();
}

void 
GLDevice::addCallback() 
{
	active = true;
	threaded_timer( 0, &GLDevice::callback, this );
}

void 
GLDevice::setX( int v) 
{
	if (active) 
		throw std::runtime_error("Window attributes are not accessible once the window has been created.\n");
	initx = v;
}

void 
GLDevice::setY( int v) 
{
	if (active) 
		throw std::runtime_error("Window attributes are not accessible once the window has been created.\n");
	inity = v;
}

void 
GLDevice::setWidth( int v) 
{
	if (active) 
		throw std::runtime_error("Window attributes are not accessible once the window has been created.\n");
	initw = v;
}

void 
GLDevice::setHeight( int v) 
{
	if (active) 
		throw std::runtime_error("Window attributes are not accessible once the window has been created.\n");
	inith = v;
}

void 
GLDevice::setFullScreen( bool tf) 
{
	if (active) 
		throw std::runtime_error("Window attributes are not accessible once the window has been created.\n");
	initfull = tf;
}

void 
GLDevice::setStereo( stereotype tf) 
{
	if (active) 
		throw std::runtime_error("Window attributes are not accessible once the window has been created.\n");
	initstereo = tf;
}

void 
GLDevice::setStereoDepth( float v) {
	if (active) 
		throw std::runtime_error("Window attributes are not accessible once the window has been created.\n");
	stereodepth = v;
}

void 
GLDevice::setNewmouse( bool tf) 
{
	newmouse_enabled = tf;
}

int 
GLDevice::getX() 
{
	if (active) 
		throw std::runtime_error("Window attributes are not accessible once the window has been created.\n");
	return initx;
}

int 
GLDevice::getY() 
{
	if (active) 
		throw std::runtime_error("Window attributes are not accessible once the window has been created.\n");
	return inity;
}

int 
GLDevice::getWidth() 
{
	if (active) 
		throw std::runtime_error("Window attributes are not accessible once the window has been created.\n");
	return initw;
}

int 
GLDevice::getHeight() 
{
	if (active) 
		throw std::runtime_error("Window attributes are not accessible once the window has been created.\n");
	return inith;
}


bool
GLDevice::getFullScreen() const
{
	return initfull;
}

stereotype
GLDevice::getStereo() const
{
	return initstereo;
}

float 
GLDevice::getStereoDepth() const
{
	return stereodepth;
}

/******* The following methods are called from a threaded_timer, potentially
             in parallel with calls to the preceding methods. *******************/

// wct and cam are initialized by this function.
// eyesign must be -1 for left eye, 0 for center, or 1 for right eye
void 
GLDevice::getProjection( tmatrix& wct, vector& cam, int eyesign) 
{
	int width = cx.width();
	//JMZ passive stereo each view is 1/2 width of window
	//  now use width throughout rather than cx.width()
	if (initstereo == PASSIVE || initstereo == CROSSEYED) {
		width /= 2;
	}

	if (display->c_uniform) {
		manual.x_column(manual_scale,0,0);
		manual.y_column(0,manual_scale,0);
		manual.z_column(0,0,manual_scale);
	} 
	else if (cx.height() < width) {
		manual.x_column(manual_scale,0,0);
		manual.y_column(0,manual_scale * cx.height() / width, 0);
		manual.z_column(0,0,manual_scale * cx.height() / width);
	} 
	else {
		manual.x_column(manual_scale * width / cx.height(), 0);
		manual.y_column(0,manual_scale,0);
		manual.z_column(0,0,manual_scale);
	}
	manual.w_column();
	manual.w_row();
	
	// the camera is at [0 0 0 1] in view space, so we
	//	 calculate (model^-1) (manual^-1) (view^-1) [0 0 0 1]T
	vector iscale(1.0/manual[0][0], 1.0/manual[1][1], 1.0/manual[2][2]);
	cam = display->imodel * ((display->iview * vector(0,0,0)).scale(iscale));

	// eyesign is either -1 for left, 0 for center, or 1 for right
	double hfov = 0.0;
	double vfov = 0.0;
	
	double aspect = static_cast<double>(cx.height()) / width;
	if (aspect < 1.0) {
		hfov = display->tanfov;
		vfov = hfov * aspect;
	}
	else {
		vfov = display->tanfov;
		hfov = vfov / aspect;
	}
	double cotfov = 1.0 / display->tanfov;
	double ext = manual_scale * display->model.times_v(display->c_extent).mag();
	if (!ext) 
		ext=1;	// if the scene is a point, any depth range will do!
	
	
	double farclip = cotfov + ext;
	double nearclip = 0.0;
	if ((cam - display->c_center).mag() < display->c_extent.mag()) {
		// Then the camera is within the scene.  Pick a value that looks OK.
		nearclip = 0.015;
	}
	else {
		nearclip = cotfov - ext*1.5;
		if (nearclip < 0.01*farclip) 
			nearclip = 0.01*farclip;
	}
	
	//tmatrix proj, iproj;
	double R = nearclip*hfov;
	double T = nearclip*vfov;
	
	double fl = 0.5*ext + ext*stereodepth + nearclip;  //focal length
	double eyeOffset = eyesign*fl/60.0;  // eye separation 1/30 of focallength
	double eyeOffset1 = eyeOffset * (nearclip/fl);
	frustum(proj, iproj, -R-eyeOffset1, R-eyeOffset1, -T, T, nearclip, farclip);
	

	// wct = display->model o manual o display->view o eyetrans o proj
	// analagous to:
	// glScale3d() for the scene scaling factor,
	// glScale3d() for the manual scaling factor (determined by the mouse)
	// gluLookAt() to position the camera
	// glTranslatef() for moving the eye to the left/right as appropriate
	// gluFrustum() for projective division.
	tmatrix model(display->model, manual);
	tmatrix modelview(model, display->view);
	
	// Now move camera to eye 
	tmatrix eyetrans = tmatrix::identity();
	eyetrans.w_column(-eyeOffset,0,0);
	tmatrix eyeview(modelview, eyetrans);
	
	wct = tmatrix( eyeview, proj );
}

vector 
GLDevice::calcMousePos( vector mpos) 
{
	vector p(mpos.x*2-1, 1-mpos.y*2, 0.5);
	p = (iproj * p)/iproj.w(p);
	/* p is in view space.	The camera is at [0,0,0]; the
	center is at c=[0 0 -1/tanfov].  We scale p so that
	p.z == -1/tanfov, without changing its direction. */
	p = p * (-1/display->tanfov) / p.z;
	
	p = display->iview * p;
	p.x /= manual[0][0]; p.y /= manual[1][1]; p.z /= manual[2][2];
	p = display->imodel * p;
	
	return p;
}

/* Update the cache of the current mouse status (in the mouseobject) and (possibly)
 *   add a new clickObject to the queue.
 * pos := the position of the mouse cursor
 * cam := The position vector of the camera, as positioned by the user
 * ray := The forward direction vector of the camera.
 * pick := The DisplayObject that is inline with and closest to the cursor
 * pickpos := The first intersection of the ray and the pick object.
 */
void 
GLDevice::mouseControl( vector pos, vector cam, vector ray, boost::shared_ptr<DisplayObject> pick, vector pickpos) 
{
	vector delta = cx.getMouseDelta(); // mouse movement in pixels
	unsigned buttons = cx.getMouseButtons();
	unsigned changed = cx.getMouseButtonsChanged();
	bool mevent = false;
	bool extraPRESS = false;
	int winh = cx.height();
	int winw = cx.width();
	boost::shared_ptr<cursorObject> curs = display->cursor;
	int Mshift = cx.getShiftKey();
	int Malt = cx.getAltKey();
	int Mctrl = cx.getCtrlKey();
	
	if (curs->visible != curs->last_visible) {
		curs->last_visible = curs->visible;
		if (curs->visible) {
			cx.showMouse();
		}
		else {
			cx.hideMouse();
		}
	}
	
	// Cache the current state of the mouse.
  	{
		boost::shared_ptr<mouseObject> m = display->mouse;
		mutex::lock L(m->mtx);
		m->position = pos;
		m->pick = pick;
		m->pickpos = pickpos;
		m->cam = cam;
		m->ray = ray;
		m->buttons = buttons;
		m->set_shift( Mshift);
		m->set_alt( Malt);
		m->set_ctrl( Mctrl);
		if (mouse_mode == INITIALIZE) {
			m->clickCount = 0;
			mouse_mode = NOBUTTONS;
		}
	}
	
	// Buttons: left down=1, right down=2, left & right down=3, (Windows) wheel down=4.
	// If zooming or rotating, don't want either down or up events to come through.
	// 7 = 0111b
	buttons = (buttons & 7);
	
	switch (mouse_mode) {
		case NOBUTTONS:
			if (changed) {
				mouse_mode = PRESS;
				clickDelta = clickTolerance;
				mevent = true;
				if (!buttons) { // buttons have already been released
					extraPRESS = true;
					mouse_mode = CLICK;
					oldButtons = changed; // simulate earlier PRESS
				}
			}
			break;
		case PRESS:
			if (buttons && changed) {
				mevent = true;
		    }
			if (!buttons && clickDelta >= 0) {
				mouse_mode = CLICK;
				mevent = true;
			} 
			else if (buttons==oldButtons && clickDelta<0) {
				if (display->c_rotation_enabled && (buttons==2)) {
					mouse_mode = SPIN;
				} 
				else if (display->c_zoom_enabled 
					&& (newmouse_enabled && (buttons==3 || buttons==4)) 
					|| (!newmouse_enabled && (buttons == 1))) {
					mouse_mode = ZOOM;
				} 
				else {
					mouse_mode = DRAG;
					mevent = true;
				}
			}
			clickDelta -= delta.mag();
			break;
		case DRAG:
			if (!buttons) {
				mouse_mode = DROP;
				mevent = true;
			} 
			else if (display->c_rotation_enabled && (buttons==2)) {
				mouse_mode = SPIN;
			} 
			else if (display->c_zoom_enabled 
				&& (newmouse_enabled && (buttons==3 || buttons==4)) 
				|| (!newmouse_enabled && (buttons == 1))) {
				mouse_mode = ZOOM;
			}
			break;	 
	} // !switch (mouse_mode)
	
	switch (mouse_mode) {
		case SPIN:
			{
				vector forward = display->c_forward;
				vector up = display->c_up;
				tmatrix m;
				
				cx.lockMouse();
				clickDelta -= delta.mag();
				rotation(m, (delta.x/winw)*-2, up.norm());
				forward = m * forward;
				
				double ya = -(delta.y/winh)*2;
				double limit = std::acos( norm_dot( up,forward) );
				if (ya < limit-pi || ya > limit)
                    display->set_up( -up);
				
				rotation( m, ya, forward.cross( up).norm());
				forward = m * forward;
				display->setForward( forward );
			}
			if (buttons!=2) {
				if (buttons==0) {
					mouse_mode = NOBUTTONS;
					if (last_mouse_mode==DRAG) {
						mouse_mode = DROP;
						mevent = true;
					}
					cx.unlockMouse();
				} 
				else if (buttons==3 || buttons==4) {
					mouse_mode = ZOOM;
				} 
				else if (buttons==oldButtons) {
					mouse_mode = last_mouse_mode;
					cx.unlockMouse();
				} 
				else { 
					// must be new (buttons==1)
					mouse_mode = PRESS;
					clickDelta = clickTolerance;
					mevent = true;
					cx.unlockMouse();
				}
			}
			break;
		case ZOOM:
			cx.lockMouse();
			clickDelta -= delta.mag();
			manual_scale *= std::pow( 10, -delta.y/winh);
			if (!(buttons==3 || buttons==4)) {
				if (buttons==0) {
					mouse_mode = NOBUTTONS;
					if (last_mouse_mode==DRAG) {
						mouse_mode = DROP;
						mevent = true;
					}
					cx.unlockMouse();
				} 
				else if (buttons==2) {
					mouse_mode = SPIN;
				} 
				else if (buttons==oldButtons) {
					mouse_mode = last_mouse_mode;
					cx.unlockMouse();
				} 
				else { 
					// must be new (buttons==1)
					mouse_mode = PRESS;
					clickDelta = clickTolerance;
					mevent = true;
					cx.unlockMouse();
				}
			}
			break;
	} // !switch (mouse_mode)

  	// Extra PRESS event, immediately preceding a CLICK event
	if (extraPRESS) {
		boost::shared_ptr<clickObject> c(new clickObject());
		c->position = pos;
		c->pick = pick;
		c->pickpos = pickpos;
		c->cam = cam;
		c->ray = ray;
		c->buttons = changed;
		c->set_press( true);
		c->set_click( false);
		c->set_drag( false);
		c->set_drop( false);
		c->set_release( false);
		c->set_shift( Mshift); // Using implicit type conversions here.
		c->set_alt( Malt);
		c->set_ctrl( Mctrl);
		{
			mutex::lock L(display->mouse->mtx);
			display->mouse->clicks.push( std::make_pair( c, false));
		}
		// A click event uses the info from the press event:
		lastEvent.position = pos;
		lastEvent.pick = pick;
		lastEvent.pickpos = pickpos;
		lastEvent.cam = cam;
		lastEvent.ray = ray;
	} // !if (extraPRESS)
	
	// Real mouse event, not just zoom or rotate
	if (mevent) {
		boost::shared_ptr<clickObject> c( new clickObject());
		bool clicked = false;
		c->position = pos;
		c->pick = pick;
		c->pickpos = pickpos;
		c->cam = cam;
		c->ray = ray;
		c->buttons = buttons;
		c->set_press( false);
		c->set_click( false);
		c->set_drag( false);
		c->set_drop( false);
		c->set_release( false);
		c->set_shift( Mshift); // Relying on implicit type conversions from int->bool
		c->set_alt( Malt);
		c->set_ctrl( Mctrl);
			
		switch (mouse_mode) {
			case PRESS:
				c->set_press( true);
				lastEvent.position = pos;
				lastEvent.pick = pick;
				lastEvent.pickpos = pickpos;
				lastEvent.cam = cam;
				lastEvent.ray = ray;
				break;
			case CLICK:
				c->set_click( true);
				c->set_release( true);
				c->buttons = changed;
				mouse_mode = NOBUTTONS;
				clicked = (oldButtons == 1);
				c->position = lastEvent.position; //report original press state
				c->pick = lastEvent.pick;
				c->pickpos = lastEvent.pickpos;
				c->cam = lastEvent.cam;
				c->ray = lastEvent.ray;
				break;
			case DRAG:
				c->set_drag( true);
				c->position = lastEvent.position; // report original press state
				c->pick = lastEvent.pick;
				c->pickpos = lastEvent.pickpos;
				c->cam = lastEvent.cam;
				c->ray = lastEvent.ray;
				break;
			case DROP:
				c->set_drop( true);
				c->set_release( true);
				c->buttons = oldButtons;
				mouse_mode = NOBUTTONS;
				break;
		} // !switch (mouse_mode)
		oldButtons = buttons;
		last_mouse_mode = mouse_mode;
		{
			mutex::lock L(display->mouse->mtx);
			display->mouse->clicks.push( std::make_pair(c, clicked));
			if (clicked) {
				display->mouse->clickCount += 1;
			}
		}
	} // !if (mevent)				
}

/* Thread-safely copy keys from the glContext cache to the the kbObject cache.
 */
void 
GLDevice::kbControl() 
{
	boost::shared_ptr<kbObject> kb = display->kb;
	while (1) {
		std::string s = cx.getKeys();
		if (s == "") 
			break;
		kb->push_new_key(s);
	}
}

int 
GLDevice::render_control() 
{
	if (mode == HIDE) {
		cx.cleanup();
		return -1;
	}
	
	{ 
		DisplayObject::read_lock D(display->mtx);
		display->updateCache();
	}
	
	if (mode == SHOW) {
		int flags = glContext::DEFAULT;
		if (initfull)
			flags |= glContext::FULLSCREEN;
		if (initstereo == PASSIVE || initstereo == CROSSEYED){
			initw *= 2;  // double window width for two views
		}
		else if (initstereo == ACTIVE){
			flags |= glContext::QB_STEREO;
		}
	
		if( !cx.initWindow( display->c_title.c_str(), initx, inity, initw, inith, flags)) {
			error_message = cx.lastError();
			return -1;
		}
		cx.makeCurrent();
		vendor = (const char*)glGetString(GL_VENDOR);
		version = (const char*)glGetString(GL_VERSION);
		renderer = (const char*)glGetString(GL_RENDERER);
		extensions = (const char*)glGetString(GL_EXTENSIONS);
		cx.makeNotCurrent();
		mode = DISPLAY;
		return -1;
	}
	
	if (!cx.isOpen()) {
		cx.cleanup();
		if (quitOnClose) {
			Display::internal_shutdown();
		}
		return -1;
	}
	
	if (display->c_title_changed)
		cx.changeWindow( display->c_title.c_str(), -1, -1, -1, -1, -1 );
	
	cx.makeCurrent();
	glDepthMask(GL_TRUE);
	glViewport( 0, 0, cx.width(), cx.height() );
	if (initstereo == RED_BLUE) {
		float intensity = display->bgcolor.grayscale();
		glClearColor( intensity, 0.0, intensity, 1 );
	}
	else {
		glClearColor( display->bgcolor.r, display->bgcolor.g, display->bgcolor.b, 1 );
	}

	glClearDepth( 1 );
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
	glEnable(GL_DEPTH_TEST);
	glDepthFunc( GL_LEQUAL );
	glDisable(GL_BLEND);
	
	switch (initstereo) {
		case NONE:
			draw();
			break;
		case ACTIVE: // Single screen quad buffered.
			glDrawBuffer( GL_BACK_LEFT);
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
			draw( -1, false, false, false);
			glDrawBuffer( GL_BACK_RIGHT);
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
			draw( 1, true, false, false);
			break;
		case PASSIVE: // side-by-side passive stereo JMZ
			glViewport( 0, 0, cx.width()/2, cx.height());
			draw( -1, false, false, false);
			glViewport( cx.width()/2+1, 0, cx.width()/2, cx.height());
			draw( 1, true, false, false);
			break;
		case CROSSEYED: // side-by-side passive stereo
			// This mode is the same as PASSIVE, but the left pane holds the
			// view for the right eye
			glViewport( 0, 0, cx.width()/2, cx.height());
			draw( 1, false, false, false);
			glViewport( cx.width()/2+1, 0, cx.width()/2, cx.height());
			draw( -1, true, false, false);
			break;
		case RED_BLUE: // Color separation
			// Red channel
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
			glColorMask( GL_TRUE, GL_FALSE, GL_FALSE, GL_TRUE);
			draw( -1, false, true, false);
			// Blue channel
			glColorMask( GL_FALSE, GL_FALSE, GL_TRUE, GL_TRUE);
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
			draw( 1, true, true, false);
			// Put everything back
			glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
			break;
		case RED_CYAN: // Color separation
			// Red channel
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
			glColorMask( GL_TRUE, GL_FALSE, GL_FALSE, GL_TRUE);
			draw( -1, false, true, true);
			// Cyan channel
			glColorMask( GL_FALSE, GL_TRUE, GL_TRUE, GL_TRUE);
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
			draw( 1, true, true, true);
			// Put everything back
			glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
			break;
		case YELLOW_BLUE: // Color separation
			// Yellow channel
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
			glColorMask( GL_TRUE, GL_TRUE, GL_FALSE, GL_TRUE);
			draw( -1, false, true, true);
			// Blue channel
			glColorMask( GL_FALSE, GL_FALSE, GL_TRUE, GL_TRUE);
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
			draw( 1, true, true, true);
			// Put everything back
			glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
			break;
		case GREEN_MAGENTA:
			// Green channel
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
			glColorMask( GL_FALSE, GL_TRUE, GL_FALSE, GL_TRUE);
			draw( -1, false, true, true);
			// Magenta channel
			glColorMask( GL_TRUE, GL_FALSE, GL_TRUE, GL_TRUE);
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
			draw( 1, true, true, true);
			// Put everything back
			glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
		default:
			// We shouldn't ever get here, but if we do, fail safe.
			draw();
	}
	
	cx.swapBuffers();
	cx.makeNotCurrent();
	
	kbControl();
	
	if (mode == FRAME) {
		mode = DISPLAY;
		return -1;
	}

	// roughly 30 fps.
	return 33;
}

void
GLDevice::draw( int eyeOffset, bool doPick, bool anaglyph, bool coloranaglyph)
{
	// anaglyph is true if we are using color separation stereo.
	// coloranaglyph is true if we are using a full color mode, false if we are
	//   using a grayscale mode.
	// doPick is true if we are going to process pick events
	// eyeOffset is one of -1 (left eye), 0 (center), or 1 (right eye)
	tmatrix wct;
	vector cam;
	getProjection(wct, cam, eyeOffset);
	last_wct = wct;
	
	// The position of the mouse in the model space
	vector mpos = calcMousePos(cx.getMousePos());
	// A unit vector from the mouse pointing into the model space.
	vector ray = (mpos - cam).norm();
	// The closest intersection distance between the ray and an object.
	double best_intersect = 1e300;
	// The nearest object that has been selected by the mouse at this moment.
	boost::shared_ptr<DisplayObject> pick;
	
	// Together, these may represent the rectangle into which the scene is 
	// enclosed
	vector min_ext(display->c_center);
	vector max_ext(display->c_center);
	rView view( wct, display->lights, tmatrix::identity(), cx, min_ext, max_ext, 
	            anaglyph, coloranaglyph);

	{
		mutex::lock L( display->list_mutex);

		std::list<boost::shared_ptr<DisplayObject> >::iterator p;
		for ( p = display->objects.begin(); p != display->objects.end(); ++p) {
			DisplayObject::read_lock L( (*p)->mtx);
			
			(*p)->updateCache();
			vector local_cam = cam;
			vector local_ray = ray;
			
			// Manage the color of the target object when using anaglyph modes.
			// Only 'curve' needs to manage its own color.
			const rgb actual_color = (*p)->color;
			if (anaglyph) {
				if (coloranaglyph) {
					// Full color anaglyph modes.
					(*p)->color = actual_color.unsaturate();
				}
				else {
					// Use greyscale colors.
					float intensity = actual_color.grayscale();
					(*p)->color = rgb(intensity, intensity, intensity);
				}
			}

			boost::shared_ptr<visual::frame> parent = (*p)->getParent();
			if (parent.get()) {
				// Apply local transformations from the frame's space to view space.
				DisplayObject::read_lock parent_lock(parent->mtx);
				tmatrix iframe;
				tmatrix frame = parent->getChildTransform();
				iframe.invert_ortho(frame);
				local_cam = iframe * cam;
				local_ray = iframe.times_v( ray );
				rView local( tmatrix(frame, wct)
					, lighting(view.lights, iframe)
					, frame, cx
					, view.min_extent
					, view.max_extent
					, anaglyph
					, coloranaglyph);
				(*p)->glRender( local );
			
				view.absorb_local( local );
			} 
			else
				(*p)->glRender( view );
			
			// Restore the object's color.
			(*p)->color = actual_color;
		
			if (view.min_extent[0] < min_ext.x) 
				min_ext.x = view.min_extent[0];
			if (view.min_extent[1] < min_ext.y) 
				min_ext.y = view.min_extent[1];
			if (view.min_extent[2] < min_ext.z) 
				min_ext.z = view.min_extent[2];
			if (view.max_extent[0] > max_ext.x) 
				max_ext.x = view.max_extent[0];
			if (view.max_extent[1] > max_ext.y) 
				max_ext.y = view.max_extent[1];
			if (view.max_extent[2] > max_ext.z) 
				max_ext.z = view.max_extent[2];
		
			double dist = (*p)->rayIntersect( local_cam, local_ray);
			if (dist>0.0 && dist < best_intersect) {
				best_intersect = dist;
				pick = *p;
			}
			// nothread_sleep( 0.3);
		} // for
	} // destroy L
	
	glEnable( GL_BLEND);
	glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDepthMask( GL_FALSE);
	
	for (size_t i=0; i < view.sortlists.size(); ++i)
		glCallList( view.sortlists[i]);
	for (size_t i=0; i < view.sortlists.size(); ++i)
		glDeleteLists( view.sortlists[i],1);
	
	if (doPick){
		display->setRegion(min_ext, max_ext);
		mouseControl(mpos, cam, ray, pick, cam + ray*best_intersect);
	}
	glDisable(GL_BLEND);
	glDepthMask(GL_TRUE);
}

bool 
GLDevice::callback( GLDevice *dev ) 
{
	if (dev->active) {
		try {
			int next = dev->render_control();
			if ( next >= 0 ) {
				threaded_timer( next*1e-3, &GLDevice::callback, dev );
			} 
			else {
				dev->active = false;
			}
		} 
		catch (...) {
			std::cerr << "Caught an unexpected exception.\n";
			dev->active = false;
		}
	}
	
	return false;  // don't call me again
}

} // !namespace visual
