/*
 * CMat4D.h
 * $Id: CMat4D.h,v 1.4 2003/06/24 14:50:02 anxo Exp $
 *
 * Copyright (C) 1999, 2000 Michael Meissner, Markus Janich, Rainer Jaeger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Class CMat4D
//  Purpose     : Provides funcionality of a matrix


#ifndef __CMAT4D_H_
#define __CMAT4D_H_


// Qt
///////


// System
///////////
#if _MSC_VER >= 1200
#include <iostream>
#else
#include <iostream.h>
#endif
#include <math.h>
#include <string.h>


// Own
///////////
#include "CV4D.h"
#include "CQuat.h"
#include "CP4D.h"



/// Matrix class
/**
  * This class implements a matrix for homogenous coordinates
  * including functionality to matrix calculations.
  *
  * @author Markus Janich
  *

           |  0  4   8  12 |   row 0      | sx r  r  tx |
           |  1  5   9  13 |   row 1      | r  sy r  ty |
           |  2  6  10  14 |   row 2      | r  r  sz tz |
           |  3  7  11  15 |   row 3      | .  .  .  .  |

              c  c   c   c
              o  o   o   o
              l  l   l   l
              0  1   2   3            */

/** Note: CMat4D objects are handled in classic mathematical
 *  manner, i.e. to "transform a vector" multiply it like Tv -
 *  i.e. we have column vectors multiplied from the right side. [richi]
 */


class CMat4D {

public:
  /** Default Constructor. */
  CMat4D(void);

  /** Construct new matrix with the values passed by the vectors. The 
    * values of 'col1' are put in the first column, the values of
    * 'col2' are put to the second column and so on. */
  CMat4D(const CV4D&, const CV4D&, const CV4D&, const CV4D&);

  /** Construct new matrix with the values passed in '*field'.
    * The first four values of the array passed by 'field' are stored
    * in the first column of the matrix and so on. */
  CMat4D(const double*);

  /** Construct new matrix with values passed in [] order. */
  CMat4D(double, double, double, double, double, double, double, double,
	 double, double, double, double, double, double, double, double);

  /** Copy constructor.*/
  CMat4D(const CMat4D&);

  /** Construtor for identity matrix. */
  static CMat4D PIdentity(void);

  /** Construtor generating identity matrix with a translatation by a vector.
   * Please note, only x, y, and z of the vector are used. W is omitted. */
  static CMat4D PTranslate(const CV4D&);

  /** Construtor generating identity matrix with a translatation by a vector. */
  static CMat4D PTranslate(const CV3D&);

  /** Construtor generating identity matrix with a translatation by a vector. */
  static CMat4D PTranslate(double, double, double);

  /** Construtor generating identity matrix with scaling by a vector. 
   * Please note, only x, y, and z of the vector are used. W is omitted. */
  static CMat4D PScale(const CV4D&);

  /** Construtor generating identity matrix with scaling by a vector. */
  static CMat4D PScale(const CV3D&);

  /** Construtor generating identity matrix with scaling by a vector. */
  static CMat4D PScale(double, double, double);

  /** Construtor generating identity matrix with rotation around an axis by
   * a certain angle in rad. Please note that only x, y, and z of the vector are used,
   * w is omitted. */
  static CMat4D PRotate(const CV4D&, double);

  /** Construtor generating identity matrix with rotation around an axis by
   * a certain angle in rad. */
  static CMat4D PRotate(const CV3D&, double);

  /** Construtor generating identity matrix with rotation defined by the quaternion. */
  static CMat4D PRotate(CQuat&);

  /** Default Destructor. */
  ~CMat4D(void);



  //////////////////////////
  // OVERLOADED OPERATORS //
  //////////////////////////

  /** Assign one matrix to another.*/
  const CMat4D& operator=(const CMat4D&);

  /** Set the diagonal to the passed value. */
  const CMat4D& operator=(const double a);

  /** Add matrix to this matrix */
  CMat4D& operator+=(const CMat4D &m);

  /* Substract matrix from this matrix */
  CMat4D& operator-=(const CMat4D &m);

  /** Multiplication of a matrix with this matrix. */
  CMat4D& operator*=(const CMat4D&);


  /** Addition of two matrices. */
  CMat4D operator+(const CMat4D&) const;

  /** Subtraction of two matrices. */
  CMat4D operator-(const CMat4D&) const;

  /** Multiplication of a matrix (this) with another matrix. */
  CMat4D operator*(const CMat4D &m) const;

  /** Multiplication of a matrix with a scalar. */
  CMat4D operator*(double scalar) const;

  /** boolean operator ( EQUAL ) */
  bool operator == (const CMat4D &m) const;

  /** boolean operator ( NOT EQUAL ) */
  bool operator != (const CMat4D &m) const;


  /** Returns the element in the i-th row and j-th coloumn. */
  double operator()(int i,int j) const;

  /** Returns the i-th row of the matrix. */
  CV4D operator[](int i) const;

  /** Returns the j-th column of the matrix. */
  CV4D operator()(int j) const;


  /** Multiplication of a matrix with a vector. */
  friend CV4D operator*(const CMat4D&, const CV4D&);

  /** Multiplication of a matrix with a point. */
  friend CP4D operator*(const CMat4D&, const CP4D&);

  /** Multiplication of a matrix with a scalar. */
  friend CMat4D operator*(double, const CMat4D&);

  /** Multiplication of a scalar with a matrix. */
  friend CMat4D operator*(const CMat4D& M, double rdFactor) { return rdFactor*M; };
 


  /////////////
  // METHODS //
  /////////////
 
  /** Returns the pointer to the array of the matrix. */
  const double *getArray() { return m_ardValues; };

  /** Initialize the matrix with 0. */
  void clear(void);

  /** Set the coefficient of the matrix to the values given in *field.
    * The first four values of the array passed by 'field' are stored
    * in the first column of the matrix and so on. */
  void setValues(double* field);
 
  /** Returns the element in the i-th row and j-th coloumn. */
  double getCoeff(int i,int j) const;

  /** Sets the element in the i-th row and j-th coloumn. */
  void setCoeff(int i,int j, double value);
 
  /** Tranpose the matrix. */
  CMat4D getTransposed(void) const;

  // Inverted
  CMat4D getInverted(void) const;

  /** invert this matrix */
  bool invert(void);

  /** transpose this matrix */
  void transpose(void);

  /** Print the matrix to standard output. */
  void print(void);

  /** Print the matrix to output stream. */
  friend ::ostream& operator << (::ostream& s, const CMat4D &m );

  /** Set matrix to identity. */
  void setIdentity( void );

  /** Returns the row of the matrix (0 through 3). */
  CV4D getRow(int);

  /** Returns the column of the matrix (0 through 3). */
  CV4D getCol(int);

  /** Sets the specified row of the matrix with the given vector. */
  void setRow(int nRow, const CV4D&);
 
  /** Sets the specified column of the matrix with the given vector. */
  void setCol(int nCol, const CV4D&);
                                                                                                                      /** Sets all four rows of the matrix with the given vectors. */
  void setRows(const CV4D&, const CV4D&, const CV4D&, const CV4D&);

  /** Sets all four columns of the matrix with the given vectors. */
  void setCols(const CV4D&, const CV4D&, const CV4D&, const CV4D&);

  /** Sets the scaling of the matrix (diagonal elements). Note Before 
   * this, the matrix is cleard and afterwards, the fourth diagonal 
   * element is set to 1!!! */
  void setScaling(const CV4D&);

  /** Sets the scaling of the matrix (diagonal elements). Note Before 
   * this, the matrix is cleard and afterwards, the fourth diagonal 
   * element is set to 1!!! */
  void setScaling(const CV3D&);

  /** Sets the scaling of the matrix (diagonal elements). Note Before 
   * this, the matrix is cleard and afterwards, the fourth diagonal 
   * element is set to 1!!! */
  void setScaling(double, double, double);

  /** Sets the scaling of the matrix (diagonal elements). Note Before 
   * this, the matrix is cleard and afterwards, the fourth element in
   * the last row is set to 1!!! */
  void setTranslation(const CV4D&);

  /** Sets the scaling of the matrix (diagonal elements). Note Before 
   * this, the matrix is cleard and afterwards, the fourth element in
   * the last row is set to 1!!! */
  void setTranslation(const CV3D&);

  /** Sets the scaling of the matrix (diagonal elements). Note Before 
   * this, the matrix is cleard and afterwards, the fourth element in
   * the last row is set to 1!!! */
  void setTranslation(double, double, double);

  /** Rotates around an axis by a certain angle. */
  void setRotation(CV4D&, double);

  /** Rotates around an axis by a certain angle. */
  void setRotation(CV3D&, double);

  /** Rotates using the given quaternion. */
  void setRotation(CQuat&);



 private:
  /** Returns a one dimensional array of the matrix elements. */
  double*  get1DField(void);

  /** Returns a two dimensional array of the matrix elements. */
  double** get2DField(void);

  double m_ardValues[16];  // In the array the values of the matrix are saved by columns.
};

#endif // __CMAT4D_H_
