/*
 * CBoundingBox3D.cpp
 *
 * Copyright (C) 1999, 2000 Rainer Jaeger, Michael Meissner
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */


/** documentation stuff

  @author Rainer Jaeger, Michael Meissner
    
  @version 0.0 //see cvs docu
          
*/



// Qt
///////

// System
///////////
#if _MSC_VER >= 1300
#include <iostream>
#else
#include <iostream.h>
#endif
#include <math.h>

// Own
///////////
#include "CBoundingBox3D.h"



// Initialisation of statics:
double CBoundingBox3D::epsilon= 1e-6; //EPS_CBoundingBox3D

//////////////////
// CONSTRUCTORS //
//////////////////

CBoundingBox3D::CBoundingBox3D(double rdXmin, double rdYmin, double rdZmin,
                           double rdXmax, double rdYmax, double rdZmax)
/*********************************************************************/
{
  CommonConstructor(CP3D(rdXmin, rdYmin, rdZmin), 
                    CP3D(rdXmax, rdYmax, rdZmax));
  return;
}


CBoundingBox3D::CBoundingBox3D(const CP4D &minVec, const CP4D &maxVec)
/*********************************************************************/
{
  CommonConstructor(CP3D(minVec[0], minVec[1], minVec[2]),
                    CP3D(maxVec[0], maxVec[1], maxVec[2]));
  return;
}


CBoundingBox3D::CBoundingBox3D(const CP3D &minPnt, const CP3D &maxPnt)
/*********************************************************************/
{
  CommonConstructor(CP3D(minPnt[0], minPnt[1], minPnt[2]),
                    CP3D(maxPnt[0], maxPnt[1], maxPnt[2]));
  return;
}


CBoundingBox3D::CBoundingBox3D(const CBoundingBox3D &bbox)
/*********************************************************************/
{
  m_LowerLeft  = bbox.getLowerLeft();
  m_UpperRight = bbox.getUpperRight();
  return;
}


/////////////////
// DESTRUCTORS //
/////////////////
CBoundingBox3D::~CBoundingBox3D()
{
  // Empty
}

  
//////////////////////
// MEMBER FUNCTIONS //
//////////////////////


// Function  : CommonConstructor
// Parameters: CP3D Point1, CP3D Point2
// Purpose   :
// Comments  :
void CBoundingBox3D::CommonConstructor(CP3D Point1, CP3D Point2)
/*********************************************************************/
{
  m_LowerLeft  = Min(Point1, Point2); // returns min of all components!
  m_UpperRight = Max(Point1, Point2); // returns max of all components!
  return;
}



// Function  : operator+
// Parameters:
// Purpose   :
// Comments  :
CBoundingBox3D CBoundingBox3D::operator+(const CBoundingBox3D& bbox) const
/*********************************************************************/
{
  return CBoundingBox3D(Min(m_LowerLeft, bbox.getLowerLeft()),
                      Max(m_UpperRight, bbox.getUpperRight()));
}



// Function  : operator+=
// Parameters:
// Purpose   :
// Comments  :
CBoundingBox3D& CBoundingBox3D::operator+=(const CBoundingBox3D& bbox)
/*********************************************************************/
{
  m_LowerLeft  = Min(m_LowerLeft, bbox.getLowerLeft());
  m_UpperRight = Max(m_UpperRight, bbox.getUpperRight());

  return *this;
}



// Function  : operator=
// Parameters:
// Purpose   :
// Comments  :
const CBoundingBox3D& CBoundingBox3D::operator=(const CBoundingBox3D& bbox)
/*********************************************************************/
{
  m_LowerLeft  = bbox.getLowerLeft();
  m_UpperRight = bbox.getUpperRight();

  return *this;
}



// Function  : setBBox
// Parameters: CBoundingBox3D &bbox
// Purpose   :
// Comments  :
void CBoundingBox3D::setBBox(CBoundingBox3D &bbox)
/*********************************************************************/
{
  m_LowerLeft  = bbox.getLowerLeft();
  m_UpperRight = bbox.getUpperRight();
  
  return;
}



// Function  : addPoint
// Parameters: CP3D& point
// Purpose   :
// Comments  :
void CBoundingBox3D::addPoint(const CP3D &point)
{
   if (point.getX() < m_LowerLeft.getX())
      m_LowerLeft.setX(point.getX());
   if (point.getY() < m_LowerLeft.getY())
      m_LowerLeft.setY(point.getY());
   if (point.getZ() < m_LowerLeft.getZ())
      m_LowerLeft.setZ(point.getZ());

   if (point.getX() > m_UpperRight.getX())
      m_UpperRight.setX(point.getX());
   if (point.getY() > m_UpperRight.getY())
      m_UpperRight.setY(point.getY());
   if (point.getZ() > m_UpperRight.getZ())
      m_UpperRight.setZ(point.getZ());
}



// Function  : addPoint
// Parameters: CP4D& point
// Purpose   :
// Comments  :
void CBoundingBox3D::addPoint(const CP4D &point)
{
   addPoint((CP3D)point);
}


void CBoundingBox3D::move(const CV3D&v)
{
   m_LowerLeft += v;
   m_UpperRight += v;
}

void CBoundingBox3D::scale(const CV3D&v)
{
   m_LowerLeft.setX(m_LowerLeft.getX() * v.getX());
   m_LowerLeft.setY(m_LowerLeft.getY() * v.getY());
   m_LowerLeft.setZ(m_LowerLeft.getZ() * v.getZ());
   m_UpperRight.setX(m_UpperRight.getX() * v.getX());
   m_UpperRight.setY(m_UpperRight.getY() * v.getY());
   m_UpperRight.setZ(m_UpperRight.getZ() * v.getZ());
}


// Function  : getMinSize
// Parameters: void
// Purpose   :
// Comments  :
double CBoundingBox3D::getMinSize(void) const
/*********************************************************************/
{
  double rdX = fabs(m_UpperRight[0] - m_LowerLeft[0]);
  double rdY = fabs(m_UpperRight[1] - m_LowerLeft[1]);
  double rdZ = fabs(m_UpperRight[2] - m_LowerLeft[2]);

  return (rdX < rdY) ? ((rdX < rdZ) ? rdX : rdZ) : ((rdY < rdZ) ? rdY : rdZ);
}



// Function  : getMaxSize
// Parameters: void
// Purpose   :
// Comments  :
double CBoundingBox3D::getMaxSize(void) const
/*********************************************************************/
{
  double rdX = fabs(m_UpperRight[0] - m_LowerLeft[0]);
  double rdY = fabs(m_UpperRight[1] - m_LowerLeft[1]);
  double rdZ = fabs(m_UpperRight[2] - m_LowerLeft[2]);

  return (rdX > rdY) ? ((rdX > rdZ) ? rdX : rdZ) : ((rdY > rdZ) ? rdY : rdZ);
}



// Function  : getSize
// Parameters: int nDimension
// Purpose   :
// Comments  :
double CBoundingBox3D::getSize(int nDimension) const
/*********************************************************************/
{
  return fabs(m_UpperRight[nDimension] - m_LowerLeft[nDimension]);
}



// Function  : getDiagonal
// Parameters: void
// Purpose   :
// Comments  :
double CBoundingBox3D::getDiagonal(void) const
/*********************************************************************/
{ 
  return sqrt(getSize(0)*getSize(0) + getSize(1)*getSize(1) 
                                    + getSize(2)*getSize(2));
}



// Function  : getVolume
// Parameters: void
// Purpose   :
// Comments  :
double CBoundingBox3D::getVolume(void) const
/*********************************************************************/
{ 
  return getSize(0)*getSize(1)*getSize(2);
}



// Function  : getCornerVertex
// Parameters: void
// Purpose   :
// Comments  :
CP3D CBoundingBox3D::getCornerVertex(int i) const
/*********************************************************************/
{
  switch (i) {
     case 0: return CP3D(m_LowerLeft[0],  m_LowerLeft[1],  m_LowerLeft[2]);
     case 1: return CP3D(m_UpperRight[0], m_LowerLeft[1],  m_LowerLeft[2]);
     case 2: return CP3D(m_LowerLeft[0],  m_UpperRight[1], m_LowerLeft[2]);
     case 3: return CP3D(m_UpperRight[0], m_UpperRight[1], m_LowerLeft[2]);
     case 4: return CP3D(m_LowerLeft[0],  m_LowerLeft[1],  m_UpperRight[2]);
     case 5: return CP3D(m_UpperRight[0], m_LowerLeft[1],  m_UpperRight[2]);
     case 6: return CP3D(m_LowerLeft[0],  m_UpperRight[1], m_UpperRight[2]);
     case 7: return CP3D(m_UpperRight[0], m_UpperRight[1], m_UpperRight[2]);
     default: cerr << "Only indices from 0 to 7 allowed!" << endl;
  }
  return CP3D(0,0,0);
}



// Function  : getCenter
// Parameters: void
// Purpose   :
// Comments  :
const CP3D CBoundingBox3D::getCenter(void) const
/*********************************************************************/
{ 
  return MidPoint(m_LowerLeft, m_UpperRight);
}



// Function  : isInside
// Parameters: const CP3D& Point
// Purpose   :
// Comments  :
bool CBoundingBox3D::isInside(const CP3D& Point) const
/*********************************************************************/
{
  return((Point[0] >= m_LowerLeft[0]) && (Point[0] <= m_UpperRight[0])
	    && (Point[1] >= m_LowerLeft[1]) && (Point[1] <= m_UpperRight[1])
	    && (Point[2] >= m_LowerLeft[2]) && (Point[2] <= m_UpperRight[2]));
}



// Function  : isInside
// Parameters: const CP4D& Point
// Purpose   :
// Comments  :
bool CBoundingBox3D::isInside(const CP4D& Point) const
/*********************************************************************/
{
  return((Point[0] >= m_LowerLeft[0]) && (Point[0] <= m_UpperRight[0])
            && (Point[1] >= m_LowerLeft[1]) && (Point[1] <= m_UpperRight[1])
            && (Point[2] >= m_LowerLeft[2]) && (Point[2] <= m_UpperRight[2]));
}



// Function  : print
// Parameters: void
// Purpose   :
// Comments  :
void CBoundingBox3D::print(void) const
/*********************************************************************/
{
  cerr << "\nCBoundingBox3D:" << endl;
  cerr << "------------" << endl;
  cerr << "lower left: " << m_LowerLeft << endl;
  cerr << "upper right: " << m_UpperRight << endl;
  cerr << "------------" << endl;

  return;
}



// Function  : operator<<
// Parameters: ostream& s, const CP3D& v
// Purpose   :
// Comments  :
ostream& operator<<(ostream& s, const CBoundingBox3D& bbox)
/*********************************************************************/
{
  return s << "(" << bbox.getLowerLeft() << ", " << bbox.getUpperRight() << ")";
}



// Function  : operator>>
// Parameters: void
// Purpose   :
// Comments  :
istream& operator>>(istream& s, CBoundingBox3D& bbox)
/*********************************************************************/
{
  return s; // STILL TO BE DONE OR TP BE FIXED :-)
}
