// infobar.C
//
// This program is free software. See the file COPYING for details.
// Author: Mattias Engdegrd, 1997-1999

#include <stdio.h>
#include <qframe.h>
#include <qbrush.h>
#include <qdrawutil.h>
#include <qbitmap.h>
#include "infobar.h"
#include "proc.h"
#include "qps.h"


Infobar::Infobar(QWidget *parent)
       : QWidget(parent)
{
    is_vertical = Qps::vertical_cpu_bar;
    QFont f = font();
    f.setBold(FALSE);

    load_graph = new LoadGraph(this, 70);
    load_graph->setFrameStyle(QFrame::Panel | QFrame::Sunken);

    load_lbl = new ClickLabel(this);
    load_lbl->setFrameStyle(QFrame::Box | QFrame::Sunken);
    load_lbl->setAlignment(AlignRight | AlignVCenter);
    load_lbl->setFont(f);
    load_lbl->hide();

    connect(load_graph, SIGNAL(clicked()), SLOT(toggle_load()));
    connect(load_lbl, SIGNAL(clicked()), SLOT(toggle_load()));

    cpu_bar = new BarGraph(this, "cpu", Procinfo::CPUTIMES,
			   Procinfo::num_cpus);
    int i = 0;
    cpu_bar->setColor(i, Qps::color_set[Qps::COLOR_CPU_USER]);
    cpu_bar->setLabel(i++, "user");
#ifdef LINUX
    cpu_bar->setColor(i, Qps::color_set[Qps::COLOR_CPU_NICE]);
    cpu_bar->setLabel(i++, "nice");
#endif
    cpu_bar->setColor(i, Qps::color_set[Qps::COLOR_CPU_SYS]);
    cpu_bar->setLabel(i++, "sys");
#ifdef SOLARIS
    cpu_bar->setColor(i, Qps::color_set[Qps::COLOR_CPU_WAIT]);
    cpu_bar->setLabel(i++, "wait");
#endif
    cpu_bar->setColor(i, Qps::color_set[Qps::COLOR_CPU_IDLE]);
    cpu_bar->setLabel(i, "idle");
    cpu_bar->setFrameStyle(QFrame::Box | QFrame::Sunken);
    cpu_bar->setFont(f);

    cpu_lbl = new CpuLabel(this, Procinfo::num_cpus);
    cpu_lbl->setFrameStyle(QFrame::Box | QFrame::Sunken);
    cpu_lbl->setFont(f);

    connect(cpu_bar, SIGNAL(clicked()), SLOT(toggle_cpu()));
    connect(cpu_lbl, SIGNAL(clicked()), SLOT(toggle_cpu()));

#ifdef LINUX
    int nmem = 4;
#endif
#ifdef SOLARIS
    int nmem = 2;
#endif
    i = 0;
    mem_bar = new BarGraph(this, "mem", nmem);
    mem_bar->setColor(i, Qps::color_set[Qps::COLOR_MEM_USED]);
    mem_bar->setLabel(i++, "used");
#ifdef LINUX
    mem_bar->setColor(i, Qps::color_set[Qps::COLOR_MEM_BUFF]);
    mem_bar->setLabel(i++, "buff");
    mem_bar->setColor(i, Qps::color_set[Qps::COLOR_MEM_CACHE]);
    mem_bar->setLabel(i++, "cach");
#endif
    mem_bar->setColor(i, Qps::color_set[Qps::COLOR_MEM_FREE]);
    mem_bar->setLabel(i, "free");
    mem_bar->setFrameStyle(QFrame::Box | QFrame::Sunken);
    mem_bar->setFont(f);

    mem_lbl = new ClickLabel(this);
    mem_lbl->setFrameStyle(QFrame::Box | QFrame::Sunken);
    mem_lbl->setFont(f);
    mem_lbl->setAlignment(AlignLeft | AlignTop);
    mem_lbl->setMargin(0);

    connect(mem_bar, SIGNAL(clicked()), SLOT(toggle_mem()));
    connect(mem_lbl, SIGNAL(clicked()), SLOT(toggle_mem()));

    swap_bar = new BarGraph(this, "swap", 2, 1, TRUE);
    swap_bar->setColor(0, Qps::color_set[Qps::COLOR_SWAP_USED]);
    swap_bar->setColor(1, Qps::color_set[Qps::COLOR_SWAP_FREE]);
    swap_bar->setFrameStyle(QFrame::Box | QFrame::Sunken);
    swap_bar->setFont(f);

    swap_lbl = new ClickLabel(this);
    swap_lbl->setFrameStyle(QFrame::Box | QFrame::Sunken);
    swap_lbl->setAlignment(AlignLeft | AlignVCenter);
    swap_lbl->setFont(f);

    connect(swap_bar, SIGNAL(clicked()), SLOT(toggle_swap()));
    connect(swap_lbl, SIGNAL(clicked()), SLOT(toggle_swap()));

    up_lbl = new QLabel(this);
    up_lbl->setFrameStyle(QFrame::Box | QFrame::Sunken);
    up_lbl->setAlignment(AlignLeft | AlignVCenter | WordBreak);
    up_lbl->setFont(f);

    configure_geometry();
}

void Infobar::configure_geometry()
{
    resize(width(), is_vertical ? 100 : 56);
}

int rightOf(QWidget *w)
{
    return w->x() + w->width();
}

int below(QWidget *w)
{
    return w->y() + w->height();
}

void Infobar::resizeEvent(QResizeEvent *)
{
    // the Qt layout classes didn't allow the desired behaviour so this
    // is done by hand

    cpu_bar->setOrientation(is_vertical);
    cpu_lbl->setOrientation(is_vertical);
    mem_bar->setOrientation(is_vertical);
    swap_bar->setOrientation(is_vertical);

    int h = height() - 6;
    load_graph->setGeometry(2, 2, 70, h);
    load_lbl->setGeometry(load_graph->geometry());
    cpu_bar->setGeometry(rightOf(load_graph) + 2, 2,
			 is_vertical ? cpu_bar->width() : 200, h);
    cpu_lbl->setGeometry(cpu_bar->geometry());
    mem_bar->setGeometry(rightOf(cpu_bar) + 2, 2,
			 is_vertical ? mem_bar->width() : 200, h);
    mem_lbl->setGeometry(mem_bar->geometry());
    swap_bar->setGeometry(rightOf(mem_bar) + 2, 2,
			  is_vertical ? swap_bar->width() : 150,
			  is_vertical ? h : 24);
    swap_lbl->setGeometry(swap_bar->geometry());

    if(is_vertical)
	up_lbl->setGeometry(rightOf(swap_bar) + 2, 2, 64, height() - 6);
    else
	up_lbl->setGeometry(swap_bar->x(), below(swap_bar) + 2,
			    swap_bar->width(), swap_bar->height());
}

void Infobar::mem_string(int kbytes, char *buf)
{
    if(kbytes >= 10000) {
	int meg = kbytes >> 10;
	if(meg >= 10000)
	    sprintf(buf, "%uG", meg >> 10);
	else
	    sprintf(buf, "%uM", meg);
    } else
	sprintf(buf, "%uK", kbytes);
}

// return true if the swap meter is redlined
bool Infobar::swaplim_exceeded()
{
    if(Qps::swaplim_percent) {
	if(Procinfo::swap_total > 0) {
	    int free_p = 100 * Procinfo::swap_free / Procinfo::swap_total;
	    return free_p < Qps::swaplimit;
	} else
	    return FALSE;
    } else {
	return Procinfo::swap_free < Qps::swaplimit;
    }
}

// refresh status bar; if step_load is true, add a point to the load graph
void Infobar::refresh(bool step_load)
{
    QString s;

    if(step_load)
	load_graph->add_load_point();
    if(Qps::show_load_graph) {
	load_graph->repaint();
    } else {
	s.sprintf("%1.02f\nload  %1.02f\n%1.02f", Procinfo::loadavg[0],
		  Procinfo::loadavg[1], Procinfo::loadavg[2]);
	load_lbl->setText(s);
    }

    int usedram = Procinfo::mem_total - Procinfo::mem_free 
#ifdef LINUX
                  - Procinfo::mem_buffers - Procinfo::mem_cached
#endif
	          ;
    if(Qps::show_mem_bar) {
	int i = 0;
	mem_bar->setValue(0, i++, usedram);
#ifdef LINUX
	mem_bar->setValue(0, i++, Procinfo::mem_buffers);
	mem_bar->setValue(0, i++, Procinfo::mem_cached);
#endif
	mem_bar->setValue(0, i, Procinfo::mem_free);
	mem_bar->refresh();
    } else {
	char used[10], tot[10], free[10];
	mem_string(usedram, used);
	mem_string(Procinfo::mem_total, tot);
#ifdef LINUX
	char buff[10], cache[10];
	mem_string(Procinfo::mem_buffers, buff);
	mem_string(Procinfo::mem_cached, cache);
#endif
	mem_string(Procinfo::mem_free, free);
	if(is_vertical)
#ifdef LINUX
	    s.sprintf("free %s\ncache %s\nbuff %s\nused %s\navail %s",
		      free, cache, buff, used, tot);
#endif
#ifdef SOLARIS
	    s.sprintf("free %s\nused %s\navail %s",
		      free, used, tot);
#endif
	else
#ifdef LINUX
	    s.sprintf("mem %s avail, %s used\n%s buff, %s cache\n%s free",
		      tot, used, buff, cache, free);
#endif
#ifdef SOLARIS
	    s.sprintf("mem %s avail, %s used\n%s free",
		      tot, used, free);
#endif
	mem_lbl->setText(s);
    }

    if(Qps::show_swap_bar) {
	swap_bar->setValue(0, 0, Procinfo::swap_total - Procinfo::swap_free);
	swap_bar->setValue(0, 1, Procinfo::swap_free);
	if(swaplim_exceeded())
	    swap_bar->setColor(0, Qps::color_set[Qps::COLOR_SWAP_WARN]);
	else
	    swap_bar->setColor(0, Qps::color_set[Qps::COLOR_SWAP_USED]);

	swap_bar->refresh();

    } else {
	char used[10], tot[10];
	mem_string(Procinfo::swap_total - Procinfo::swap_free, used);
	mem_string(Procinfo::swap_total, tot);
	s.sprintf(is_vertical ? "swap\nused\n%s\ntotal\n%s":"swap used %s/%s",
		  used, tot);
	swap_lbl->setText(s);
    }

    long u = time(NULL) - Procinfo::boot_time;
    int up_days = u / (3600 * 24);
    u %= (3600 * 24);
    int up_hrs = u / 3600;
    u %= 3600;
    int up_mins = u / 60;
    if(up_days == 0) {
	if(up_hrs == 0)
	    s.sprintf("up %d min", up_mins);
	else
	    s.sprintf("up %d:%02d", up_hrs, up_mins);
    } else
	s.sprintf("up %d day%s, %d:%02d",
		  up_days, (up_days == 1) ? "" : "s", up_hrs, up_mins);
    up_lbl->setText(s);

    for(unsigned cpu = 0; cpu < Procinfo::num_cpus; cpu++) {
	unsigned user, system, idle;
#ifdef LINUX
	unsigned nice;
#endif
#ifdef SOLARIS
	unsigned wait;
#endif
	if(Procinfo::num_cpus == Procinfo::old_num_cpus) {

#define TIMEDIFF(kind) (Procinfo::cpu_times(cpu, Procinfo::kind) - \
			Procinfo::old_cpu_times(cpu, Procinfo::kind))

            user = TIMEDIFF(CPUTIME_USER);
#ifdef LINUX
	    nice = TIMEDIFF(CPUTIME_NICE);
#endif
	    system = TIMEDIFF(CPUTIME_SYSTEM);
#ifdef SOLARIS
	    wait = TIMEDIFF(CPUTIME_WAIT);
#endif
	    idle = TIMEDIFF(CPUTIME_IDLE);

#undef TIMEDIFF
	} else {
	    // number of cpus just changed
	    // FIXME: this is not enough to support dynamic addition/removal
	    // of cpus at runtime; the number of graphs should be dynamic
	    // as well.
	    user = Procinfo::cpu_times(cpu, Procinfo::CPUTIME_USER);
#ifdef LINUX
	    nice = Procinfo::cpu_times(cpu, Procinfo::CPUTIME_NICE);
#endif
	    system = Procinfo::cpu_times(cpu, Procinfo::CPUTIME_SYSTEM);
#ifdef SOLARIS
	    wait = Procinfo::cpu_times(cpu, Procinfo::CPUTIME_WAIT);
#endif
	    idle = Procinfo::cpu_times(cpu, Procinfo::CPUTIME_IDLE);
	}

	if(Qps::show_cpu_bar) {
	    int i = 0;
	    cpu_bar->setValue(cpu, i++, user);
#ifdef LINUX
	    cpu_bar->setValue(cpu, i++, nice);
#endif
	    cpu_bar->setValue(cpu, i++, system);
#ifdef SOLARIS
	    cpu_bar->setValue(cpu, i++, wait);
#endif
	    cpu_bar->setValue(cpu, i, idle);
	    cpu_bar->refresh();
	} else {
	    int i = 0;
	    cpu_lbl->setValue(cpu, i++, user);
#ifdef LINUX
	    cpu_lbl->setValue(cpu, i++, nice);
#endif
	    cpu_lbl->setValue(cpu, i++, system);
#ifdef SOLARIS
	    cpu_lbl->setValue(cpu, i++, wait);
#endif
	    cpu_lbl->setValue(cpu, i, idle);
	    cpu_lbl->repaint();
	}
    }
}

void Infobar::update_load()
{
    load_graph->add_load_point();
}

// if p, then show a and hide b; otherwise do the opposite
void Infobar::show_and_hide(bool p, QWidget *a, QWidget *b)
{
    if(!p) {
        QWidget *c = a;
        a = b;
        b = c;
    }
    if(!a->isVisible()) {
        b->hide();
        a->show();
    }
}

// show/hide components, arrange in vertical/horizontal style
void Infobar::configure()
{
    if(is_vertical != Qps::vertical_cpu_bar) {
	is_vertical = Qps::vertical_cpu_bar;
	configure_geometry();
    }
    show_and_hide(Qps::show_load_graph, load_graph, load_lbl);
    show_and_hide(Qps::show_cpu_bar, cpu_bar, cpu_lbl);
    show_and_hide(Qps::show_mem_bar, mem_bar, mem_lbl);
    show_and_hide(Qps::show_swap_bar, swap_bar, swap_lbl);
    refresh(FALSE);
}

void Infobar::toggle_load()
{
    Qps::show_load_graph = !Qps::show_load_graph;
    emit config_change();
    configure();
}

void Infobar::toggle_cpu()
{
    Qps::show_cpu_bar = !Qps::show_cpu_bar;
    emit config_change();
    configure();
}

void Infobar::toggle_mem()
{
    Qps::show_mem_bar = !Qps::show_mem_bar;
    emit config_change();
    configure();
}

void Infobar::toggle_swap()
{
    Qps::show_swap_bar = !Qps::show_swap_bar;
    emit config_change();
    configure();
}

CpuLabel::CpuLabel(QWidget *parent, int cpus)
    : QFrame(parent),
      ncpus(cpus), vert(false)
{
    vals = new unsigned long[cpus * 4];
}

void CpuLabel::setOrientation(bool vertical)
{
    vert = vertical;
}

void CpuLabel::drawContents(QPainter *p)
{
    QString s;
    QRect cr = contentsRect();
    cr.moveBy(3, 2);
    cr.setWidth(cr.width() - 6); cr.setHeight(cr.height() - 4);
    if(vert) {
	p->drawText(cr.x(), cr.y(), cr.width(), cr.height(),
		    AlignLeft | AlignTop,
#ifdef LINUX
		    "user\nnice\nsys\nidle"
#endif
#ifdef SOLARIS
		    "user\nsys\nwait\nidle"
#endif
		    );
	for(int cpu = 0; cpu < ncpus; cpu++) {
	    int i = 0;
	    unsigned long user = values(cpu, i++);
#ifdef LINUX
	    unsigned long nice = values(cpu, i++);
#endif
	    unsigned long system = values(cpu, i++);
#ifdef SOLARIS
	    unsigned long wait = values(cpu, i++);
#endif
	    unsigned long idle = values(cpu, i);
	    float tot = (user
#ifdef LINUX
			 + nice
#endif
			 + system
#ifdef SOLARIS
			 + wait
#endif
			 + idle) * 0.01;
	    if(tot == 0)
		tot = 1;
	    s.sprintf("%.1f%%\n%.1f%%\n%.1f%%\n%.1f%%\ncpu%d",
#ifdef LINUX
		      user / tot, nice / tot, system / tot,
#endif
#ifdef SOLARIS
	              user / tot, system / tot, wait / tot,
#endif
	              idle / tot, cpu);
	    p->drawText(cr.x(), cr.y(),
			cr.width() - (ncpus - 1 - cpu) * 36, cr.height(),
			AlignRight | AlignTop, s);
	}
    } else {
	unsigned long user = 0;
#ifdef LINUX
	unsigned long nice = 0;
#endif
	unsigned long system = 0;
#ifdef SOLARIS
	unsigned long wait = 0;
#endif
	unsigned long idle = 0;
	for(int cpu = 0; cpu < ncpus; cpu++) {
	    int i = 0;
	    user += values(cpu, i++);
#ifdef LINUX
	    nice += values(cpu, i++);
#endif
	    system += values(cpu, i++);
#ifdef SOLARIS
	    wait += values(cpu, i++);
#endif
	    idle += values(cpu, i);
	}
	float tot = (user
#ifdef LINUX
		     + nice
#endif
		     + system
#ifdef SOLARIS
		     + wait
#endif
		     + idle) / 100.0;
#ifdef LINUX
	s.sprintf("cpu %.1f%% user, %.1f%% nice\n"
		  "%.1f%% system, %.1f%% idle",
		  user / tot, nice / tot, system / tot, idle / tot);
#endif
#ifdef SOLARIS
	s.sprintf("cpu %.1f%% user, %.1f%% system\n"
		  "%.1f%% wait, %.1f%% idle",
		  user / tot, system / tot, wait / tot, idle / tot);
#endif
	p->drawText(cr.x(), cr.y(), cr.width(), cr.height(),
		    AlignLeft | AlignTop, s);
    }
}

void CpuLabel::mousePressEvent(QMouseEvent *)
{
    emit clicked();
}

BarGraph::BarGraph(QWidget *parent, const char *title, int divs, int bars,
		   bool nolegend)
         : QFrame(parent),
	   nbars(bars), ndivs(divs), caption(title), no_legend(nolegend),
	   vert(false)
{
    vals = new unsigned long[divs * bars];
    for(int j = 0; j < bars; j++)
	for(int i = 0; i < divs; i++)
	    values(j, i) = 0;
    colors = new QColor[divs];
    names = new QString[divs];
    setBarSize();
}

BarGraph::~BarGraph()
{
    delete[] vals;
    delete[] colors;
    delete[] names;
}

void BarGraph::setOrientation(bool vertical)
{
    vert = vertical;
    setUpdatesEnabled(FALSE);	// avoid repaint in possible resize
    if(vert) {
	int w = (no_legend ? 0 : label_width) + nbars * 36 + 4;
	setFixedWidth(w);
    } else {
	setMinimumWidth(64);
	setMaximumWidth(32767);
    }
    setUpdatesEnabled(TRUE);
}

void BarGraph::setBarSize()
{
    if(vert) {
	bar_size = height() - 12 - (no_legend ? title_height : 0);
    } else {
	bar_size = width() - 12 - title_width;
    }
}

void BarGraph::resizeEvent(QResizeEvent *)
{
    setBarSize();
    repaint();
}

void BarGraph::mousePressEvent(QMouseEvent *)
{
    emit clicked();
}

void BarGraph::refresh()
{
    QPainter p(this);
    refreshBar(&p);
}

void BarGraph::refreshBar(QPainter *p)
{
    if(vert) {
	int x;
	if(no_legend)
	    x = (width() - bar_thickness) / 2;
	else
	    x = 12 + label_width;
	for(int i = 0; i < nbars; i++)
	    drawBar(p, x + i * 36, 4,
		    bar_thickness, bar_size, i);
    } else
	drawBar(p, 4 + title_width + 4, 4, bar_size, bar_thickness, -1);
}

void BarGraph::drawContents(QPainter *p)
{
    if(vert)
	p->drawText(4, height() - title_height - 4,
		    title_width, title_height,
		    AlignLeft | AlignVCenter, caption);
    else
	p->drawText(4, 4, title_width, title_height,
		    AlignLeft | AlignVCenter, caption);
    refreshBar(p);
    if(!no_legend)
	for(int i = 0; i < ndivs; i++) {
	    if(vert)
		drawLegend(p, 4, 4 + label_height * i,
			   label_width, label_height, ndivs - 1 - i);
	    else
		drawLegend(p, 4 + label_width * i, 28,
			   label_width, label_height, i);
	}
}

void BarGraph::drawBar(QPainter *p, int x, int y, int w, int h, int bar)
{
    unsigned long *v = new unsigned long[ndivs];
    int downscale = 0;
    for(int i = 0; i < ndivs; i++) {
	if(bar == -1) {
	    v[i] = 0;
	    for(int j = 0; j < nbars; j++)
		v[i] += values(j, i) / nbars;
	} else
	    v[i] = values(bar, i);
	if(v[i] > 1000000)
	    downscale = 10;	// fix possible overflow
    }
    unsigned long sum = 0;
    for(int i = 0; i < ndivs; i++)
	sum += v[i] >>= downscale;
    int d = 0;

    QColorGroup g = colorGroup();
    QBrush fill(colors[ndivs - 1]);
    qDrawShadePanel(p, x, y, w, h, g, TRUE, 1, &fill);
    w -= 2;
    h -= 2;
    x++;
    y++;

    if(sum > 0) {
	for(int i = 0; i < ndivs - 1; i++) {
	    int size;
	    if(vert) {
		size = h * v[i] / sum;
		// fill vertical bars from the bottom up
		p->fillRect(x, y + h - d - size, w, size, colors[i]);
	    } else {
		size = w * v[i] / sum;
		p->fillRect(x + d, y, size, h, colors[i]);
	    }
	    d += size;
	}
    } else {
	p->fillRect(x, y, w, h, colorGroup().background());
	p->drawText(x, y, w, h, AlignCenter, vert ? "n\no\nn\ne" : "(none)");
    }
    delete[] v;
}

void BarGraph::drawLegend(QPainter *p, int x, int y, int w, int h, int div)
{
    QColorGroup g = colorGroup();
    QBrush fill(colors[div]);
    qDrawShadePanel(p, x, y + 2, h - 4, h - 4, g, TRUE, 1, &fill);
    p->drawText(x + h, y, w - h, h, AlignLeft | AlignVCenter, names[div]);
}

LoadGraph::LoadGraph(QWidget *parent, int history_size)
         : QFrame(parent),
	   hist_size(history_size)
{
    npoints = 0;
    history = new unsigned[hist_size];
    peak = 0;
    h_index = 0;
    dirty = TRUE;

    setBackgroundColor(Qps::color_set[Qps::COLOR_LOAD_BG]);
}

LoadGraph::~LoadGraph()
{
    delete[] history;
}

// add value to the history, updating peak
void LoadGraph::add_history_point(unsigned value)
{
    history[h_index++] = value;
    if(h_index >= hist_size)
	h_index = 0;
    if(npoints < hist_size)
	npoints++;
    if(value > peak)
	peak = value;
    else {
	peak = 0;		// no negative values
	for(int i = 0; i < npoints; i++)
	    if(history[i] > peak)
		peak = history[i];
    }
}

// Add current load (the 1 min average) as a data point
void LoadGraph::add_load_point()
{
    add_history_point((unsigned)(Procinfo::loadavg[0] * history_scale));
    dirty = TRUE;
}

// draw the load graph on the internal pixmap, if needed
void LoadGraph::make_graph(int w, int h)
{
    if(w != pm.width() || h != pm.height()) {
	pm.resize(w, h);
	pm.setMask(QBitmap());	// remove the mask
    } else if(!dirty)
	return;

    pm.fill(Qps::color_set[Qps::COLOR_LOAD_BG]);
    int npts = QMIN(npoints, w);
    int ofs = h_index - npts;
    if(ofs < 0)
	ofs += hist_size;

    QPointArray pa(npts);
    unsigned scale = QMAX(peak, history_scale);
    float mul = (float)h / scale;
    for(int i = 0; i < npts; i++) {
	int j = ofs + i;
	if(j >= hist_size) j -= hist_size;
	pa[i] = QPoint(i, h - 1 - (int)(history[j] * mul));
    }
    QPainter p(&pm);
    if(scale > history_scale) {
	// draw scale lines
	p.setPen(Qps::color_set[Qps::COLOR_LOAD_LINES]);

	for(unsigned y = history_scale; y < scale; y += history_scale)
	    p.drawLine(0, h - 1 - (int)(mul * y),
		       w - 1, h - 1 - (int)(mul * y));
    }
    p.setPen(Qps::color_set[Qps::COLOR_LOAD_FG]);
    p.drawPolyline(pa);
    dirty = FALSE;
}

// return updated pixmap for use as an icon
QPixmap *LoadGraph::make_icon(int w, int h)
{
    make_graph(w, h);
    return &pm;
}

void LoadGraph::drawContents(QPainter *p)
{
    QRect cr = contentsRect();
    make_graph(cr.width(), cr.height());
    p->drawPixmap(cr.x(), cr.y(), pm);
}

void LoadGraph::mousePressEvent(QMouseEvent *)
{
    emit clicked();
}

void ClickLabel::mousePressEvent(QMouseEvent *)
{
    emit clicked();
}

