/*
 *      videoCapture.cpp -- Kapture
 *
 *      Copyright (C) 2006-2009
 *          Detlev Casanova (detlev.casanova@gmail.com)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 */
#define DEBUG
#include "videocapture.h"
#include "configparams.h"
#include "global.h"
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/select.h>

#include <QtGui>
#include <QApplication>
#include <QMainWindow>
#include <QImage>
#include <QPixmap>
#include <QLabel>
#include <QSize>



#include "logging.h"

#define CLIP(x) ( (x)>=0xFF ? 0xFF : ( (x) <= 0x00 ? 0x00 : (x) ) )


videoCapture::videoCapture()
{
	dev = 0;
	opened = false;
	allocated = false;
  localImage=NULL;
  numBuffers=4;

}

videoCapture::~videoCapture()
{
  if(localImage!=NULL) delete localImage;
	close();
}

void videoCapture::close()
{
  if(!opened) return;
	::close(dev);
	opened = false;
	allocated = false;
}

bool videoCapture::open()
{
	struct v4l2_capability cap;
	int ret;


  if (opened) return TRUE;
  addToLog("opening Videocapture device",DBCAM);

  dev = ::open(videoDevice.toLatin1().data(), O_RDWR);
	if (dev < 0) 
    {
     addToLog(QString("Error opening %1, %2").arg(videoDevice).arg(errno),DBCAM);
      return FALSE;
    }

  memset(&cap, 0, sizeof cap);
  ret = ioctl(dev, VIDIOC_QUERYCAP, &cap);
  if (ret < 0)
    {
      addToLog(QString("Error querying capabilities for %1, %2").arg(videoDevice).arg(errno),DBCAM);
      return FALSE;
    }
  dumpCaps(cap);

	if ((cap.capabilities & V4L2_CAP_VIDEO_CAPTURE) == 0) 
    {
      addToLog(QString("Error checking caps for %1").arg(videoDevice),DBCAM);
      return FALSE;
    }

  formatList=getFormatList(descripList);
  getFormat();
  setFormat(currentWidth(), currentHeight(), currentPixelFormat());
  sizeList=getSizesList();

  //Allocate buffers
  if (!allocated)
    {
      memset(&rb, 0, sizeof rb);
      rb.count = numBuffers;
      rb.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
      rb.memory = V4L2_MEMORY_MMAP;

        ret = ioctl(dev, VIDIOC_REQBUFS, &rb);
        if (ret < 0)
          {
            addToLog(QString("Unable to allocate buffersfor %1, %2").arg(videoDevice).arg(errno),DBCAM);
            return FALSE;
          }
        allocated = true;
     }




	opened = true;
  return TRUE;
}

QList<int> videoCapture::getFormatList(QList<QString> &description) const
{
	QList<int> formatList;
	int ret;
	struct v4l2_fmtdesc fmtList;
  addToLog("getFomatList()",DBCAM);
	memset(&fmtList, 0, sizeof fmtList);
	fmtList.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	int i = 0;

	do
	{
		fmtList.index = i;
		if ((ret = ioctl(dev, VIDIOC_ENUM_FMT, &fmtList)) < 0)
			break;
		else
		{
			formatList.append((int)fmtList.pixelformat);
			description.append((char*)fmtList.description);
		}
		i++;
	}
	while (ret != EINVAL);
	return formatList;
}

QList<QSize> videoCapture::getSizesList() const
{
	int i = 0;
	QList<QSize> rSizes;
	QSize tmp;
#ifdef V4L2_CAP_VIDEO_OUTPUT_OVERLAY // sort of test for v4l2 if this one does not exist, v4l2_frmsizeenum will not exist
	struct v4l2_frmsizeenum sizes;
  addToLog("getSizesList()",DBCAM);
	memset(&sizes, 0, sizeof sizes);
	sizes.pixel_format = currentPixelFormat();
	sizes.index = i;
	while(ioctl(dev, VIDIOC_ENUM_FRAMESIZES, &sizes) != -1)
	{
		tmp.setWidth((int)sizes.discrete.width);
		tmp.setHeight((int)sizes.discrete.height);
		rSizes.append(tmp);
		i++;
		sizes.index = i;
	}
#else
	tmp.setWidth(320);
	tmp.setHeight(240);
	rSizes.append(tmp);
#endif
	return rSizes;
}


bool videoCapture::setFormat(unsigned int width, unsigned int height, int pixelformat)
{
  addToLog("setFormat",DBCAM);
  memset(&fmt, 0, sizeof fmt);
  fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  fmt.fmt.pix.width = width;
  fmt.fmt.pix.height = height;
  fmt.fmt.pix.field = V4L2_FIELD_ANY;
  fmt.fmt.pix.pixelformat = pixelformat;
	if (ioctl(dev, VIDIOC_S_FMT, &fmt) < 0)
    {
      addToLog(QString("Error while setting format , %1").arg(strerror(errno)),DBCAM);
      return FALSE;
    }
  return TRUE;
}

bool videoCapture::getFormat()
{
  addToLog("getFormat",DBCAM);
  memset(&fmt, 0, sizeof fmt);
  fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  if (ioctl(dev, VIDIOC_G_FMT, &fmt) < 0)
    {
      addToLog(QString("Error while getting format , %1").arg(errno),DBCAM);
      return FALSE;
    }
  return TRUE;
}



bool videoCapture::getFrame()
{
	int ret = 0;
 // addToLog("getFrame",DBCAM);

	// Dequeue a buffer.
  ret = ioctl(dev, VIDIOC_DQBUF, &buf);
  //addToLog(QString("Dequeue buffer %1").arg(buf.index),DBCAM);
  if (ret < 0)
    {
      addToLog(QString("Unable to dequeue buffer , %1").arg(strerror(errno)),DBCAM);
      return FALSE;
  }
//  while(ioctl(dev, VIDIOC_DQBUF, &buf)<0)
//  {
//    qApp->processEvents();
//  }

	// Save the image.
	if (fmt.fmt.pix.pixelformat == V4L2_PIX_FMT_MJPEG)
	{
    addToLog(QString("Unsupported MJPEG Format"),DBCAM);
    return FALSE;
	}

	if (fmt.fmt.pix.pixelformat == V4L2_PIX_FMT_YUYV)
	{
    YUV422toRGB888_ITU_R(mem[buf.index],currentWidth(), currentHeight());

	}
	
	// Requeue the buffer.
	ret = ioctl(dev, VIDIOC_QBUF, &buf);
  //addToLog(QString("Requeue buffer %1").arg(buf.index),DBCAM);
	if (ret < 0) 
	{
    addToLog(QString("Unable to requeue buffer %1").arg(errno),DBCAM);
    return FALSE;
	}

  return TRUE;
}

/**
  Convert from YUV422 format to RGB888 using ITU_R_FLOAT. Formulae are described on http://en.wikipedia.org/wiki/YUV

  \param width width of image
  \param height height of image
  \param src source
  \param dst destination
*/
void videoCapture::YUV422toRGB888_ITU_R( unsigned char *src,int width, int height)
{
  int line, column;
  unsigned char *py, *pu, *pv;
  unsigned char r,g,b;
  QRgb *dst;

  if (localImage!=NULL) delete localImage;
  localImage=new QImage( width,height,QImage::Format_RGB32);

  /* In this format each four bytes is two pixels. Each four bytes is two Y's, a Cb and a Cr.
     Each Y goes to one of the pixels, and the Cb and Cr belong to both pixels. */
  py = src;
  pu = src + 1;
  pv = src + 3;
  for (line = 0; line < height; ++line)
    {
      dst=(QRgb *)localImage->scanLine(line);
      for (column = 0; column < width; ++column) // ITU-R float
        {
          r = CLIP((double)*py + 1.402*((double)*pv-128.0));
          g = CLIP((double)*py - 0.344*((double)*pu-128.0) - 0.714*((double)*pv-128.0));
          b = CLIP((double)*py + 1.772*((double)*pu-128.0));
          dst[column]=qRgb(r,g,b);
          // increase py every time
          py += 2;
          // increase pu,pv every second time
          if ((column & 1)==1)
            {
              pu += 4;
              pv += 4;
            }
        }
    }
}


/**
  Convert from YUV422 format to RGB888 using NTSC. Formulae are described on http://en.wikipedia.org/wiki/YUV

  \param width width of image
  \param height height of image
  \param src source
  \param dst destination
*/

void videoCapture::YUV422toRGB888_NTSC(unsigned char *src,int width, int height)
{
  int line, column;
  unsigned char *py, *pu, *pv;
  unsigned char r,g,b;
  QRgb *dst;

  if (localImage!=NULL) delete localImage;
  localImage=new QImage( width,height,QImage::Format_RGB32);

  /* In this format each four bytes is two pixels. Each four bytes is two Y's, a Cb and a Cr.
     Each Y goes to one of the pixels, and the Cb and Cr belong to both pixels. */
  py = src;
  pu = src + 1;
  pv = src + 3;
  for (line = 0; line < height; ++line)
    {
      dst=(QRgb *)localImage->scanLine(line);
      for (column = 0; column < width; ++column)
        {
          // NTSC integer
          r = CLIP( (298*(*py-16) + 409*(*pv-128) + 128) >> 8 );
          g = CLIP( (298*(*py-16) - 100*(*pu-128) - 208*(*pv-128) + 128) >> 8 );
          b = CLIP( (298*(*py-16) + 516*(*pu-128) + 128) >> 8 );
          dst[column]=qRgb(r,g,b);
          // increase py every time
          py += 2;
          // increase pu,pv every second time
          if ((column & 1)==1)
            {
              pu += 4;
              pv += 4;
            }
        }
    }
}


int videoCapture::changeCtrl(int ctrl, int value) // an enum for formats and reorganisation would be great...
{
	struct v4l2_queryctrl queryctrl;
	struct v4l2_control control;
	
	if(!opened) // At the begining of the function.
	{
		return -1;
	}
/*
 * ctrl values :
 * 	0 : Saturation
 * 	1 : Power line Frequency (néons)
 * 	2 : Brightness
 * 	3 : Contrast
 * 	4 : Sharpness
 * 	5 : Reset Pan/Tilt
 */
	__u32 CTRL;
	switch(ctrl)
	{
		case Saturation: 
		{
			CTRL = V4L2_CID_SATURATION;
			break;
		}
		case Brightness: 
		{
			CTRL = V4L2_CID_BRIGHTNESS;
			break;
		}
  case Hue:
  {
    CTRL = V4L2_CID_HUE;
    break;
  }
		case Contrast: 
		{
			CTRL = V4L2_CID_CONTRAST;
			break;
		}
//#ifdef V4L2_CID_POWER_LINE_FREQUENCY
//		case PowerLineFreq:
//		{
//			CTRL = V4L2_CID_POWER_LINE_FREQUENCY;
//			break;
//		}
//#endif
		case Sharpness:
		{
#ifdef 	V4L2_CID_SHARPNESS
		CTRL = V4L2_CID_SHARPNESS;
#else
	        CTRL=0;
#endif

		break;
		}

		default:
			CTRL = 0;
	}

	memset (&queryctrl, 0, sizeof queryctrl);
	memset (&control, 0, sizeof control);
	queryctrl.id = CTRL;
	if (-1 == ioctl (dev, VIDIOC_QUERYCTRL, &queryctrl)) 
	{
	        if (errno != EINVAL) 
		{
#ifdef DEBUG
			perror ("VIDIOC_QUERYCTRL");
#endif
			return EXIT_FAILURE;
		} 
	} else 
	{
		control.id = CTRL;
		control.value = value;
		if (-1 == ioctl (dev, VIDIOC_S_CTRL, &control)) {
#ifdef DEBUG
			perror("VIDIOC_S_CTRL");
			printf(" * Error while setting control\n");
#endif
			return EXIT_FAILURE;
        	}
	}
	return EXIT_SUCCESS;
}

int videoCapture::currentWidth() const
{
	return (int) fmt.fmt.pix.width;
}

int videoCapture::currentHeight() const
{
	return (int) fmt.fmt.pix.height;
}

int videoCapture::currentPixelFormat() const
{
	return (int) fmt.fmt.pix.pixelformat;
}


int videoCapture::defaultCtrlVal(unsigned int control, int &defaultValue)
{
	struct v4l2_queryctrl queryctrl;
	QString ctrl;
	
	if(!opened)
	{
		return false;
	}
	
	memset(&queryctrl, 0, sizeof queryctrl);
	switch(control)
	{
		case Saturation : 
		{
			ctrl = "Saturation";
			queryctrl.id = V4L2_CID_SATURATION;
			break;
		}
		case Brightness : 
		{
			ctrl = "Brightness";
			queryctrl.id = V4L2_CID_BRIGHTNESS;
			break;
		}
  case Hue :
  {
    ctrl = "Hue";
    queryctrl.id = V4L2_CID_HUE;
    break;
  }
		case Contrast : 
		{
			ctrl = "Contrast";
			queryctrl.id = V4L2_CID_CONTRAST;
			break;
		}

//		case PowerLineFreq :
//		{
//			ctrl = "Powerline Frequecy";
//			queryctrl.id = V4L2_CID_POWER_LINE_FREQUENCY;
//			break;
//		}

		case Sharpness : 
		{
			ctrl = "Sharpness";
#ifdef V4L2_CID_SHARPNESS
			queryctrl.id = V4L2_CID_SHARPNESS;
#endif
			break;
		}

		default :
			ctrl = "ERROR";
	}

	QString str;
	if (-1 == ioctl(dev, VIDIOC_QUERYCTRL, &queryctrl))
	{
    addToLog(QString("Unable to set control %1, %2").arg(ctrl).arg(errno),DBCAM);
    return FALSE;
	}

	defaultValue = (int)queryctrl.default_value;

	return true;
}

//bool videoCapture::panTiltSupported()
//{
//	struct v4l2_queryctrl queryctrl;

//  if(!opened)	return false;

//	memset(&queryctrl, 0, sizeof queryctrl);
//	queryctrl.id = V4L2_CID_TILT_RELATIVE; // Could be V4L2_CID_PAN_RELATIVE.

//  if (ioctl(dev, VIDIOC_QUERYCTRL, &queryctrl)<0)
//	{
//     addToLog(QString("Unable to check wether Pan Tilt is supported, %1").arg(errno),DBCAM);
//      return FALSE;
//	}

//	if (queryctrl.flags & V4L2_CTRL_FLAG_DISABLED)
//	{
//     addToLog("Pan & Tilt not supported."),DBCAM);
//     return FALSE; //FLAG_NOT_SUPPORTED;
//	}

//	return true;
//}

//void videoCapture::turnRight()
//{
//#ifdef V4L2_CID_PAN_RELATIVE
//	struct v4l2_queryctrl queryctrl;
//	struct v4l2_control control;

//	memset (&queryctrl, 0, sizeof queryctrl);
//	memset (&control, 0, sizeof control);
//	queryctrl.id = V4L2_CID_PAN_RELATIVE;
//	if (-1 == ioctl (dev, VIDIOC_QUERYCTRL, &queryctrl))
//	{
//	        if (errno != EINVAL)
//		{
//			perror ("VIDIOC_QUERYCTRL");
//			return;
//		}
//	} else
//	{
//		control.id = V4L2_CID_PAN_RELATIVE;
//		control.value = -320;
//		if (-1 == ioctl (dev, VIDIOC_S_CTRL, &control)) {
//			perror("VIDIOC_S_CTRL");
//			return;
//        	}
//	}
//#endif
//}

//void videoCapture::turnLeft()
//{
//#ifdef V4L2_CID_PAN_RELATIVE
//	struct v4l2_queryctrl queryctrl;
//	struct v4l2_control control;

//	memset (&queryctrl, 0, sizeof queryctrl);
//	memset (&control, 0, sizeof control);
//	queryctrl.id = V4L2_CID_PAN_RELATIVE;
//	if (-1 == ioctl (dev, VIDIOC_QUERYCTRL, &queryctrl))
//	{
//	        if (errno != EINVAL)
//		{
//			perror ("VIDIOC_QUERYCTRL");
//			return;
//		}
//	} else
//	{
//		control.id = V4L2_CID_PAN_RELATIVE;
//		control.value = 320;
//		if (-1 == ioctl (dev, VIDIOC_S_CTRL, &control)) {
//			perror("VIDIOC_S_CTRL");
//			return;
//        	}
//	}
//#endif
//}

//void videoCapture::turnUp()
//{
//#ifdef V4L2_CID_TILT_RELATIVE
//	struct v4l2_queryctrl queryctrl;
//	struct v4l2_control control;
	
//	memset (&queryctrl, 0, sizeof queryctrl);
//	memset (&control, 0, sizeof control);
//	queryctrl.id = V4L2_CID_TILT_RELATIVE;
//	if (-1 == ioctl (dev, VIDIOC_QUERYCTRL, &queryctrl))
//	{
//	        if (errno != EINVAL)
//		{
//			perror ("VIDIOC_QUERYCTRL");
//			return;
//		}
//	} else
//	{
//		control.id = V4L2_CID_TILT_RELATIVE;
//		control.value = -320;
//		if (-1 == ioctl (dev, VIDIOC_S_CTRL, &control)) {
//			perror("VIDIOC_S_CTRL");
//			return;
//        	}
//	}
//#endif
//}

//void videoCapture::turnDown()
//{
//#ifdef V4L2_CID_TILT_RELATIVE
//	struct v4l2_queryctrl queryctrl;
//	struct v4l2_control control;

//	memset (&queryctrl, 0, sizeof queryctrl);
//	memset (&control, 0, sizeof control);
//	queryctrl.id = V4L2_CID_TILT_RELATIVE;
//	if (-1 == ioctl (dev, VIDIOC_QUERYCTRL, &queryctrl))
//	{
//	        if (errno != EINVAL)
//		{
//			perror ("VIDIOC_QUERYCTRL");
//			return;
//		}
//	} else
//	{
//		control.id = V4L2_CID_TILT_RELATIVE;
//		control.value = 320;
//		if (-1 == ioctl (dev, VIDIOC_S_CTRL, &control)) {
//			perror("VIDIOC_S_CTRL");
//			return;
//        	}
//	}
//#endif
//}


bool videoCapture::captureStop()
{
  int type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  int ret;

  if(!streaming) return FALSE;
  ret = ioctl(dev, VIDIOC_STREAMOFF, &type);
  if (ret < 0)
    {
     addToLog(QString("Unable  to stop capture, %1").arg(errno),DBCAM);
     return FALSE;
    }

  streaming = false;
  return TRUE;
}

bool videoCapture::captureStart()
{
  int i, ret;
  if (!opened) return FALSE;
  abortCap=FALSE;
  addToLog("captureStart",DBCAM);

//  if ((ret = setFormat(currentWidth(), currentHeight(), currentPixelFormat())) != 0)
//    {
//      addToLog(QString("Set format error :  %1").arg(ret),DBCAM);
//      return FALSE;
//    }

////Allocate buffers
//  if (!allocated)
//    {
//      memset(&rb, 0, sizeof rb);
//      rb.count = 2;
//      rb.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
//      rb.memory = V4L2_MEMORY_MMAP;

//      ret = ioctl(dev, VIDIOC_REQBUFS, &rb);
//      if (ret < 0)
//        {
//         addToLog(QString("Unable to allocate buffers %1").arg(ret),DBCAM);
//         return FALSE;
//      }
//      allocated = true;
//    }

  // Map the buffers.
  memset(&buf, 0, sizeof buf);
  buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  buf.memory = V4L2_MEMORY_MMAP;
  for (i = 0; i < numBuffers; i++)
    {
      buf.index = i;
      ret = ioctl(dev, VIDIOC_QUERYBUF, &buf);
      if (ret < 0)
        {
          addToLog(QString("Unable to query buffe").arg(ret),DBCAM);
          return FALSE;
        }
      mem[i] = (uchar *) mmap(0, buf.length, PROT_READ | PROT_WRITE, MAP_SHARED, dev, buf.m.offset);
      if (mem[i] == MAP_FAILED)
         {
           addToLog(QString("Unable to map buffers %1").arg(ret),DBCAM);
           return FALSE;
         }
       bufLength = buf.length;
       mmaped = true;
     }

 // Queue the buffers

  for (i = 0; i < numBuffers; i++)
  {
    buf.index = i;
    ret = ioctl(dev, VIDIOC_QBUF, &buf);
    if (ret < 0)
    {
      addToLog(QString("Unable to queue buffer %1").arg(errno),DBCAM);
      return FALSE;
    }
  }

  // Start streaming.
  ret = ioctl(dev, VIDIOC_STREAMON, &buf.type);
  if (ret < 0)
  {
    addToLog(QString("Unable to start capture %1").arg(errno),DBCAM);
    return FALSE;
  }
  streaming = true;
  return TRUE;
}

bool videoCapture::stopStreaming()
{
  if(!streaming) return FALSE;
  if (munmap(mem[0], bufLength) == -1)
  {
    addToLog(QString("videoCapture::stopStreaming : munmap 0 failed. errno = %1").arg(errno),DBCAM);
  }

  if (munmap(mem[1], bufLength) == -1)
  {
    addToLog(QString("videoCapture::stopStreaming : munmap 1 failed. errno = %1").arg(errno),DBCAM);
  }

  if (munmap(mem[2], bufLength) == -1)
  {
    addToLog(QString("videoCapture::stopStreaming : munmap 2 failed. errno = %1").arg(errno),DBCAM);
  }

  if (munmap(mem[3], bufLength) == -1)
  {
    addToLog(QString("videoCapture::stopStreaming : munmap 3 failed. errno = %1").arg(errno),DBCAM);
  }




  else
    mmaped = false;

  if(captureStop() == 0)
  {
    streaming = false;
    addToLog(" * Succesful Stopped",DBCAM);
  }
  return TRUE;
}

bool videoCapture::takeSnapshot()
{
  if(!open()) return FALSE;
  if(!captureStart()) return FALSE;
  if(!getFrame()) return FALSE; // first frame always corrupted
  if(!getFrame()) return FALSE;
  return stopStreaming();
}

bool videoCapture::startSnapshots()
{
  if(!open()) return FALSE;
  if(!captureStart()) return FALSE;
  return TRUE;
}


void videoCapture::dumpCaps(v4l2_capability &cap)
{
  if (cap.capabilities & V4L2_CAP_VIDEO_CAPTURE) addToLog("The device supports the Video Capture interface",DBCAM);
  if (cap.capabilities & V4L2_CAP_VIDEO_OUTPUT) addToLog("The device supports the Video Output interface",DBCAM);
  if (cap.capabilities & V4L2_CAP_VIDEO_OVERLAY) addToLog("The device supports the Video Overlay interface",DBCAM);
  if (cap.capabilities & V4L2_CAP_VBI_CAPTURE) addToLog("The device supports the Raw VBI Capture interface",DBCAM);
  if (cap.capabilities & V4L2_CAP_VBI_OUTPUT) addToLog("The device supports the Raw VBI Output interface",DBCAM);
  if (cap.capabilities & V4L2_CAP_SLICED_VBI_CAPTURE) addToLog("The device supports the Sliced VBI Capture interface",DBCAM);
  if (cap.capabilities & V4L2_CAP_SLICED_VBI_OUTPUT) addToLog("The device supports the Sliced VBI Output interface",DBCAM);
  if (cap.capabilities & V4L2_CAP_RDS_CAPTURE) addToLog("[to be defined]",DBCAM);
#ifdef V4L2_CAP_VIDEO_OUTPUT_OVERLAY
  if (cap.capabilities & V4L2_CAP_VIDEO_OUTPUT_OVERLAY) addToLog("The device supports the Video Output Overlay (OSD) interface",DBCAM);
#endif
  if (cap.capabilities & V4L2_CAP_TUNER) addToLog("The device has some sort of tuner or modulator",DBCAM);
  if (cap.capabilities & V4L2_CAP_AUDIO) addToLog("The device has audio inputs or outputs.",DBCAM);
  if (cap.capabilities & V4L2_CAP_READWRITE) addToLog("The device supports the read() and/or write() I/O methods.",DBCAM);
  if (cap.capabilities & V4L2_CAP_ASYNCIO) addToLog("The device supports the asynchronous I/O methods.",DBCAM);
  if (cap.capabilities & V4L2_CAP_STREAMING) addToLog("The device supports the streaming I/O method.",DBCAM);
}

