/* Copyright (C) 2003 Nikos Chantziaras.
 *
 * This file is part of the QTads program.  This program is free
 * software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#include "config.h"

#include "qtadssettings.h"

#include <qsettings.h>
#include <qcolor.h>
#include <qfont.h>
#include <qaction.h>

#include "qtadsio.h"
#include "qtadsmainwindow.h"
#include "qtadsgamewindow.h"


static const QString WINDOWS_REGISTRY("/Real NC");
static const QString APP_KEY("/QTads/");
static const QString COLORS_KEY(APP_KEY + "Colors/");
static const QString FONTS_KEY(APP_KEY + "Fonts/");
static const QString FORMAT_KEY(APP_KEY + "Formatting/");
static const QString TERP_KEY(APP_KEY + "Interpreter/");
static const QString GEOM_KEY(APP_KEY + "Geometry/");
static const QString RECENT_KEY(APP_KEY + "RecentGames/");
static const QString THEMES_KEY(APP_KEY + "Themes/");
// The theme's name will be appended to this key.
static const QString SAVED_THEME_KEY(APP_KEY + "SavedTheme_");


/* --------------------------------------------------------------------
 * QTadsTheme
 */

QTadsTheme::QTadsTheme( const QColor& gameBg, const QColor& gameText, const QColor& statusBg,
		        const QColor& statusText, const QFont& gameFnt,
		        const QFont& statusFnt, const Qt::AlignmentFlags& alignFlags,
		        int leftMarg, int rightMarg, bool doubleSpc, bool boldInp,
		        bool italicInp, bool underlinedInp, bool curlyQuot,
		        bool curlyApostr, bool dashConv )
: fGameBgColor(gameBg), fGameTextColor(gameText), fStatusBgColor(statusBg),
  fStatusTextColor(statusText), fGameFont(gameFnt), fStatusFont(statusFnt),
  fAlignment(alignFlags), fLeftMargin(leftMarg), fRightMargin(rightMarg),
  fDoubleSpace(doubleSpc), fBoldInput(boldInp), fItalicInput(italicInp),
  fUnderlinedInput(underlinedInp), fCurlyQuotes(curlyQuot),
  fCurlyApostrophes(curlyApostr), fDashConversion(dashConv)
{
}


/* --------------------------------------------------------------------
 * QTadsSettings
 */

QTadsSettings::QTadsSettings()
{
	// Always use a text file for saving/restoring state, even on
	// non-Unix systems.  This means that in Windows we don't use
	// the registry and in Mac OS X we don't use the Carbon
	// preferences API.  The intention is to make uninstallation of
	// the application as simple as possible.  This could change in
	// the future, so we also provide a root-key for the Windows
	// registry (but it's not used for now).
	//
	// TODO: Some early 3.x versions of Qt lack the ctor that
	// accepts a QSettings::Format argument.  Find out which is the
	// version that introduced this ctor.  For now, we assume it is
	// version 3.1.0.
#if (QT_VERSION >= 0x030100)
	QSettings sett(QSettings::Ini);
#else
	QSettings sett;
#endif
	sett.insertSearchPath(QSettings::Windows, WINDOWS_REGISTRY);

	switch (sett.readNumEntry(TERP_KEY + "ScrollBuffer", BufferSize1)) {
	  case BufferSize1:
		this->fScrollBufferSize = BufferSize1;
		break;
	  case BufferSize2:
		this->fScrollBufferSize = BufferSize2;
		break;
	  case BufferSize3:
		this->fScrollBufferSize = BufferSize3;
		break;
	  case BufferSize4:
		this->fScrollBufferSize = BufferSize4;
		break;
	  case BufferSize5:
		this->fScrollBufferSize = BufferSize5;
		break;
	  case BufferSize6:
		this->fScrollBufferSize = BufferSize6;
		break;
	  case BufferSize7:
		this->fScrollBufferSize = BufferSize7;
		break;
	  case BufferSize8:
		this->fScrollBufferSize = BufferSize8;
		break;
	  default:
		// Paranoia.
		this->fScrollBufferSize = BufferSize1;
	}
	this->fImmediateQuit = sett.readBoolEntry(TERP_KEY + "ImmediateQuit", true);
	this->fImmediateRestart = sett.readBoolEntry(TERP_KEY + "ImmediateRestart", true);
	QTadsIO::moveWindow(sett.readNumEntry(GEOM_KEY + "PositionX"),
			    sett.readNumEntry(GEOM_KEY + "PositionY"));
	QTadsIO::resizeWindow(sett.readNumEntry(GEOM_KEY + "Width", 400),
			      sett.readNumEntry(GEOM_KEY + "Height", 300));
	QTadsIO::mainWindow().displayFullScreenAction->setOn(sett.readBoolEntry(GEOM_KEY
										+ "FullScreen",
										false));

	// Create the default theme and activate it.
	this->fCurrentTheme = "Default";
	this->fThemes[this->fCurrentTheme];

	// Load the stored themes (if any).
	int themeCount = sett.readNumEntry(THEMES_KEY + "Count");
	for (int i = 0; i < themeCount; ++i) {
		QString themeName;
		QString themeKey;
		QColor gameBg;
		QColor gameText;
		QColor statusBg;
		QColor statusText;
		QFont gameFnt;
		QFont statusFnt;
		Qt::AlignmentFlags alignFlags;
		int leftMarg;
		int rightMarg;
		bool doubleSpc;
		bool boldInp;
		bool italicInp;
		bool underlinedInp;
		bool curlyQuot;
		bool curlyApostr;
		bool dashConv;

		// Read the name of the next theme.
		themeName = sett.readEntry(THEMES_KEY + QString::number(i));

		// The name we just read will be used as the key to
		// access the theme's data.
		themeKey = SAVED_THEME_KEY + themeName + "/";

		// Load the theme's data.
		gameBg.setNamedColor(sett.readEntry(themeKey + "GameBgColor"));
		gameText.setNamedColor(sett.readEntry(themeKey + "GameTextColor"));
		statusBg.setNamedColor(sett.readEntry(themeKey + "StatusBgColor"));
		statusText.setNamedColor(sett.readEntry(themeKey + "StatusTextColor"));
		gameFnt.fromString(sett.readEntry(themeKey + "GameFont"));
		statusFnt.fromString(sett.readEntry(themeKey + "StatusFont"));
		alignFlags = static_cast<Qt::AlignmentFlags>(sett.readNumEntry(themeKey +
									       "Alignment"));
		leftMarg = sett.readNumEntry(themeKey + "LeftMargin");
		rightMarg = sett.readNumEntry(themeKey + "RightMargin");
		doubleSpc = sett.readBoolEntry(themeKey + "DoubleSpace");

		boldInp = sett.readBoolEntry(themeKey + "BoldInput");
		italicInp = sett.readBoolEntry(themeKey + "ItalicInput");
		underlinedInp = sett.readBoolEntry(themeKey + "UnderlinedInput");

		curlyQuot = sett.readBoolEntry(themeKey + "CurlyQuotes");
		curlyApostr = sett.readBoolEntry(themeKey + "CurlyApostrophes");
		dashConv = sett.readBoolEntry(themeKey + "DashConversion");

		// Construct the theme.
		this->fThemes[themeName] = QTadsTheme(gameBg, gameText, statusBg, statusText,
						      gameFnt, statusFnt, alignFlags,
						      leftMarg, rightMarg, doubleSpc, boldInp,
						      italicInp, underlinedInp, curlyQuot,
						      curlyApostr, dashConv);
	}
	this->currentTheme(sett.readEntry(THEMES_KEY + "Current", "Default"));

	this->fRecentGamesCapacity = sett.readNumEntry(RECENT_KEY + "Capacity", 10);
	this->fRecentGames.reserve(this->fRecentGamesCapacity);
	this->fRecentGamesCount = sett.readNumEntry(RECENT_KEY + "Count");
	this->fRecentGames.resize(this->fRecentGamesCount);
	for (int i = 0; i < this->fRecentGamesCount; ++i) {
		this->fRecentGames[i] = sett.readEntry(RECENT_KEY + QString::number(i));
	}
}


QTadsSettings::~QTadsSettings()
{
#if (QT_VERSION >= 0x030100)
	QSettings sett(QSettings::Ini);
#else
	QSettings sett;
#endif
	sett.insertSearchPath(QSettings::Windows, WINDOWS_REGISTRY);

	sett.writeEntry(TERP_KEY + "ScrollBuffer", this->fScrollBufferSize);
	sett.writeEntry(TERP_KEY + "ImmediateQuit", this->fImmediateQuit);
	sett.writeEntry(TERP_KEY + "ImmediateRestart", this->fImmediateRestart);
	sett.writeEntry(GEOM_KEY + "PositionX", QTadsIO::mainWindow().pos().x());
	sett.writeEntry(GEOM_KEY + "PositionY", QTadsIO::mainWindow().pos().y());
	sett.writeEntry(GEOM_KEY + "Width", QTadsIO::mainWindow().size().width());
	sett.writeEntry(GEOM_KEY + "Height", QTadsIO::mainWindow().size().height());
	sett.writeEntry(GEOM_KEY + "FullScreen", QTadsIO::fullScreen());

	sett.writeEntry(THEMES_KEY + "Current", this->fCurrentTheme);
	int themeCount = 0; // How many themes were saved.
	for (ThemeMap::const_iterator i = this->fThemes.begin(); i != this->fThemes.end(); ++i) {
		QString themeKey;
		if (i->first.isEmpty()) {
			// std::map sometimes has an empty entry.
			// Don't save it.
			continue;
		}
		sett.writeEntry(THEMES_KEY + QString::number(themeCount), i->first);
		themeKey = SAVED_THEME_KEY + i->first + "/";
		sett.writeEntry(themeKey + "GameBgColor", i->second.gameBgColor().name());
		sett.writeEntry(themeKey + "GameTextColor", i->second.gameTextColor().name());
		sett.writeEntry(themeKey + "StatusBgColor", i->second.statusBgColor().name());
		sett.writeEntry(themeKey + "StatusTextColor", i->second.statusTextColor().name());
		sett.writeEntry(themeKey + "GameFont", i->second.gameFont().toString());
		sett.writeEntry(themeKey + "StatusFont", i->second.statusFont().toString());
		sett.writeEntry(themeKey + "Alignment", i->second.alignment());
		sett.writeEntry(themeKey + "LeftMargin", i->second.leftMargin());
		sett.writeEntry(themeKey + "RightMargin", i->second.rightMargin());
		sett.writeEntry(themeKey + "DoubleSpace", i->second.doubleSpace());
		sett.writeEntry(themeKey + "BoldInput", i->second.boldInput());
		sett.writeEntry(themeKey + "ItalicInput", i->second.italicInput());
		sett.writeEntry(themeKey + "UnderlinedInput", i->second.underlinedInput());
		sett.writeEntry(themeKey + "CurlyQuotes", i->second.curlyQuotes());
		sett.writeEntry(themeKey + "CurlyApostrophes", i->second.curlyApostrophes());
		sett.writeEntry(themeKey + "DashConversion", i->second.dashConversion());
		++themeCount;
	}
	sett.writeEntry(THEMES_KEY + "Count", themeCount);

	for (int i = 0; i < this->fRecentGamesCount; ++i) {
		sett.writeEntry(RECENT_KEY + QString::number(i), this->fRecentGames[i]);
	}
	sett.writeEntry(RECENT_KEY + "Count", this->fRecentGamesCount);
	sett.writeEntry(RECENT_KEY + "Capacity", this->fRecentGamesCapacity);
}


bool
QTadsSettings::currentTheme( const QString& themeName )
{
	if (this->fThemes.find(themeName) == this->fThemes.end()) {
		return false;
	}
	this->fCurrentTheme = themeName;
	return true;
}


QTadsTheme&
QTadsSettings::theme( const QString& themeName )
{
	if (this->fThemes.find(themeName) == this->fThemes.end()) {
		return this->fThemes["Default"];
	}
	return this->fThemes[themeName];
}


bool
QTadsSettings::removeTheme( const QString& themeName )
{
	if (themeName == "Default") {
		// The default theme can't be removed.
		return false;
	}
	if (themeName == this->fCurrentTheme) {
		// We can't delete the theme because it's active.
		// Switch to the default theme first.
		this->currentTheme("Default");
	}
	return this->fThemes.erase(themeName) != 0;
}


std::vector<QString>
QTadsSettings::themes() const
{
	std::vector<QString> res;
	for (ThemeMap::const_iterator i = this->fThemes.begin(); i != this->fThemes.end(); ++i) {
		if (i->first.isEmpty()) {
			continue;
		}
		res.push_back(i->first);
	}
	return res;
}


std::vector<QString>
QTadsSettings::recentGames() const
{
	//std::vector<QString> res(this->fRecentGames);
	//res.resize(this->fRecentGamesCount);
	//return res;
	return this->fRecentGames;
}


void
QTadsSettings::addRecentGame( const QString& filename )
{
	// Check if the given filename is already in the vector.
	for (int i = 0; i < this->fRecentGamesCount; ++i) {
		if (this->fRecentGames[i] == filename) {
			// It's already there.
			return;
		}
	}

	if (this->fRecentGamesCount == this->fRecentGamesCapacity) {
		// The list is full; decrease the element counter by 1.
		// This will result in the last element being
		// overriden.
		--this->fRecentGamesCount;
	} else {
		// Enlarge vector by 1 item.
		this->fRecentGames.resize(this->fRecentGames.size() + 1);
	}

	// Shift everything down by 1, and store the new entry at
	// position one.  This operation is slow, but since we're
	// talking about a maximum of about 10 elements, it won't hurt.
	for (int i = this->fRecentGamesCount; i > 0; --i) {
		this->fRecentGames[i] = this->fRecentGames[i-1];
	}

	// Now store it as the newest entry.
	this->fRecentGames[0] = filename;
	++this->fRecentGamesCount;
}


QTadsSettings::ScrollBufferSize
QTadsSettings::scrollBufferSize() const
{
	Q_ASSERT(static_cast<unsigned int>(this->fScrollBufferSize)
		 == QTadsIO::gameWindow().scrollBufferSize());

	return this->fScrollBufferSize;
}
