/***************************************************************************
 *   Copyright (C) 2008 by Robert Keevil                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; under version 2 of the License.         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <iostream>
#include "libscrobble.h"

bool Scrobble::mtp_connect()
{
    if (mtp_connected)
        return true; // or mtp_disconnect()??

    LIBMTP_Init();

  add_log(LOG_DEBUG, "MTP: Attempting to connect device(s)");


    switch(LIBMTP_Get_Connected_Devices(&mtp_device_list))
    {
    case LIBMTP_ERROR_NO_DEVICE_ATTACHED:
        add_log(LOG_ERROR, "MTP: No Devices have been found");
        mtp_device_list = NULL;
        return false;
    case LIBMTP_ERROR_CONNECTING:
        add_log(LOG_ERROR, "MTP: Error connecting.");
        mtp_device_list = NULL;
        return false;
    case LIBMTP_ERROR_MEMORY_ALLOCATION:
        add_log(LOG_ERROR, "MTP: Memory Allocation Error.");
        mtp_device_list = NULL;
        return false;

    // Unknown general errors - This should never execute
    case LIBMTP_ERROR_GENERAL:
    default:
        add_log(LOG_ERROR, "MTP: Unknown error!");
        mtp_device_list = NULL;
    return false;

    // Successfully connected at least one device, so continue
    case LIBMTP_ERROR_NONE:
        add_log(LOG_INFO, "MTP: Successfully connected");
    }
    return true;
}

void Scrobble::mtp_disconnect()
{
    if (mtp_connected)
    {
        LIBMTP_Release_Device_List(mtp_device_list);
        mtp_connected = false;
    }
}

void Scrobble::mtp_get_tracks()
{
    mtp_iterate(false);
    // No info on when the track was played with MTP devices
    recalc_now();

    if (entries.size() > 0)
        scrobble_method = SCROBBLE_MTP;
}

void Scrobble::mtp_clear_tracks()
{
    mtp_iterate(true);
}

void Scrobble::mtp_iterate(bool clear_tracks)
{
// This is mostly taken straight from libmtp examples/tracks.c

    LIBMTP_mtpdevice_t *iter;
    LIBMTP_track_t *tracks;

    // iterate through connected MTP devices
    for(iter = mtp_device_list; iter != NULL; iter = iter->next)
    {
        char *friendlyname;
        // Echo the friendly name so we know which device we are working with

        friendlyname = LIBMTP_Get_Friendlyname(iter);
        if (friendlyname == NULL)
        {
            add_log(LOG_DEBUG, "MTP: Friendly name: (NULL)");
        } else {
            add_log(LOG_DEBUG, "MTP: Friendly name: " + string(friendlyname));
            free(friendlyname);
        }

        // Get track listing.
        tracks = LIBMTP_Get_Tracklisting_With_Callback(iter, NULL, NULL);
        if (tracks == NULL)
        {
            add_log(LOG_INFO, "MTP: No tracks.");
        } else {
            LIBMTP_track_t *track, *tmp;
            track = tracks;
            while (track != NULL)
            {
                mtp_trackinfo(iter, track, clear_tracks);
                tmp = track;
                track = track->next;
                LIBMTP_destroy_track_t(tmp);
            }
        }
    }
}

void Scrobble::mtp_trackinfo(LIBMTP_mtpdevice_t *device, LIBMTP_track_t *track, bool clear_track)
{
    if (track->artist != NULL // check if missing
        && track->title != NULL // check if missing
        && track->usecount > 0 // check track has been played
        )
    {
        if (clear_track)
        {
            track->usecount = 0;
            LIBMTP_Update_Track_Metadata(device, track);
        }
        else
        {
            scrob_entry tmp;
            tmp.artist = track->artist;
            tmp.title = track->title;
            tmp.album = (track->album==NULL)?"":track->album;
            tmp.tracknum = int2string(track->tracknumber);
            tmp.length = track->duration/1000;
            tmp.played = 'L';
            tmp.when = 0; // no idea when it was played :(
            tmp.mb_track_id = "";

            for (unsigned int i = 0; i < track->usecount; i++)
                entries.push_back(tmp);
        }
    }
}
