#!/usr/bin/python

"""
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
"""
__version__ = "$Revision: 1.13 $"
# $Source: /cvsroot/quantlib/QuantLib-Python/QuantLib/test/european_with_dividends.py,v $

import QuantLib
import unittest

def relErr(x1, x2, reference):
    if reference != 0.0:
        return abs(x1-x2)/reference
    else:
        return 10e10

class FdDividendEuropeanOptionTest(unittest.TestCase):
    def runTest(self):
        "Testing European option pricer with dividends"
        pricer = QuantLib.FdDividendEuropeanOption
        nstp = 150
        ngrd = nstp+1

        div=[3.92,4.21]
        dates=[0.333,0.667]

        rangeUnder = [100]
        rangeQrate = [0.0, 0.05,0.15]
        rangeResTime = [1.0,2.0]
        rangeStrike = [50, 99.5, 100, 100.5, 150]
        rangeVol = [0.04,0.2,0.7]
        rangeRrate = [0.01,0.1,0.3]

        err_delta = 1e-4
        err_gamma = 1e-4
        err_theta = 1e-4
        err_rho  =  1e-4
        err_vega =  1e-4

        for typ in ['Call','Put','Straddle']:
          resuDelta = [];  resuGamma = [];  resuTheta = []
          resuRho   = [];  resuVega  = []
          for under in rangeUnder:
            for Qrate in rangeQrate:
              for resTime in rangeResTime:
                for Rrate in rangeRrate:
                  for strike in rangeStrike:
                    for vol in rangeVol:
                      #Check Greeks
                      dS = under/10000.0
                      dT = resTime/nstp
                      dVol = vol/10000.0
                      dR = Rrate/10000.0
                      opt = pricer(typ, under, strike, Qrate, Rrate, resTime,
                        vol, div, dates)
                      opt_val = opt.value()
                      if opt_val>0.00001*under:
                        optPs = pricer(typ, under+dS, strike, Qrate,
                          Rrate   , resTime   , vol,      div, dates)
                        optMs = pricer(typ, under-dS, strike, Qrate,
                          Rrate   , resTime   , vol,      div, dates)
                        optPt = pricer(typ, under   , strike, Qrate,
                          Rrate   , resTime+dT, vol,      div,
                          map(lambda t,dT=dT:t+dT, dates))
                        optMt = pricer(typ, under   , strike, Qrate,
                          Rrate   , resTime-dT, vol,      div,
                          map(lambda t,dT=dT:t-dT, dates))
                        optPr = pricer(typ, under   , strike, Qrate,
                          Rrate+dR, resTime   , vol,      div, dates)
                        optMr = pricer(typ, under   , strike, Qrate,
                          Rrate-dR, resTime   , vol,      div, dates)
                        optPv = pricer(typ, under   , strike, Qrate,
                          Rrate   , resTime   , vol+dVol, div, dates)
                        optMv = pricer(typ, under   , strike, Qrate,
                          Rrate   , resTime   , vol-dVol, div, dates)

                        deltaNum = (optPs.value()-optMs.value())/(2*dS)
                        gammaNum = (optPs.delta()-optMs.delta())/(2*dS)
                        thetaNum =-(optPt.value()-optMt.value())/(2*dT)
                        rhoNum   = (optPr.value()-optMr.value())/(2*dR)
                        vegaNum  = (optPv.value()-optMv.value())/(2*dVol)

                        delta = opt.delta()
                        gamma = opt.gamma()
                        theta = opt.theta()
                        rho   = opt.rho()
                        vega  = opt.vega()

                        if not (relErr(delta,deltaNum,under)<=err_delta
                                and relErr(gamma,gammaNum,under)<=err_gamma
                                and relErr(theta,thetaNum,under)<=err_theta
                                and relErr(rho,  rhoNum,  under)<=err_rho
                                and relErr(vega, vegaNum, under)<=err_vega):
                            self.fail("""
Option details: %(typ)s %(under)f %(strike)f %(Qrate)f %(Rrate)f %(resTime)f %(vol)f
    value = %(opt_val)+9.5f
    delta = %(delta)+9.5f, deltaNum = %(deltaNum)+9.5f
    gamma = %(gamma)+9.5f, gammaNum = %(gammaNum)+9.5f
    theta = %(theta)+9.5f, thetaNum = %(thetaNum)+9.5f
    rho   = %(rho)+9.5f, rhoNum   = %(rhoNum)+9.5f
    vega  = %(vega)+9.5f, vegaNum  = (vegaNum)%+9.5f
	                                  """ % locals())


if __name__ == '__main__':
    print 'testing QuantLib', QuantLib.__version__, QuantLib.QuantLibc.__file__, QuantLib.__file__
    import sys
    suite = unittest.TestSuite()
    suite.addTest(FdDividendEuropeanOptionTest())
    if sys.hexversion >= 0x020100f0:
        unittest.TextTestRunner(verbosity=2).run(suite)
    else:
        unittest.TextTestRunner().run(suite)
    raw_input('press any key to continue')


