/***************************************************************************
 *            fileout.c
 * 
 *  Sun Sep 24 12:10:36 2006
 *  Copyright  2006-2007  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/** @file fileout.c
	@brief  Export routines.
	@author Copyright 2006-2007  Neil Williams <linux@codehelp.co.uk>
	@author Copyright 1999  Robert Lissner 
*/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "config.h"

#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gtkextra/gtksheet.h>
#include "types.h"
#include "dialog_initial.h"
#include "dim_list_menu.h"
#include "edit.h"
#include "filein.h"
#include "main.h"
#include "report.h"

typedef enum
{
	/** closing one notebook tab */
	QL_CLOSE,
	/** quitting QuickList completely */
	QL_QUIT
} QlCloseMode;

QlCloseMode close_mode;

/** for writing less than whole file */
GtkSheetRange write_range;		

typedef enum
{
	/** default */
	QL_PRINT_NONE,
	/** print the entire sheet */
	QL_PRINT_ALL,
	/** print only those cells that are selected. */
	QL_PRINT_LIT,
	/** print cells that match the active filter. */
	QL_PRINT_FILTER
} QlPrintType;

static QlPrintType print_all = QL_PRINT_NONE;

static void
all_data_clicked (GtkWidget G_GNUC_UNUSED * w, gpointer G_GNUC_UNUSED data)
{
	print_all = QL_PRINT_ALL;
}

static void
filter_data_clicked (GtkWidget G_GNUC_UNUSED * w, gpointer G_GNUC_UNUSED data)
{
	print_all = QL_PRINT_FILTER;
}

static void
highlighted_data_clicked (GtkWidget G_GNUC_UNUSED * w, gpointer G_GNUC_UNUSED data)
{
	print_all = QL_PRINT_LIT;
}

/**  These buttons offer to export less than the whole file */
static void
ask_about_all (QlTabData * tab, GtkWidget * chooser)
{
	GtkWidget *all_data;
	GtkWidget *filter_data;
	GtkWidget *highlighted_data;
	GtkWidget *hbox;

	print_all = QL_PRINT_ALL;
	hbox = gtk_hbox_new (FALSE, 5);
	all_data = gtk_radio_button_new_with_label (NULL, _("Entire file"));
	gtk_box_pack_start (GTK_BOX (hbox), all_data, FALSE, FALSE, 0);
	g_signal_connect (all_data, "clicked",
		G_CALLBACK (all_data_clicked), NULL);

	filter_data = gtk_radio_button_new_with_label
		(gtk_radio_button_get_group (GTK_RADIO_BUTTON (all_data)),
		_("Allowed by filter"));
	if (!tab->file->filter_ptr)
		gtk_widget_set_sensitive (filter_data, FALSE);
	gtk_box_pack_start (GTK_BOX (hbox), filter_data, FALSE, FALSE, 0);
	g_signal_connect (filter_data, "clicked",
		G_CALLBACK (filter_data_clicked), NULL);

	highlighted_data = gtk_radio_button_new_with_label
		(gtk_radio_button_get_group (GTK_RADIO_BUTTON (all_data)),
		_("Highlighted cells"));
	gtk_box_pack_start (GTK_BOX (hbox), highlighted_data, FALSE, FALSE, 0);
	g_signal_connect (highlighted_data, "clicked",
		G_CALLBACK (highlighted_data_clicked), NULL);

	/* there can only be a selection if some cells are selected, more than one */
	if (tab->view->dialog_mode == MODE_SAVE ||
		(tab->view->sel_type == SELECT_ENTRY || 
			tab->view->sel_type != SELECT_COLUMN))
		gtk_widget_set_sensitive (highlighted_data, FALSE);
	gtk_widget_show_all (hbox);
	gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER(chooser), hbox);
}								/* end of ask_about_all */

/** Routine removes backslash and trailing blanks,
which are of no value.

\todo Replace with glib functions g_strchug and g_strchomp
*/
static void
remove_ql_chars (gchar * text)
{
	gchar *linebufp = text;
	while (*linebufp)
	{
		if (*linebufp == '\\')
			*linebufp = '/';
		linebufp++;
	}
	linebufp--;					/* skip back over ending zero */
	linebufp = g_strstrip (linebufp); /* removes trailing blanks */
}								/* end of remove_ql_chars */

static void
print_field_cb (gpointer G_GNUC_UNUSED field_index, gpointer field_ptr, gpointer user_data)
{
	QlFieldInfo * field = (QlFieldInfo*)field_ptr;
	FILE * f; /**< \bug stop passing FILE pointers around. */
	f = (FILE*)user_data;

	remove_ql_chars (field->name);
	fprintf (f, "FIELD %u %u %u %u %u %u %u %u %s\n",
		field->type,
		field->formatting,
		field->decimal_places,
		field->justification,
		field->sheet_column,
		field->width,
		field->unused1,
		field->unused2, 
		field->name);
}

/** Write the file onto disk.  Called by save, save_as, close, quit */
/** \bug stop passing f, just pass the filename, if at all. */
static void
write_ql_file (QlTabData * tab, FILE *f)
{
	gchar *text;
	gint32 rowx;
	gint16 fieldx = 0;
	gint16 sortx = 0;
	gint16 filterx = 0;
	gint16 subx = 0;
	gint16 reportx = 0;
	/** indexes into array */
	gint16 colx = 0;
	/** try to not write trailing slashes */
	gint16 blank_fields;
	/** try to not write blank records */
	gboolean got_data;

	g_return_if_fail (tab);
	get_window_size_loc (GTK_WIDGET (tab->qlc->parent));

	fprintf (f, "QUICKLIST %s 06/10/99 %u %u %u %u %u %u %u %u\n",
		VERSION,
		tab->file->last_field + 1,
		tab->file->sort_ct,
		tab->file->filter_ct,
		tab->file->report_ct, tab->view->width, tab->view->height, 
		tab->view->x, tab->view->y);
	ql_fieldinfo_foreach (tab, print_field_cb, f);
/*
	for (fieldx = 0; fieldx <= tab->file->last_field; fieldx++)
	{
		remove_ql_chars (tab->file->fields[fieldx]->name);
		fprintf (f, "FIELD %u %u %u %u %u %u %u %u %s\n",
			tab->file->fields[fieldx]->type,
			tab->file->fields[fieldx]->formatting,
			tab->file->fields[fieldx]->decimal_places,
			tab->file->fields[fieldx]->justification,
			tab->file->fields[fieldx]->sheet_column,
			tab->file->fields[fieldx]->width,
			tab->file->fields[fieldx]->unused1,
			tab->file->fields[fieldx]->unused2, 
			tab->file->fields[fieldx]->name);
	}
*/
	for (sortx = 0; sortx < tab->file->sort_ct; sortx++)
	{
		remove_ql_chars (tab->file->sorts[sortx].name);
		fprintf (f, "SORT %u %u %s\\",
			tab->file->sorts[sortx].unused1,
			tab->file->sorts[sortx].unused2, 
			tab->file->sorts[sortx].name);
		for (subx = 0; subx < tab->file->sorts[sortx].line_ct; subx++)
			fprintf (f, " %u %u",
				tab->file->sorts[sortx].line[subx].field,
				tab->file->sorts[sortx].line[subx].ascending);
		fprintf (f, "\n");
	}							/* end of printing one sort record */

	/* Here we need to write a filter. */
	for (filterx = 0; filterx < tab->file->filter_ct; filterx++)
	{
		remove_ql_chars (tab->file->filters[filterx].name);
		fprintf (f, "FILTER %d %d %s",
			tab->file->filters[filterx].by_and,
			tab->file->filters[filterx].use_nocase,
			tab->file->filters[filterx].name);
		for (subx = 0; subx < tab->file->filters[filterx].line_ct; subx++)
		{
			fprintf (f, "\\%u %u",
				tab->file->filters[filterx].line[subx].field,
				tab->file->filters[filterx].line[subx].type);
			remove_ql_chars (tab->file->filters[filterx].line[subx].compare);
			fprintf (f, " %s", tab->file->filters[filterx].line[subx].compare);
		}
		fprintf (f, "\n");		/* Finish off the filter */
	}

	/* Now a report column */
	for (reportx = 0; reportx < tab->file->report_ct; reportx++)
	{
		for (colx = 0; colx <= tab->file->reports[reportx].last_column; colx++)
			fprintf (f, "COLUMN %u %u %u %u %u %u\n",
				tab->file->reports[reportx].column[colx].field,
				tab->file->reports[reportx].column[colx].width,
				tab->file->reports[reportx].column[colx].group,
				tab->file->reports[reportx].column[colx].total,
				tab->file->reports[reportx].column[colx].unused1,
				tab->file->reports[reportx].column[colx].unused2);

		remove_ql_chars (tab->file->reports[reportx].name);
		fprintf (f, "REPORT %i %i %u %u %s\\\\\n",
			tab->file->reports[reportx].sort,
			tab->file->reports[reportx].filter,
			tab->file->reports[reportx].width,
			tab->file->reports[reportx].height, 
			tab->file->reports[reportx].name);

	}							/* end of writing report columns and report records */
	fprintf (f, "DATA 0 0\n");

	/* NOW WRITE THE DATA RECORDS.  Last row is always empty. 
	   The logic is difficult enough that I use an intermediate variable */
	for (rowx = 0; rowx < tab->file->last_row; rowx++)
	{
		gint last_field;
		got_data = FALSE;		/* to not write empty records */
		blank_fields = 0;		/* to not write trailing \\\\\ */
		last_field = ql_get_last_field (tab);
		for (fieldx = 0; fieldx <= last_field; fieldx++)
		{
			QlFieldInfo * field;
			field = ql_get_fieldinfo (tab, fieldx);
			colx = field->sheet_column;
			text = gtk_sheet_cell_get_text (tab->view->sheet, rowx, colx);
			if (text)
			{
				while (blank_fields)
				{				/* write the skipped blank fields */
					fprintf (f, "\\");
					blank_fields--;
				}

				remove_ql_chars (text);
				fprintf (f, "%s\\", text);
				got_data = TRUE;
			}
			else
				blank_fields++;
		}
		if (got_data)
			fprintf (f, "\n");	/* write nothing if nothing there */
	}
	fprintf (f, "\n");			/* write nothing if nothing there */
	tab->file->changed = FALSE;
}

/** \todo replace with gtkhtml. 

In effect, create a dummy report that includes all sheet data.
*/
static void
write_html (QlTabData * tab, FILE *f)
{
	gchar *text;
	gint32 rowx;
	gint16 fieldx;
	gint16 justification;
	QlFieldInfo * field;
	gint16 colx = 0;			/* indexes into array */

	/* Output Head Info Stuff */
	fprintf (f,
		"<html>\n"
		"<head>\n"
		"<title>%s</title>\n"
		"<meta name=\"Generator\" content=\"QUICKLIST %s\">\n"
		"</head>\n\n"
		"<body bgcolor=\"#ffffff\">\n"
		"<table width=\"2\">\n" "  <tr>\n", 
		tab->file->file_path, VERSION);
	for (colx = write_range.col0; colx <= write_range.coli; colx++)
	{
		fieldx = tab->file->col_to_field[colx];
		field = ql_get_fieldinfo (tab, fieldx);
		fprintf (f, "    <th>%s</th>\n", field->name);
	}
	fprintf (f, "  </tr>\n");

	for (rowx = write_range.row0; rowx <= write_range.rowi; rowx++)
	{
		if (print_all != 'A' && !row_is_visible (tab, rowx))
			continue;
		fprintf (f, "  <tr>\n");
		for (colx = write_range.col0; colx <= write_range.coli; colx++)
		{
			text = gtk_sheet_cell_get_text (tab->view->sheet, rowx, colx);
			fputs ("<td>", f);
			if (text)
			{
				fieldx = tab->file->col_to_field[colx];
				field = ql_get_fieldinfo (tab, fieldx);
				justification = field->justification;
				if (justification == GTK_JUSTIFY_CENTER)
					fputs ("<center>", f);
				else if (justification == GTK_JUSTIFY_RIGHT)
					fputs ("<right>", f);
				fprintf (f, "%s", text);
				if (justification == GTK_JUSTIFY_CENTER)
					fputs ("</center>", f);
				else if (justification == GTK_JUSTIFY_RIGHT)
					fputs ("</right>", f);
			}
			fputs ("</td>\n", f);
		}
		fprintf (f, "  </tr>\n");
	}
	fprintf (f, "</table>\n</body>\n</html>\n");
}								/* end of html output */

/**  Write the file onto disk with called char as a separator.  */
static void
write_delimited (QlTabData * tab, FILE * f, gchar separator)
{
	/** \todo replace this static buffer! */
	gchar linebuf[4096];
	gchar *text;
	gint32 rowx;
	gint16 colx;				/* indexes into array */
	gint x, y;
	gboolean first_in_row;

	/* NOW WRITE THE DATA RECORDS.  Last row is always empty. */
	for (rowx = write_range.row0; rowx < write_range.rowi; rowx++)
	{
		if (print_all != QL_PRINT_ALL && !row_is_visible (tab, rowx))
			continue;
		first_in_row = TRUE;
		for (colx = write_range.col0; colx <= write_range.coli; colx++)
		{
			text = gtk_sheet_cell_get_text (tab->view->sheet, rowx, colx);
			if (!first_in_row)
				fprintf (f, "%c", separator);
			first_in_row = FALSE;
			if (text)
			{
				x = y = 0;		/* remove separator characters from data */
				do
				{
					if (text[x] != separator)
						linebuf[y++] = text[x];
				}
				while (text[x++]);	/* stop at end of string */
				fprintf (f, "%s", linebuf);
			}
		}
		fprintf (f, "\n");		/* write nothing if nothing there */
	}
}								/* end of write_delimited */

/**  This puts all file types onto disk */
static void
write_all_types (QlTabData * tab, gchar * name)
{
	FILE * f;
	gchar * backup_name;
	gchar * fullpath;

	g_return_if_fail (tab);
	g_return_if_fail (tab->file->file_type > QL_NO_FILE);

	tab->file->file_name = name;
	fullpath = tab->file->file_path;

	backup_name = g_strconcat (fullpath, "~", NULL);
	rename (fullpath, backup_name);

	/* determine if whole file is to be written */
	write_range.row0 = write_range.col0 = 0;
	write_range.rowi = tab->file->last_row - 1;
	write_range.coli = tab->file->last_field;
	if (print_all == QL_PRINT_LIT)		/* highlighted */
		write_range = tab->view->sel_range;

	f = NULL;
	switch (tab->file->file_type)
	{
		case QL_OLD_QLF:
		{
			if (!g_str_has_suffix(fullpath, "qlf"))
			{
				fullpath = g_strconcat (fullpath, ".qlf", NULL);
			}
			if (!(f = fopen (fullpath, "w")))
			{
				level1_error (tab, _("Can't write the file.  Strange.")); 
				return;
			}
			/** \todo stop using f */
			write_ql_file (tab, f);
			tab->file->file_name = g_path_get_basename (fullpath);
			tab->file->file_path = fullpath;
			tab->file->changed = FALSE;
			break;
		}
		case QL_EXPORT_HTML:
		{
			if (!g_str_has_suffix(name, "html"))
			{
				name = g_strconcat (name, ".html", NULL);
			}
			if (!(f = fopen (name, "w")))
			{
				level1_error (tab, _("Can't write the file.  Strange.")); 
				return;
			}
			/** \todo stop passing f */
			write_html (tab, f);
			tab->file->file_name = name;
			break;
		}
		case QL_CSV:
		{
			if (!g_str_has_suffix(name, "csv"))
			{
				name = g_strconcat (name, ".csv", NULL);
			}
			if (!(f = fopen (name, "w")))
			{
				level1_error (tab, _("Can't write the file.  Strange.")); 
				return;
			}
			/** \todo stop passing f */
			write_delimited (tab, f, ',');
			tab->file->file_name = name;
			break;
		}
		case QL_TSV:
		{
			if (!g_str_has_suffix(name, "tsv"))
			{
				name = g_strconcat (name, ".tsv", NULL);
			}
			if (!(f = fopen (name, "w")))
			{
				level1_error (tab, _("Can't write the file.  Strange.")); 
				return;
			}
			/** \todo stop passing f */
			write_delimited (tab, f, '\t');
			tab->file->file_name = name;
			break;
		}
		default:
			break;
	}
	fclose (f);
	dim_all_menus (tab->qlc);
}								/* end of write_all_types */

static void
update_title (gchar G_GNUC_UNUSED * filename, QlTabData * tab)
{
	gint tbindex;
	GtkWidget * page;

	tbindex = gtk_notebook_get_current_page (tab->qlc->notebook);
	page = gtk_notebook_get_nth_page (tab->qlc->notebook, tbindex);
	gtk_notebook_set_tab_label (tab->qlc->notebook,
		page, tab_label_box(tab, tab->file->file_name));
	gtk_widget_show_all (GTK_WIDGET(tab->qlc->notebook));
	tab->view->dialog_mode = MODE_SAVE;
}

static void
any_save (QlTabData * tab)
{
	GtkWidget * chooser;
	GtkFileFilter * filter;
	gchar * filename;

	filename = NULL;
	g_return_if_fail (tab);
	if (tab->view->dialog_mode == MODE_SAVE)
	{
		filename = g_filename_to_utf8 (tab->file->file_path, -1, NULL, NULL, NULL);
		write_all_types (tab, filename);
		return;
	}
	if (tab->view->dialog_mode == MODE_NEW)
	{
		filter = gtk_file_filter_new ();
		gtk_file_filter_set_name (filter, _("QuickList files *.qlf"));
		gtk_file_filter_add_pattern (filter, "*.qlf");
		chooser = gtk_file_chooser_dialog_new ((_("Save this list")),
			GTK_WINDOW(tab->qlc->parent), GTK_FILE_CHOOSER_ACTION_SAVE,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, NULL);
		gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (chooser), 
			_("Untitled list"));
		gtk_file_chooser_set_do_overwrite_confirmation (
			GTK_FILE_CHOOSER (chooser), TRUE);
		gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(chooser), filter);
		if (gtk_dialog_run (GTK_DIALOG (chooser)) == GTK_RESPONSE_ACCEPT)
		{
			tab->view->dialog_mode = MODE_SAVE;
			filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
			tab->file->file_name = g_path_get_basename (filename);
			tab->file->file_path = filename;
			filename = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
			write_all_types (tab, filename);
			update_title (tab->file->file_name, tab);
		}
		gtk_widget_destroy (chooser);
	}
	if (tab->view->dialog_mode == MODE_EXPORT)
	{
		chooser = gtk_file_chooser_dialog_new
			((_("Export to a file")), GTK_WINDOW(tab->qlc->parent),
			GTK_FILE_CHOOSER_ACTION_SAVE,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT, NULL);
		gtk_file_chooser_set_do_overwrite_confirmation (
			GTK_FILE_CHOOSER (chooser), TRUE);
		ask_about_all (tab, chooser);
		file_type_menu (tab, chooser);		/* output file */
		tab->file->file_type = QL_EXPORT_HTML;
		if (gtk_dialog_run (GTK_DIALOG (chooser)) == GTK_RESPONSE_ACCEPT)
		{
			filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
			tab->file->file_path = filename;
			tab->file->file_name = g_path_get_basename (filename);
			filename = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
			write_all_types (tab, filename);
		}
		gtk_widget_destroy (chooser);
	}
}

/** This dialogue box is called by the close and quit routines to 
   just ask if the front file should be saved onto disk before
   the window is closed.

@return TRUE to continue with quicklist, FALSE to quit.

\bug should indicate what kind of data is new:
fields, sort patterns, filters or reports. */
gboolean
ask_about_save (QlTabData * tab)
{
	GtkWidget * ask_dlg;
	gboolean ret;
	gint result;

	ret = FALSE;
	/* no tabs, no data to save */
	if (!tab)
		return FALSE;
	if (gtk_notebook_get_current_page (tab->qlc->notebook) < 0)
		return FALSE;
	if (!check_if_changed(tab))
		return FALSE;
	ask_dlg = gtk_message_dialog_new (GTK_WINDOW(tab->qlc->parent),
		GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
		GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE,
		"%s", _("This file has been changed. "
			"Do you want to save it to disk?"));
	gtk_dialog_add_buttons (GTK_DIALOG(ask_dlg),
		GTK_STOCK_YES, GTK_RESPONSE_YES,
		GTK_STOCK_NO,  GTK_RESPONSE_NO,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL);
	
	result = gtk_dialog_run (GTK_DIALOG (ask_dlg));
	switch (result)
	{
		case GTK_RESPONSE_YES:
		{
			/* if user said Yes, save the file */
			any_save (tab);
			break;
		}
		case GTK_RESPONSE_NO:
		{
			gint tab_index;
			tab_index = gtk_notebook_get_current_page (tab->qlc->notebook);
			gtk_notebook_remove_page (tab->qlc->notebook, tab_index);
			if (tab->file->file_path)
				g_free (tab->file->file_path);
			tab->file->file_path = NULL;
			break;
		}
		case GTK_RESPONSE_CANCEL :
			ret = TRUE;
			break;
	}
	gtk_widget_destroy (ask_dlg);
	return ret;
}								/* End of ask_about_save */

void
close_file (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	gint tab_index;
	QlContext * qlc;
	QlTabData * tab;

	g_return_if_fail (data);
	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	g_return_if_fail (tab);
	if (check_if_changed (tab))
	{
		close_mode = QL_CLOSE;
		/* ask_about_save will save or close the tab. */
		ask_about_save (tab);
		return;
	}
	tab_index = gtk_notebook_get_current_page (tab->qlc->notebook);
	gtk_notebook_remove_page (tab->qlc->notebook, tab_index);
	if (tab->file->file_path)
		g_free (tab->file->file_path);
	tab->file->file_path = NULL;
}								/* End of close_file callback */

gboolean
exit_action (GtkAction G_GNUC_UNUSED * a, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;
	gboolean more;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	close_mode = QL_QUIT;
	more = ask_about_save (tab);
	if (!more)
		gtk_main_quit ();
	return TRUE;
}

gboolean
file_exit (GtkWidget * w, GdkEvent G_GNUC_UNUSED * event, gpointer G_GNUC_UNUSED data)
{
	QlContext * qlc;
	QlTabData * tab;
	gboolean more;

	qlc = ql_get_context (GTK_WIDGET(w));
	tab = ql_get_tabdata (qlc);
	close_mode = QL_QUIT;
	more = ask_about_save (tab);
	if (!more)
		gtk_main_quit ();
	return TRUE;
}								/* End of exit routine */

void
file_export (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	tab->view->dialog_mode = MODE_EXPORT;
	any_save (tab);
}

void
save_as (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	g_return_if_fail (data);
	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	print_all = QL_PRINT_ALL;
	any_save (tab);
}

/** Write the file onto disk with its present file path  */
void
save_file (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	print_all = QL_PRINT_ALL;
	if ((tab->file->current_file == QL_NO_FILE) ||
		(tab->view->dialog_mode == MODE_NEW))
	{
		tab->file->current_file = tab->file->file_type = QL_OLD_QLF;
		tab->view->dialog_mode = MODE_NEW;
		any_save (tab);
	}
	else
	{
		tab->file->file_type = QL_OLD_QLF;
		tab->view->dialog_mode = MODE_SAVE;
		save_as (NULL, data);
	}
}
