/***************************************************************************
 *            main.h
 *
 *  Fri Aug 25 15:12:00 2006
 *  Copyright  2006-2007  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
 
#ifndef _MAIN_H
#define _MAIN_H

/** @file main.h
	@brief Temporary home for common routines.
	@author Copyright 2006, 2007  Neil Williams <linux@codehelp.co.uk>
	@author Copyright 1999  Robert Lissner 
*/

#define QLCONTEXT "QuickListContextID"

/** \brief Replacement QuickList date formats.

These seven formats exist for compatibility with 
QuickList prior to v0.8.6 - subsequent versions use 
standard strftime formats and are therefore localised.

Old style QLF files use the old formats but the values 
are now converted to GDate internally.
*/
typedef enum
{
	/* % symbols need to be escaped for doxygen */
	/** 12/31/99 \%m/\%d\%y New : QL_DF_ISO99 OLD: "M /D0/Y2" */
	QL_DF_USSHORT = 0,
	/** 12/31/1999 \%m/\%d/\%Y QL_DF_USLONG OLD: "M /D0/Y4" */
	QL_DF_USLONG,
	/** 12.31.99 \%m.\%d.\%y QL_DF_USSHORT_2 OLD: "M .D0.Y2" */
	QL_DF_USSHORT_2,
	/** 12.31.1999 \%m.\%d.\%Y QL_DF_USLONG_2 OLD: "M .D0.Y4" */
	QL_DF_USLONG_2,
	/** Dec 31 1999 \%b \%m \%y QL_DF_LOCAL_MONTH OLD: "MA D  Y4" */
	QL_DF_LOCAL_MONTH,
	/** 1999-12-31 \%Y-\%m-\%d QL_DF_UTC OLD: "Y4-M0-D0" */
	QL_DF_LONGYEAR,
	/** 31 Dec 1999 \%m \%b \%Y QL_DF_LOCAL_DAY OLD: "D  MA Y4" */
	QL_DF_LOCAL_DAY,
	/** 31/12/99 \%d/\%m/\%y QL_DF_EURSHORT OLD: "D /M0/Y2" */
	QL_DF_EURSHORT,
	/** 31/12/1999 \%d/\%m/\%Y QL_DF_EURLONG OLD: "D /M0/Y4" */
	QL_DF_EURLONG,
	/** unused check value */
	QL_DF_LAST
} QlDateFormat;
/** @} */

/** \name New QuickList standard formats

Automatically translated using strftime, parsed
by GDate and directly compatible with XML and xsd:date.
The new formats allow dates (and times) to be handled internally
in a consistent manner and displayed to the user in standard
formats. This list can be easily extended.
@{
*/
/** 12/31/99 - same as QL_DF_USSHORT */
#define QL_DF_ISO99 "%m/%d/%y"
/** 31/12/99 %d/%m/%y */
#define QL_DF_UK "%d/%m/%y"
/** 1999-12-31 same as %F - also used by xsd:date */
#define QL_DF_ISO8601 "%Y-%m-%d"
/** Sat, 11 Nov 2006 %a, %d %b %Y */
#define QL_DF_RFC822 "%a, %d %b %Y"
/** 31.12.2006 %d.%m.%Y */
#define QL_DF_CE "%d.%m.%Y"
/** 12.31.2006 %m.%d.%Y */
#define QL_DF_US "%m.%d.%Y"
/** Locale-specific %x - for en_GB this is: 31/12/06 */
#define QL_DF_LOCALE "%x"

/** @} */

/** \brief Replacement QuickList time formats.

These six formats exist for compatibility with 
QuickList prior to v0.8.6 - subsequent versions use 
standard strftime formats and are therefore localised.

Old style QLF files use the old formats but the values 
are now converted to GDate internally.
*/
typedef enum 
{
	/** "6:00 PM" \%l:\%M \%p */
	QL_TF_12P = 0,
	/** "18:00" \%R */
	QL_TF_24,
	/** "18.00" \%H.\%M */
	QL_TF_DOT24,
	/** "18:00pm \%H:\%M\%P */
	QL_TF_24P,
	/** "23:53:37" \%T */
	QL_TF_T,
	/** "11:53:37 PM \%l:\%M:\%S \%p */
	QL_TF_12TP,
	/** unused check value */
	QL_TF_LAST
}QlTimeFormat;

/** \name New QuickList standard formats

Automatically translated using strftime, parsed
by GDate and directly compatible with XML and xsd:time.
The new formats allow dates (and times) to be handled internally
in a consistent manner and displayed to the user in standard
formats. This list can be easily extended.
@{
*/
#define QL_TF_LOCALE "%X"

#define QL_TF_SECS   "%s"

/** @} */

/** \brief Replacement QuickList currency formats.

These seven formats exist for compatibility with 
QuickList prior to v0.8.6 - subsequent versions use 
standard strmon formats and are all therefore localised.

Old style QLF files use the old formats but the values 
are now converted internally.

The names should be descriptive, according to this formula:
QL_NF prefix
OLD if the format is deprecated and only used with .qlf files
PAR if negative values are enclosed in parentheses
CUR if the national currency symbol is used e.g. $ or £
INT if the international currency symbol is used e.g. USD or GBP
GRP if thousands separators are used (grouping), e.g. 1234.56 vs 1,234.56
*/
typedef enum 
{
	/** 1,234.56 \%!i simple grouped double */
	QL_NF_GRP = 0,
	/** 1.234,56 used for files only */
	QL_NF_OLD_A,
	/** 1234.56  \%^!i ungrouped double */
	QL_NF,
	/** "1234,56" used for files only */
	QL_NF_OLD_B,
	/** $1,234.56  default monetary display (\%n) */
	QL_NF_CURR,
	/** 1,234.56% \%!i\%\% default percentage display */
	QL_NF_PERCENT,
	/** 1.234,56% used for files only */
	QL_NF_OLD_C,
	/** unused check value */
	QL_NF_LAST
}QlCurrFormat;

/** \name New specifiers 

\todo Like the new date and time specifiers - work
out how to fold these into the options in the New dialogue,
once a suitable file format is available. QLF cannot support
these specifiers, yet.

@{
*/

/** %(n		(£12,345.99) */
#define QL_NF_PAR_CUR_GRP "%(n"

/** %^(n	(£12345.99) */
#define QL_NF_PAR_CUR     "%^(n"

/** %(!i	(12,345.99) */
#define QL_NF_PAR_GRP     "%(!i"

/** %i      12,345.99GBP */
#define QL_NF_INT_GRP     "%i"

/** %i(		(12,345.99)GBP */
#define QL_NF_PAR_INT_GRP "%i("

/** %^i(	(12345.99)GBP */
#define QL_NF_PAR_INT     "%^i("

/** %^i		12345.99GBP */
#define QL_NF_INT         "%^i"

/** @} */

#define CHECK_RANGE if (tab->view->sel_range.col0 > tab->file->last_field) return

#define CHECK_CHANGED(x) if (check_if_changed (x)) return

/** Maximum length of a date string. */
#define MAX_DATE_BUFFER   256

/** \brief compatibility function

Converts old-style QlDateFormat values into strftime-compatible
format strings.

\param df Old style QlDateFormat value to convert.

\return A strftime-compatible format string for the specified
QlDateFormat
*/
const gchar *
convert_old_df (QlDateFormat df);

const gchar *
convert_old_tf (QlTimeFormat tf);

const gchar *
convert_old_cf (QlCurrFormat cf);


/** \brief Display dates using old and new types

\param gd A valid GDate containing the date to display.
\param df An old style QlDateFormat or zero.
\param fmt A new style strftime-compatible format string or NULL.

When using an old style QlDateFormat, pass a NULL fmt string.

When using a new style format string, df is ignored.

\return A newly allocated string containing the formatted date
or NULL on error.
*/
gchar *
display_date (GDate * gd, const gchar * fmt);

gchar *
display_time (GTimeVal * gt, const gchar * fmt);

/** \brief Parse an incoming date string - any type.

Tries to parse any date string (time values are \b not
supported) to create a GDate.

\param string A user-specified or QuickList file date string.

\return a valid GDate containing the parsed date or NULL on error.
*/
GDate *
parse_date (const gchar * string, const gchar * format);

/** \brief The QuickList Context

Configuration data common to all notebook tabs
*/
typedef struct
{
	/** the toplevel parent widget */
	GtkWidget * parent;
	/** the single notebook for this instance */
	GtkNotebook * notebook;
	/** table of tab specific data, 
	indexed by GINT_TO_POINTER (tab_index). */
	GHashTable  * tab_table;
	/** Only one menu manager per instance. */
	GtkUIManager * menu_manager;
	/** Only one recent files list per instance. */
	GtkRecentManager *recent_manager;
}QlContext;

/** \brief Context for each notebook tab. */
typedef struct
{
	/** The file data: the model. */
	QlFileData * file;
	/** The view data: the view. */
	QlViewData * view;
	/** Back pointer to the context data: the controller. */
	QlContext * qlc;
} QlTabData;

/** initialise the context */
QlContext * 
ql_context_init (void);

void add1row (QlTabData * tab); 
void big_draw_start (QlTabData * tab);
void big_draw_end (QlTabData * tab);
gboolean check_if_changed (QlTabData * tab);
gint check_entry (QlTabData * tab, gchar *entry);
void connect_signals (QlTabData * tab);
gint d2qls (gchar *texto, gdouble input, QlFieldType type, 
	QlDateFormat formatting, gint dec_places);
void close_dlg (GtkWidget * w, gpointer data);
void front_is_changed (QlTabData * tab);
void get_window_size_loc (GtkWidget * win);
void level1_error (QlTabData * tab, gchar *message);
void level2_error (QlTabData * tab, gchar *message);
void make_basic_dialog1 (void);
gdouble qls2d (const gchar *input, gint type, gint formatting);
void reset_col_to_field (QlTabData * tab);

void set_window_size_loc (GtkWidget * win);

void unselect (QlTabData * tab);

#endif /* _MAIN_H */
