#!/usr/bin/perl -w
#
# dbupgrade.pl
# This script generates the SQL commands for the database alterations needed
# when upgrading versions of ratbox-services.
#
# It takes the version of ratbox-services as an argument, eg:
#   ./dbupgrade.pl ratbox-services-1.0.3
# You may leave off the "ratbox-services-" if you wish.  You should NOT
# however leave off extra version information like "rc1".
#
# Note, this script will only deal with actual releases, not svn copies.
# 
# Copyright (C) 2006 Lee Hardy <lee -at- leeh.co.uk>
# Copyright (C) 2006 ircd-ratbox development team
#
# $Id: dbupgrade.pl 23043 2006-09-09 15:41:27Z leeh $

use strict;

require "definetolength.pl";

unless($ARGV[0] && $ARGV[1])
{
	print "Usage: dbupgrade.pl <ratbox-services version> <sqlite|mysql|pgsql> [include_path]\n";
	print "Eg, ./dbupgrade.pl 1.0.3 sqlite\n";
	exit;
}

unless(-r "../include/setup.h")
{
	print("Unable to read ../include/setup.h, please run configure first\n");
	exit();
}

my %versionlist = (
	"1.0.0"		=> 1,
	"1.0.1"		=> 1,
	"1.0.2"		=> 1,
	"1.0.3"		=> 1,
	"1.1.0beta1"	=> 2,
	"1.1.0beta2"	=> 2,
	"1.1.0beta3"	=> 2,
	"1.1.0beta4"	=> 3,
	"1.1.0rc1"	=> 4
);

my $version = $ARGV[0];
my $dbtype = $ARGV[1];
my %vals;

$version =~ s/^ircd-ratbox-//;

my $currentver = $versionlist{"$version"};

if(!$currentver)
{
	print "Unknown version $version\n";
	exit;
}

if($dbtype ne "sqlite" && $dbtype ne "mysql" && $dbtype ne "pgsql")
{
	print "Unknown database type $dbtype\n";
	exit;
}

if($ARGV[2])
{
	%vals = &parse_includes("$ARGV[2]");
}
else
{
	%vals = &parse_includes("../include");
}

while(my ($key, $value) = each(%vals))
{
	if($value == 1)
	{
		print "Unable to set $key -- include path must be wrong.\n";
		exit;
	}
}

if($currentver < 2)
{
	print "-- To version 1.1.0beta1\n";

	if($dbtype eq "sqlite")
	{
		print "CREATE TABLE users_resetpass (\n";
		print "    username TEXT, token TEXT, time INTEGER,\n";
		print "    PRIMARY KEY(username)\n";
		print ");\n";
		print "CREATE TABLE users_sync (\n";
		print "    id INTEGER PRIMARY KEY, hook TEXT, data TEXT\n";
		print ");\n";
		print "ALTER TABLE users ADD COLUMN verify_token TEXT;\n";
		print "ALTER TABLE users ADD COLUMN suspend_reason TEXT;\n";
		print "ALTER TABLE channels ADD COLUMN suspend_reason TEXT;\n";
	}
	elsif($dbtype eq "mysql")
	{
		print "CREATE TABLE users_sync (\n";
		print "    id INTEGER AUTO_INCREMENT, hook VARCHAR(50) NOT NULL, data TEXT,\n";
		print "    PRIMARY KEY(id)\n";
		print ");\n";
	}
	else
	{
		print "CREATE TABLE users_sync (\n";
		print "    id SERIAL, hook VARCHAR(50) NOT NULL, data TEXT,\n";
		print "    PRIMARY KEY(id)\n";
		print ");\n";
	}

	if($dbtype eq "mysql" || $dbtype eq "pgsql")
	{
		print "CREATE TABLE users_resetpass (\n";
		print "    username VARCHAR(".$vals{"USERREGNAME_LEN"}.") NOT NULL, token VARCHAR(10), time INTEGER,\n";
		print "    PRIMARY KEY(username)\n";
		print ");\n";
		print "ALTER TABLE users ADD COLUMN verify_token VARCHAR(8);\n";
		print "ALTER TABLE users ADD COLUMN suspend_reason VARCHAR(".$vals{"SUSPENDREASONLEN"}.");\n";
		print "ALTER TABLE channels ADD COLUMN suspend_reason VARCHAR(".$vals{"SUSPENDREASONLEN"}.");\n";
	}


	print "CREATE TABLE global_welcome (\n";
	print "    id INTEGER, text TEXT,\n";
	print "    PRIMARY KEY(id)\n";
	print ");\n";

	print "\n";
}

if($currentver < 3)
{
	print "-- To version 1.1.0beta4\n";

	if($dbtype eq "mysql" || $dbtype eq "pgsql")
	{
		print "CREATE TABLE channels_dropowner (\n";
		print "    chname VARCHAR(".$vals{"CHANNELLEN"}.") NOT NULL, token VARCHAR(10), time INTEGER,\n";
		print "    PRIMARY KEY(chname)\n";
		print ");\n";
		print "CREATE TABLE users_resetemail (\n";
		print "    username VARCHAR(".$vals{"USERREGNAME_LEN"}.") NOT NULL, token VARCHAR(10),\n";
		print "    email VARCHAR(".$vals{"EMAILLEN"}.") DEFAULT NULL, time INTEGER,\n";
		print "    PRIMARY KEY (username)\n";
		print ");\n";
	}
	else
	{
		print "CREATE TABLE channels_dropowner (\n";
		print "    chname TEXT, token TEXT, time INTEGER,\n";
		print "    PRIMARY KEY(chname)\n";
		print ");\n";
		print "CREATE TABLE users_resetemail (\n";
		print "    username TEXT, token TEXT, email TEXT DEFAULT NULL, time INTEGER,\n";
		print "    PRIMARY KEY(username)\n";
		print ");\n";
	}

	if($dbtype eq "mysql")
	{
		print "ALTER TABLE channels_dropowner ADD INDEX (time);\n";
		print "ALTER TABLE users_resetpass ADD INDEX (time);\n";
		print "ALTER TABLE users_resetemail ADD INDEX (time);\n";
	}
	elsif($dbtype eq "pgsql")
	{
		print "CREATE INDEX dropowner_time_idx ON channels_dropowner (time);\n";
		print "CREATE INDEX resetpass_time_idx ON users_resetpass (time);\n";
		print "CREATE INDEX resetemail_time_idx ON users_resetemail (time);\n";
	}

	print "UPDATE operserv SET chname=LOWER(chname);\n";
	print "UPDATE operbot SET chname=LOWER(chname);\n";
	print "UPDATE operbans SET mask=LOWER(mask);\n";
	print "\n";
}

if($currentver < 4)
{
	print "-- To version 1.1.0rc1\n";

	if($dbtype eq "mysql" || $dbtype eq "pgsql")
	{
		print "CREATE TABLE email_banned_domain (\n";
		print "    domain VARCHAR(255) NOT NULL,\n";
		print "    PRIMARY KEY(domain)\n";
		print ");\n";
	}
	else
	{
		print "CREATE TABLE email_banned_domain (\n";
		print "    domain TEXT NOT NULL,\n";
		print "    PRIMARY KEY(domain)\n";
		print ");\n";
	}

	print "\n";
}

if($currentver < 5)
{
	print "-- To version 1.1.0rc2\n";

	if($dbtype eq "mysql")
	{
		print "ALTER TABLE members ADD PRIMARY KEY(chname, username);\n";
		print "ALTER TABLE members ADD INDEX (chname);\n";
		print "ALTER TABLE members ADD INDEX (username);\n";
		print "ALTER TABLE bans ADD PRIMARY KEY(chname, mask);\n";
		print "ALTER TABLE bans ADD INDEX (chname);\n";
		print "UPDATE channels SET url=SUBSTR(url,1,".$vals{"URLLEN"}.");\n";
		print "ALTER TABLE channels CHANGE url url VARCHAR(".$vals{"URLLEN"}.");\n";
	}
	elsif($dbtype eq "pgsql")
	{
		print "ALTER TABLE members ADD PRIMARY KEY(chname, username);\n";
		print "CREATE INDEX members_chname_idx ON members (chname);\n";
		print "CREATE INDEX members_username_idx ON members (username);\n";
		print "ALTER TABLE bans ADD PRIMARY KEY(chname, mask);\n";
		print "CREATE INDEX bans_chname_idx ON bans (chname);\n";
		print "UPDATE channels SET url=SUBSTR(url,1,".$vals{"URLLEN"}.");\n";
		print "ALTER TABLE channels ALTER COLUMN url TYPE VARCHAR(".$vals{"URLLEN"}.");\n";
	}

	print "\n";
}

exit;
