/******************************************************************************
*		       							      *
* config.c (part of rCalc)					       	      *
* Copyright 2000 Gary Benson <rat@spunge.org>			       	      *
*								       	      *
* This program is free software; you can redistribute it and/or modify 	      *
* it under the terms of the GNU General Public License as published by 	      *
* the Free Software Foundation; either version 2 of the License, or    	      *
* (at your option) any later version.				       	      *
*								       	      *
* This program is distributed in the hope that it will be useful,      	      *
* but WITHOUT ANY WARRANTY; without even the implied warranty of       	      *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	       	      *
* GNU General Public License for more details.			       	      *
*								       	      *
* You should have received a copy of the GNU General Public License    	      *
* along with this program; if not, write to the Free Software	       	      *
* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.	       	      *
*								       	      *
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <signal.h>
#include <gnome.h>

#include "config.h"
#include "console.h"

/* The complete state of the calculation engine;
** this is shared between both processes and must
** be locked/unlocked before and after use.
*/
static UISTATE uistate;
CALCSTATE *cfgCalc;
UISTATE   *cfgUI = &uistate;

/*****************************************************************************/

/* Some of the less convenient defaults
*/
#define DEFAULT_FONT	"-*-courier-medium-r-*-*-*-120-*-*-*-*-*-*"

int defaultPalette[16][3]=
{
	/* 0 */ { 0xFFFF, 0xFFFF, 0xFFFF },	/* Background		     */
	/* 1 */ { 0xCF3C, 0x0000, 0x0000 },	/* Errors		     */
	/* 2 */ { 0x0000, 0xCF3C, 0x0000 },	/* Debugging messages	     */
	/* 3 */ { 0x0000, 0x0000, 0x0000 },
	/* 4 */ { 0x0000, 0x0000, 0xCF3C },	/* Prompt		     */
	/* 5 */ { 0x0000, 0x0000, 0x0000 },
	/* 6 */ { 0x0000, 0x0000, 0x0000 },
	/* 7 */ { 0x0000, 0x0000, 0x0000 },	/* Foreground		     */
	/* 8 */ { 0x0000, 0x0000, 0x0000 },
	/* 9 */ { 0x0000, 0x0000, 0x0000 },
	/* A */ { 0x0000, 0x0000, 0x0000 },
	/* B */ { 0x0000, 0x0000, 0x0000 },
	/* C */ { 0x0000, 0x0000, 0x0000 },
	/* D */ { 0x0000, 0x0000, 0x0000 },
	/* E */ { 0x0000, 0x0000, 0x0000 },
	/* F */ { 0x0000, 0x0000, 0x0000 },
};

/* Load and save the configuration, using GNOME function calls.
*/
void rCalc_config_load( void )
{
	char *pos,*mod,*name;
	char *guns="rgb";
	char line[32];
	int i,j;

	/* User interface settings.
	*/
	gnome_config_push_prefix( "/rCalc/UI/" );

	pos = gnome_config_get_string( "scrollbarPos=right" );
	if( !strcasecmp( pos, "left" ) )
		cfgUI->scrollbarPos = scrollbarLeft;
	else if( !strcasecmp( pos, "right" ) )
		cfgUI->scrollbarPos = scrollbarRight;
	else
		cfgUI->scrollbarPos = scrollbarHidden;
	g_free( pos );

	cfgUI->termBell        = gnome_config_get_bool( "termBell=true" );
	cfgUI->cursBlink       = gnome_config_get_bool( "cursBlink=false" );
	cfgUI->scrollbackLines = gnome_config_get_int(  "scrollbackLines=100");
	cfgUI->scrollOnKey     = gnome_config_get_bool( "scrollOnKey=true" );
	cfgUI->scrollOnOut     = gnome_config_get_bool( "scrollOnOut=true" );

	cfgUI->fontName = gnome_config_get_string( "fontName=" DEFAULT_FONT );
	for( i=0; i<16; i++ )
	{
		for( j=0; j<3; j++ )
		{
			sprintf( line, "colour%d%c=%d", i, guns[j],
				 defaultPalette[i][j] );
			cfgUI->palette[j][i] = gnome_config_get_int( line );
		}
	}
	gnome_config_pop_prefix();
	
	/* Calculation engine state.
	*/
	gnome_config_push_prefix( "/rCalc/engine/" );	

	mod = gnome_config_get_string( "angleMode=deg" );
	if( !strcasecmp( pos, "rad" ) )
		cfgCalc->angleMode = angleRadians;
	else
		cfgCalc->angleMode = angleDegrees;
	g_free( mod );

	cfgCalc->nVariables = gnome_config_get_int( "nVariables=0" );
	for( i=0; i<cfgCalc->nVariables; i++ )
	{
		sprintf( line, "vName%d=%s", i, "<error>" );
		name = gnome_config_get_string( line );
		strcpy( cfgCalc->variables[i].name, name );
		g_free( name );

		sprintf( line, "vValue%d=%s", i, "0" );
		cfgCalc->variables[i].value = gnome_config_get_float( line );
	}
	
	gnome_config_pop_prefix();
}
void rCalc_config_save( void )
{
	char *guns="rgb";
	char line[32];
	int i,j;

	/* User interface settings.
	*/
	gnome_config_push_prefix( "/rCalc/UI/" );

	gnome_config_set_string( "scrollbarPos",
				 cfgUI->scrollbarPos==scrollbarLeft ? "left" :
				 cfgUI->scrollbarPos==scrollbarRight ? "right":
				 "hidden" );

	gnome_config_set_bool( "termBell", cfgUI->termBell );
	gnome_config_set_bool( "cursBlink", cfgUI->cursBlink );
	gnome_config_set_int(  "scrollbackLines", cfgUI->scrollbackLines );
	gnome_config_set_bool( "scrollOnKey", cfgUI->scrollOnKey );
	gnome_config_set_bool( "scrollOnOut", cfgUI->scrollOnOut );

	gnome_config_set_string( "fontName", cfgUI->fontName );
	g_free( cfgUI->fontName );

	for( i=0; i<16; i++ )
	{
		for( j=0; j<3; j++ )
		{
			sprintf( line, "colour%d%c", i, guns[j] );
			gnome_config_set_int( line, cfgUI->palette[j][i] );
		}
	}
	gnome_config_pop_prefix();
	
	/* Calculation engine state.
	*/
	gnome_config_push_prefix( "/rCalc/engine/" );	

	gnome_config_set_string( "angleMode",
				 cfgCalc->angleMode==angleRadians ? "rad" :
				 "deg" );
	gnome_config_set_int( "nVariables", cfgCalc->nVariables );
	for( i=0; i<cfgCalc->nVariables; i++ )
	{
		sprintf( line, "vName%d", i  );
		gnome_config_set_string( line, cfgCalc->variables[i].name );

		sprintf( line, "vValue%d", i );
		gnome_config_set_float( line, cfgCalc->variables[i].value );
	}
	
	gnome_config_pop_prefix();

	gnome_config_sync();
}

/*****************************************************************************/

/* From the man page of semctl(2)
*/
#if defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)
/* union semun is defined by including <sys/sem.h> */
#else
/* according to X/OPEN we have to define it ourselves */
union semun {
	int val;                    /* value for SETVAL */
	struct semid_ds *buf;       /* buffer for IPC_STAT, IPC_SET */
	unsigned short int *array;  /* array for GETALL, SETALL */
	struct seminfo *__buf;      /* buffer for IPC_INFO */
};
#endif

static int shmID;
static int semID;

/* Initialise and free shared memory segment and semaphores
** for the calculator's internal configuration.
*/
void rCalc_config_init( void )
{
	union semun arg;

	/* Do the shared memory
	*/
	if(( shmID = shmget( IPC_PRIVATE, sizeof(CALCSTATE),
			     0600|IPC_CREAT|IPC_EXCL )) == -1 )
	{
		perror( "shmget" );
		exit(EXIT_FAILURE);
	}
	if((int)(cfgCalc = shmat( shmID, NULL, 0 )) == -1 )
	{
		perror( "shmat" );
		exit(EXIT_FAILURE);
	}

	/* Do the semaphore
	*/
	if(( semID = semget( IPC_PRIVATE, 1, 0600|IPC_CREAT|IPC_EXCL )) == -1 )
	{
		perror( "semget" );
		exit(EXIT_FAILURE);
	}
	arg.val = 1;
	if( semctl( semID, 0, SETVAL, arg )==-1 )
	{
		perror( "semctl" );
		exit(EXIT_FAILURE);
	}
}

void rCalc_config_free( void )
{
	union semun arg;

	if(( shmctl( shmID, IPC_RMID, NULL ) )== -1 )
	{
		perror( "shmctl" );
	}
	if( semctl( semID, 0, IPC_RMID, arg )== -1 )
	{
		perror( "semctl" );
	}
}

/*****************************************************************************/

static void moveSemaphore( int direction )
{
	struct sembuf buf;

	buf.sem_num = 0;
	buf.sem_op  = direction;
	buf.sem_flg = SEM_UNDO;

	if(( semop( semID, &buf, 1 ))==-1 )
	{
		perror( "semop" );
		exit(EXIT_FAILURE);
	}
}

/* Lock and unlock the shared memory.
*/
void rCalc_config_Lock( void )
{
	Debug( "%s", __FUNCTION__ );
	moveSemaphore( -1 );
}
void rCalc_config_Unlock( void )
{
	Debug( "%s", __FUNCTION__ );
	moveSemaphore( 1 );
}

void rCalc_config_NotifyChange( void )
{
	pid_t pidParent;
	
	Debug( "%s", __FUNCTION__ );
	pidParent = getppid();
	kill( pidParent, SIGUSR1 );
}

/*** end of config.c *********************************************************/

