#!@@PERL@@ @@PERLOPTS@@

# Copyright 1999, 2000, 2001 (c) Thomas Erskine <@@AUTHOR@@>
# See the COPYRIGHT file with the distribution.

# log-collector - a remstats collector for remote logs
# $Id: log-collector.pl,v 1.12 2001/08/31 21:07:20 remstats Exp $

# - - -   Configuration   - - -

use strict;

# What is this program called, for error-messages and file-names
$main::prog = 'log-collector';
# Which collector is this
$main::collector = 'log';
# Where is the default configuration dir
$main::config_dir = '@@CONFIGDIR@@';
# Which port is the log-server sitting on
$main::port = 1958;
# How long to wait for a response
$main::timeout = 10; # seconds

# - - -   Version History   - - -

(undef, $main::version) = split(' ', '$Revision: 1.12 $');

# - - -   Setup   - - -

use lib '.', '@@LIBDIR@@', '@@RRDLIBDIR@@';
require "remstats.pl";
use Getopt::Std;
use RRDs;
require "socketstuff.pl";

# Parse the command-line
my %opt = ();
my @hosts;
getopts('d:f:FhH:p:t:u', \%opt);

if (defined $opt{'h'}) { &usage; } # no return
if (defined $opt{'d'}) { $main::debug = $opt{'d'}; } else { $main::debug = 0; }
if (defined $opt{'f'}) { $main::config_dir = $opt{'f'}; }
if (defined $opt{'F'}) { $main::force_collection = 1; } else { $main::force_collection = 0; }
if( defined $opt{'H'}) { @hosts = split(',', $opt{'H'}); }
if (defined $opt{'p'}) { $main::port = $opt{'p'}; }
if (defined $opt{'t'}) { $main::timeout = $opt{'t'}+0; }
if (defined $opt{'u'}) { $main::use_uphosts = 0; } else { $main::use_uphosts = 1; }

&read_config_dir($main::config_dir, 'general', 'oids', 'rrds', 'groups', 
	'host-templates', 'hosts');
%main::uphosts = &get_uphosts if ($main::use_uphosts);

# No buffering
$| = 1;

# - - -   Mainline   - - -

my ($host, $hosts, $ip, $realrrd, $wildrrd, $wildpart, $logfile, $fixedrrd,
	$tmpfile, $lastfile, $start_time, $run_time);
$start_time = time();
$tmpfile = $main::config{DATADIR} .'/LAST/'. $main::collector .'.'. $$;
$lastfile = $main::config{DATADIR} .'/LAST/'. $main::collector;
$main::entries_collected = $main::entries_used = $main::requests = 0;

open (TMP, ">$tmpfile") or &abort("can't open $tmpfile: $!");

unless( defined @hosts) { @hosts = keys %{$main::config{HOST}}; }

foreach $host (@hosts) {
	next if ($host eq '_remstats_');
	if ($main::use_uphosts and not defined $main::uphosts{$host}) {
		&debug("$host is down(uphosts); skipped") if ($main::debug);
		next;
	}
	$ip = &get_ip($host);
	next unless (defined $ip);
	unless (defined $main::config{HOSTCOLLECTEDBY}{$main::collector}{$host}) {
		&debug("  $host isn't collected by $main::collector; skipped") if ($main::debug>1);
		next;
	}
	$main::data_from_host = 0;

# collect the data from the remote collector first
	foreach $realrrd (@{$main::config{HOST}{$host}{RRDS}}) {
		($wildrrd, $wildpart, undef, $fixedrrd) = &get_rrd($realrrd);
		next unless (defined $main::config{COLLECTOR}{$main::collector}{$wildrrd});
		$logfile = $main::config{HOST}{$host}{EXTRA}{$realrrd};
		unless (defined $logfile) {
			&error("$realrrd for $host has no logfile specified");
			next;
		}
		unless ($main::force_collection or 
				&check_collect_time($host, $wildrrd, $fixedrrd)) {
			&debug("  not time yet for $realrrd($wildrrd): skipped") if ($main::debug>1);
			next;
		}
		&collect_rrd($host, $realrrd, $wildrrd, $wildpart, $logfile);
	}
}

# Now remstats instrumentation info
my $now = time;
$run_time = $now - $start_time;
print <<"EOD_INSTRUMENTATION";
_remstats_ $now ${main::collector}-collector:requests $main::requests
_remstats_ $now ${main::collector}-collector:collected $main::entries_collected
_remstats_ $now ${main::collector}-collector:used $main::entries_used
_remstats_ $now ${main::collector}-collector:runtime $run_time
EOD_INSTRUMENTATION

close(TMP) or &abort("can't open $tmpfile: $!");
rename $tmpfile, $lastfile or &abort("can't rename $tmpfile to $lastfile: $!");

exit 0;

#----------------------------------------------------------------- usage ---
sub usage {
	print STDERR <<"EOD_USAGE";
$main::prog version $main::version
usage: $0 [options]
where options are:
   -d nnn  enable debugging output at level 'nnn'
   -f fff  use config-dir 'fff'[$main::config_dir]
   -F      force collection, even if it's not time
   -h      show this help
   -H HHH  only do hosts from 'HHH', a comma-separated list
   -p ppp  contact log-server on port 'ppp' [$main::port]
   -t ttt  timeout each port attempt after 'ttt' seconds [$main::timeout]
   -u      ignore uphosts file
EOD_USAGE
	exit 0;
}

#----------------------------------------------------------------- debug ---
sub debug {
	my ($msg) = @_;

	if ($main::debug) { print STDERR "DEBUG: $msg\n"; }
0;
}

#------------------------------------------------------------------ abort ---
sub abort {
	my ($msg) = @_;
	print STDERR "$main::prog: ABORT: $msg\n";
	exit 1;
}

#------------------------------------------------------------------- error ---
sub error {
	my ($msg) = @_;
	print STDERR "$main::prog: ERROR: $msg\n";
}

#----------------------------------------------------------- collect_rrd ---
sub collect_rrd {
	my ($host, $realrrd, $wildrrd, $wildpart, $logfile) = @_;
	my ($line, $variable, $value, $data, $extra, $now, $socket, $status);
	my $timeout = $main::timeout;

	&debug("doing host $host") if ($main::debug);

	($socket, $status, $timeout) = &open_socket( $host, $main::port, $timeout);
	if ($status != $main::SOCKET_OK) {
		&error("collect_rrd: couldn't connect to ${host}:${main::port}: $!");
		return undef;
	}
	&debug("  connected") if ($main::debug);

# Send the request
	($status, $timeout) = &write_socket( $socket, "LOGFILE $logfile\n", $timeout,
		"'LOGFILE $logfile' to $host");
	unless ($status == $main::SOCKET_OK) {
		$socket->close();
		return undef;
	}
	&debug("  sent 'LOGFILE $logfile'") if ($main::debug);

	if ($main::debug) {
		($status, $timeout) = &write_socket( $socket, "TEST\n", $timeout,
			"'TEST' to $host");
		unless ($status == $main::SOCKET_OK) {
			$socket->close();
			return undef;
		}
		&debug("  sent 'TEST'") if ($main::debug);

		($status, $timeout) = &write_socket( $socket, "DEBUG\n", $timeout,
			"'DEBUG' to $host");
		unless ($status == $main::SOCKET_OK) {
			$socket->close();
			return undef;
		}
		&debug("  sent 'DEBUG'") if ($main::debug);

	}

# And all the variable requests
	foreach $data (@{$main::config{RRD}{$wildrrd}{DATANAMES}}) {
		$extra = $main::config{RRD}{$wildrrd}{$data}{EXTRA};
		unless (defined $extra) {
			&error("collect_rrd: field $data has no request defined; skipped");
			next;
		}

# Allow magic cookies in patterns
		if ($extra =~ /##[A-Z0-9]+##/) {
			$extra = &do_subs(undef, $realrrd, $wildpart, $host, undef, undef, undef, $extra);
		}

		($status, $timeout) = &write_socket( $socket, "$data $extra\n", $timeout,
			"'$data $extra' to $host");
		unless ($status == $main::SOCKET_OK) {
			$socket->close();
			return undef;
		};
		++$main::requests;
		&debug("  sent '$data $extra'") if ($main::debug);
	}

	($status, $timeout) = &write_socket( $socket, "GO\n", $timeout,
		"'GO' to $host'");
	unless ($status == $main::SOCKET_OK) {
		$socket->close();
		return undef;
	};
	$socket->flush();
	&debug("  sent 'GO'") if ($main::debug);


# Collect the results
	while (($line, $status, $timeout) = &read_socket( $socket, $timeout, 
			"response from $host"), 
			((defined $line) and ($status == $main::SOCKET_OK))) {
		$line =~ tr/\015\012//d;
		++$main::entries_collected;
		&debug("  raw data: '$line'") if ($main::debug>1);
		next if ($line =~ /^#/ or $line =~/^\s*$/);

# Deal with special output
		if ($line =~ /^DEBUG:\s*(.*)/) {
			&debug("REMOTE($host): $1") if ($main::debug);
		}
		elsif ($line =~ /^ERROR:\s*(.*)/) {
			&error("collect_rrd: REMOTE($host): $1");
		}
		elsif ($line =~ /^ABORT:\s*(.*)/) {
			&abort("REMOTE($host) $1");
		}
		else {
			$main::data_from_host++;
			print "$host $line\n";
			print TMP "$host $line\n";
			++$main::entries_used;
		}
	}
	$socket->close();
	if ($status == $main::SOCKET_TIMEOUT) {
		&debug("timeout reading logs from ${host}:$main::port") if ($main::debug);
		return undef;
	}
	elsif ($status != $main::SOCKET_OK) {
		&debug("error reading logs from ${host}:$main::port: $_") if ($main::debug);
		return undef;
	}
}

#----------------------------------------------- keep_strict_happy ---
sub keep_strict_happy {
	$main::SOCKET_TIMEOUT = 0;
}
