///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "Pk.h"
#include "rheolef/rheostream.h"

#include <boost/numeric/ublas/matrix.hpp>
namespace ublas = boost::numeric::ublas;
#include "rheolef/ublas_matrix_range.h"
#include "rheolef/ublas-invert.h"
#include "rheolef/ublas-io.h"

namespace rheolef {
using namespace std;

// =========================================================================
// 0) utility: TODO: move in reference element
// =========================================================================
template<class T>
static
void
reference_element_barycenter (
  reference_element hat_K,
  point_basic<T>&   c)
{
  switch (hat_K.variant()) {
    case reference_element::p: c = point_basic<T> (0); break;
    case reference_element::e: c = point_basic<T> (0.5); break;
    case reference_element::t: c = point_basic<T> (T(1)/T(3), T(1)/T(3)); break;
    case reference_element::q: c = point_basic<T> (0, 0); break;
    case reference_element::T: c = point_basic<T> (0.25, 0.25, 0.25); break;
    case reference_element::P: c = point_basic<T> (T(1)/T(3), T(1)/T(3), 0); break;
    case reference_element::H: c = point_basic<T> (0, 0, 0); break;
    default: error_macro ("unexpected element type `"<<hat_K.name()<<"'");
  }
}
// =========================================================================
// 1) Lagrange node: TODO electrostatic, feteke
// =========================================================================
template<class T>
static
void
pointset_lagrange_equispaced (
  reference_element              hat_K,
  size_t                         degree,
  std::vector<point_basic<T> >&  hat_xnod)
{
  typedef reference_element::size_type  size_type;
  typedef point_basic<size_type>        ilat;
  hat_xnod.resize (reference_element::n_node(hat_K.variant(), degree));
  if (degree == 0) {
    reference_element_barycenter (hat_K, hat_xnod[0]);
    return;
  }
  switch (hat_K.variant()) {
    case reference_element::p: {
      hat_xnod [0] = point_basic<T> (T(0));
      break;
    } 
    case reference_element::e: {
      for (size_type i = 0; i <= degree; i++) { 
        size_type loc_idof = reference_element_e::ilat2loc_inod (degree, ilat(i));
        hat_xnod [loc_idof] = point_basic<T> ((T(i))/T(degree));
      }
      break;
    } 
    case reference_element::t: {
      for (size_type j = 0; j <= degree; j++) { 
        for (size_type i = 0; i+j <= degree; i++) { 
          size_type loc_idof = reference_element_t::ilat2loc_inod (degree, ilat(i,j));
          hat_xnod [loc_idof] = point_basic<T> ((T(i))/T(degree), T(j)/T(degree));
        }
      }
      break;
    } 
    case reference_element::q: {
      for (size_type j = 0; j <= degree; j++) { 
        for (size_type i = 0; i <= degree; i++) { 
          size_type loc_idof = reference_element_q::ilat2loc_inod (degree, ilat(i,j));
          hat_xnod [loc_idof] = point_basic<T> (-1+2*(T(i))/T(degree), -1+2*T(j)/T(degree));
        }
      }
      break;
    } 
    case reference_element::T: {
      for (size_type k = 0; k <= degree; k++) {
        for (size_type j = 0; j+k <= degree; j++) {
          for (size_type i = 0; i+j+k <= degree; i++) { 
            size_type loc_idof = reference_element_T::ilat2loc_inod (degree, ilat(i,j,k));
            hat_xnod [loc_idof] = point_basic<T> ((T(i))/T(degree), T(j)/T(degree), T(k)/T(degree));
          }
        }
      }
      break;
    } 
    case reference_element::P: {
      for (size_type k = 0; k <= degree; k++) { 
        for (size_type j = 0; j <= degree; j++) { 
          for (size_type i = 0; i+j <= degree; i++) { 
            size_type loc_idof = reference_element_P::ilat2loc_inod (degree, ilat(i,j,k));
            hat_xnod [loc_idof] = point_basic<T> ((T(i))/T(degree), T(j)/T(degree), -1+2*T(k)/T(degree));
          }
        }
      }
      break;
    } 
    case reference_element::H: {
      for (size_type k = 0; k <= degree; k++) { 
        for (size_type j = 0; j <= degree; j++) { 
          for (size_type i = 0; i <= degree; i++) { 
            size_type loc_idof = reference_element_H::ilat2loc_inod (degree, ilat(i,j,k));
            hat_xnod [loc_idof] = point_basic<T> (-1+2*(T(i))/T(degree), -1+2*T(j)/T(degree), -1+2*T(k)/T(degree));
          }
        }
      }
      break;
    } 
    default: error_macro ("unexpected element type `"<<hat_K.name()<<"'");
  }
}
// =========================================================================
// 3) monomial basis
// =========================================================================
static
void
Pk_monomial_basis (
  reference_element                  hat_K,
  size_t                             degree,
  std::vector<point_basic<size_t> >& m)
{
  typedef reference_element::size_type  size_type;
  typedef point_basic<size_type>        monom;
  m.resize (reference_element::n_node(hat_K.variant(), degree));
  switch (hat_K.variant()) {
    case reference_element::p: {
      m[0] = monom();
      break;
    }
    case reference_element::e: {
      for (size_type i = 0; i <= degree; i++) { 
        m[i] = monom(i);
      }
      break;
    }
    case reference_element::t: {
      for (size_type j = 0, iloc = 0; j <= degree; j++) {
        for (size_type i = 0; i+j <= degree; i++, iloc++) {
          m[iloc] = monom(i,j);
        }
      }
      break;
    }
    case reference_element::q: {
      for (size_type j = 0, iloc = 0; j <= degree; j++) { 
        for (size_type i = 0; i <= degree; i++, iloc++) { 
          m [iloc] = monom(i,j);
        }
      }
      break;
    }
    case reference_element::T: {
      for (size_type k = 0, iloc = 0; k <= degree; k++) {
        for (size_type j = 0; j+k <= degree; j++) {
          for (size_type i = 0; i+j+k <= degree; i++, iloc++) { 
            m[iloc] = monom(i,j,k);
          }
        }
      }
      break;
    }
    case reference_element::P: {
      for (size_type k = 0, iloc = 0; k <= degree; k++) { 
        for (size_type j = 0; j <= degree; j++) { 
          for (size_type i = 0; i+j <= degree; i++, iloc++) { 
            m[iloc] = monom(i,j,k);
          }
        }
      }
      break;
    }
    case reference_element::H: {
      for (size_type k = 0, iloc = 0; k <= degree; k++) { 
        for (size_type j = 0; j <= degree; j++) { 
          for (size_type i = 0; i <= degree; i++, iloc++) { 
            m[iloc] = monom(i,j,k);
          }
        }
      }
      break;
    }
  }
}
// =========================================================================
// 4) utility for monoms
// =========================================================================
// mx[k][i] = x[i]^k, k=0..degree, i=0..d-1
template<class T>
void
precompute_power (
    const point_basic<T>&      x,
    size_t                     degree,
    size_t                     d,
    std::vector<point_basic<T> >& xm)
{
    xm.resize (degree+1);
    for (size_t i = 0; i < d; i++) xm[0][i] = 1;
    for (size_t k = 1; k <= degree; k++) {
      for (size_t i = 0; i < d; i++) xm[k][i] = xm[k-1][i]*x[i]; 
    }
}
// result = prod_{i=0}^{d-1} x[i]^m[i]
// use predomputed x[i]^m for efficiency
template<class T>
static
void
eval_monom (
    const std::vector<point_basic<T> >& xm,
    const point_basic<size_t>&          m, 
    size_t                              d,
    T&                                  result)
{
    if (d == 2) {
      result = xm[m[0]][0]
              *xm[m[1]][1];
    } else if (d == 3) {
      result = xm[m[0]][0]
              *xm[m[1]][1]
              *xm[m[2]][2];
    } else if (d == 1) {
      result = xm[m[0]][0];
    } else { // d == 0
      result = 1;
    }
}
// x_i^k -> k*x_i^{k-1}
template<class T>
static
inline
T
dxk_dx (
    const std::vector<point_basic<T> >& xm,
    size_t k,
    size_t i)
{
  return (k == 0) ? 0 : k*xm[k-1][i];
}
// grad of prod_{i=0}^{d-1} x[i]^m[i]
template<class T>
static
void
grad_eval_monom (
    const std::vector<point_basic<T> >& xm,
    const point_basic<size_t>&          m, 
    size_t                              d,
    point_basic<T>&                     result)
{
    if (d == 2) {
      result[0] = dxk_dx(xm,m[0],0)*xm[m[1]][1];
      result[1] = dxk_dx(xm,m[1],1)*xm[m[0]][0];
    } else if (d == 3) {
      result[0] = dxk_dx(xm,m[0],0)*xm[m[1]][1]*xm[m[2]][2];
      result[1] = dxk_dx(xm,m[1],1)*xm[m[2]][2]*xm[m[0]][0];
      result[2] = dxk_dx(xm,m[2],2)*xm[m[0]][0]*xm[m[1]][1];
    } else if (d == 1) {
      result[0] = dxk_dx (xm,m[0],0);
    } else { // d == 0
      result[0] = 0;
    }
}
// =========================================================================
// 5) vdm: evaluate monom at Lagrange nodes
// =========================================================================
/*
 * Build the Lagrange basis, associated to nodes.
 *
 * input: nodes[n] and polynomial_basis[n]
 * output: node_basis[n]
 *   such that node_basis[j](node[i]) = kronecker[i][j]
 *
 * algorithm: let:
 *   b_i = \sum_i a_{i,j} p_j
 *  where:
 *       p_j = polynomial basis [1, x, y, ..]
 *       b_i = basis associated to node :
 *  we want:
 *   b_i(x_k) = delta_{i,k} 
 *   <=>
 *   a_{i,j} p_j(x_k) = \delta_{i,k}
 * Let A = (a_{k,j})_{i,j} and c_{i,j} = (p_j(x_i))
 * Then a_{i,j} c_{k,j} = delta_{i,k}
 *        <=>
 *    A = C^{-T}
 *
 * A is the Vandermonde matrix and the nodal basis writes:
     node_basis := trans(A)*polynomial_basis
 */
template<class T, class Matrix>
static
void
vandermonde_matrix (
    const std::vector<point_basic<size_t> >& m,
    const std::vector<point_basic<T> >&      hat_x,
    size_t                                   degree,
    size_t                                   d,
    std::vector<point_basic<T> >&            hat_xm_i, // working array, size=degree+1
    Matrix&                                  vdm)
{
  hat_xm_i.resize (degree+1);
  for (size_t i = 0, n = hat_x.size(); i < n; i++) {
    precompute_power (hat_x[i], degree, d, hat_xm_i);
    for (size_t j = 0; j < n; j++) {
      eval_monom (hat_xm_i, m[j], d, vdm(i,j));
    }
  }
}
// =========================================================================
// 6) matrix utility: det(vdm) -> 0 when degree -> infty and equispaced nodes
// =========================================================================
template<class T>
static
void
ublas_check_invert (
    const ublas::matrix<T>& m,
    const ublas::matrix<T>& inv_m,
    const T tol = std::numeric_limits<T>::epsilon())
{
    ublas::matrix<T> id;
    id.resize (m.size1(), m.size2());
    id.clear();
    for (size_t i = 0; i < id.size1(); i++) id(i,i)=1;
    ublas::matrix<T> check1 = prod(m,inv_m) - id;
    T err_linf = 0;
    T err_l2 = 0;
    for (size_t i = 0; i < id.size1(); i++) {
      for (size_t j = 0; j < id.size2(); j++) {
        err_linf = std::max(err_linf, fabs(check1(i,j)));
        err_l2  += sqr(check1(i,j));
      }
    }
    err_l2 = sqrt(err_l2);
    warning_macro ("invert error_l2   = " << err_l2);
    warning_macro ("invert error_linf = " << err_linf);
    check_macro (err_linf <= tol, "invert error=" << err_linf << " > tolerance=" << tol);
}
// =========================================================================
// 7) Pk members
// =========================================================================
template<class T>
basis_Pk<T>::~basis_Pk()
{
}
template<class T>
void
basis_Pk<T>::_initialize_hat_node (reference_element hat_K) const
{
  size_type variant = hat_K.variant();
  check_macro (variant < reference_element::max_variant, "unexpected element type: "<<variant);
  if (_hat_xnod[variant].size() != 0) return;
  // if not already computed by Pk::size(hat_K)
  pointset_lagrange_equispaced (hat_K, _degree, _hat_xnod[variant]);
}
template<class T>
void
basis_Pk<T>::_initialize (reference_element hat_K) const
{
  _initialize_hat_node (hat_K);
  _hat_xm.resize (_degree+1); // allocate working array, one time for all:
  size_type variant = hat_K.variant();
  Pk_monomial_basis (hat_K, _degree, _monom[variant]);
  size_type d = hat_K.dimension();
  size_type n = _hat_xnod[variant].size();
  ublas::matrix<T> vdm (n, n);
  vandermonde_matrix (_monom[variant], _hat_xnod[variant], _degree, d, _hat_xm, vdm);
  trace_macro ("init P"<<_degree<<"("<<hat_K.name()<<"): invert vdm(n="<<n<<")...");
  _inv_vdm[variant].resize(n,n);
  // TODO: invert take a while for d=3 and degree >= 5: use BLAS when T=double ?
  // see gmsh/Numeric/fullMatrix.cpp for BLAS c++ wrapper
  bool regular = invert (vdm, _inv_vdm[variant]);
  check_macro (regular, "unisolvence failed for P"<<_degree<<"(" << hat_K.name() << ")");
#undef DO_CHECK
#ifdef  DO_CHECK
  // check takes long time
  T tol = sqrt(std::numeric_limits<T>::epsilon());
  ublas_check_invert (vdm, _inv_vdm[variant], tol);
#endif // DO_CHECK
  trace_macro ("init P"<<_degree<<"("<<hat_K.name()<<"): done");
}
template<class T>
basis_Pk<T>::basis_Pk (std::string name) 
  : basis_rep<T> (name),
    _degree(0),
    _initialized(),
    _monom(),
    _hat_xnod(),
    _inv_vdm(),
    _hat_xm()
{
  if (name.length() > 0 && (name[0] == 'P' || name[0] == 'R')) {
    // TODO: check also that name fits "Pk" where is an k integer
    _degree = atoi(name.c_str()+1);
  } else if (name.length() > 0) { // missing 'P' !
    error_macro ("invalid polynomial name `"<<name<<"' for the Pk element");
  } else {
    // empty name : default cstor
    _degree = 0;
  }
  std::fill (_initialized.begin(), _initialized.end(), false);
}
template<class T>
typename basis_Pk<T>::size_type
basis_Pk<T>::size (reference_element hat_K) const
{
  return reference_element::n_node (hat_K.variant(), _degree);
}
// get Lagrange nodes associated to dofs:
template<class T>
void
basis_Pk<T>::hat_node (reference_element hat_K, std::vector<point_basic<T> >& hat_x) const
{
  // TODO: return a const ref ? but hat_node is used at space::cstor and in loop, in gnuplot visu
  _initialize_hat_node (hat_K);
  size_type variant = hat_K.variant();
  hat_x.resize (_hat_xnod[variant].size());
  std::copy (_hat_xnod[variant].begin(), _hat_xnod[variant].end(), hat_x.begin());
}
// evaluation of all basis functions at hat_x:
template<class T>
void
basis_Pk<T>::eval (reference_element hat_K, const point_basic<T>& hat_x, std::vector<T>& values) const
{
  _initialize_guard (hat_K);
  size_type variant = hat_K.variant();
  size_type d = hat_K.dimension();
  size_type loc_ndof = _inv_vdm[variant].size1();
  precompute_power (hat_x, _degree, d, _hat_xm);
  values.resize (loc_ndof);
  std::fill (values.begin(), values.end(), T(0));
  // phi_i(x) = vdm^{-T}_ij*monom_j(x)
  for (size_t loc_jdof = 0; loc_jdof < loc_ndof; loc_jdof++) {
    T mx_loc_jdof;
    eval_monom (_hat_xm, _monom[variant][loc_jdof], d, mx_loc_jdof);
    for (size_t loc_idof = 0; loc_idof < loc_ndof; loc_idof++) {
      values [loc_idof] += _inv_vdm [variant](loc_jdof,loc_idof)*mx_loc_jdof;
    }
  }
}
template<class T>
void
basis_Pk<T>::grad_eval (reference_element hat_K, const point_basic<T>& hat_x, std::vector<point_basic<T> >& values) const
{
  _initialize_guard (hat_K);
  size_type variant = hat_K.variant();
  size_type d = hat_K.dimension();
  size_type loc_ndof = _inv_vdm[variant].size1();
  precompute_power (hat_x, _degree, d, _hat_xm);
  values.resize (loc_ndof);
  std::fill (values.begin(), values.end(), point_basic<T>(0,0,0));
  // grad_phi_i(x) = vdm^{-T}_ij*grad_monom_j(x)
  for (size_t loc_jdof = 0; loc_jdof < loc_ndof; loc_jdof++) {
    point_basic<T> mx_loc_jdof;
    grad_eval_monom (_hat_xm, _monom[variant][loc_jdof], d, mx_loc_jdof);
    for (size_t loc_idof = 0; loc_idof < loc_ndof; loc_idof++) {
      for (size_t i = 0; i < d; i++) {
        values [loc_idof][i] += _inv_vdm [variant](loc_jdof,loc_idof)*mx_loc_jdof[i];
      }
    }
  }
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T)                                             	\
template class basis_Pk<T>;

_RHEOLEF_instanciation(Float)

}// namespace rheolef
