///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/inv_mass.h"
#include "rheolef/ublas_matrix_range.h"
#include "rheolef/ublas-invert.h"

namespace rheolef {
using namespace std;
using namespace ublas;

template<class T, class M>
void 
inv_mass<T,M>::operator() (const geo_element& K, ublas::matrix<T>& inv_m) const
{
  // space is vectorial or tensorial case : weight could be scalar, tensor or tensor4
  bool singular = false;
  space_constant::valued_type space_valued = base::get_first_space().valued_tag();
  if (space_valued == space_constant::scalar) {
    ublas::matrix<T> m;
    base::build_scalar_mass (K, m);
    inv_m = invert (m, singular);
    check_macro (!singular, "singular mass element matrix on " << K.dis_ie() << "-th element");
    return;
  }
  size_type n_comp = base::get_first_space().size();
  reference_element hat_K = K;
  size_type n = base::get_second_basis().size(hat_K);
  space_constant::valued_type weight_valued   = space_constant::scalar;
  if (base::is_weighted()) {
    weight_valued = base::_wh.get_space().valued_tag();
  }
  if (weight_valued == space_constant::scalar) {
    // --------------------------------------------------------------------------
    // all components have the same weight and the local matrix is block-diagonal
    // --------------------------------------------------------------------------
    ublas::matrix<T> m_ij;
    base::build_scalar_mass (K, m_ij);
    check_macro (m_ij.size1() == m_ij.size2(),
  	"inv_mass: incompatible `" << base::get_first_basis().name()
  	<< "' and `" << base::get_second_basis().name() << "' basis");
    matrix<T> inv_m_ij = invert (m_ij, singular);
    check_macro (!singular, "singular weighted-mass element matrix on " << K.dis_ie() << "-th element");
    inv_m.resize (n_comp*n, n_comp*n);
    inv_m.clear();
    space_constant::valued_type valued_tag = base::get_first_space().valued_tag();
    switch (valued_tag) {
     case space_constant::scalar:
     case space_constant::vector:
     case space_constant::tensor: {
      space_constant::coordinate_type sys_coord = base::coordinate_system();
      // symmetric tensor => 1/2 factor for (12,13,23) subscripts
      for (size_type ij_comp = 0; ij_comp < n_comp; ij_comp++) {
        std::pair<size_type,size_type> ij = space_constant::tensor_subscript (valued_tag, sys_coord, ij_comp);
        size_t i = ij.first;
        size_t j = ij.second;
        T factor_ij = ((i == j) ? 1 : 0.5); // symmetry => multiplicity factor: when i!=j : 2*w_ij
        mr_set (inv_m, range(ij_comp*n,(ij_comp+1)*n), range(ij_comp*n,(ij_comp+1)*n), factor_ij*inv_m_ij);  
      }
      break;
     }
     case space_constant::unsymmetric_tensor: {
      for (size_type i = 0; i < n_comp; i++)
        mr_set (inv_m, range(i*n,(i+1)*n), range(i*n,(i+1)*n), inv_m_ij);  
      break;
     }
     default: {
      string valued_name = base::get_first_space().valued();
      error_macro ("unexpected `" << valued_name << "'-valued space for the `inv_mass' form.");
     }
    }
  } else { // weight_valued != space_constant::scalar
    // ------------------------------------------------------------------------------
    // components have different weights : the local matrix is no more block-diagonal
    // ------------------------------------------------------------------------------
    ublas::matrix<T> m;
    base::build_general_mass (K, m);
    inv_m = invert (m, singular);
    check_macro (!singular, "singular weighted-mass element matrix on " << K.dis_ie() << "-th element");
#undef DO_CHECK_INV
#ifdef  DO_CHECK_INV
    ublas::matrix<T> id;
    id.resize (m.size1(), m.size2());
    id.clear();
    for (size_t i = 0; i < id.size1(); i++) id(i,i)=1;
    ublas::matrix<T> check1 = prod(m,inv_m) - id;
    T err = 0;
    for (size_t i = 0; i < id.size1(); i++)
      for (size_t j = 0; j < id.size2(); j++)
	    err += sqr(check1(i,j));
    if (err > 1e-16) {
	warning_macro ("K" << K.dis_ie() << ": invert error = " << err);
        cerr << setprecision(16);
        cerr << "M" << K.dis_ie() << " = [";
        for (size_t i = 0; i < id.size1(); i++) {
          for (size_t j = 0; j < id.size2(); j++) {
	    if (j == 0) cerr << "      ";
            cerr << m(i,j);
	    if (j != id.size2()-1) cerr << ",";
	    else if (i != id.size1()-1) cerr << ";" << endl;
	    else cerr << "];" << endl;
          }
        }
        cerr << "cond(M" << K.dis_ie() << ")" << endl;
    }
#endif //  DO_CHECK_INV
  }

}
template<class T, class M>
void
inv_mass<T,M>::initialize () const
{
  base::set_n_derivative(0);

  check_macro (base::get_first_space().stamp() == base::get_second_space().stamp(),
	"unsupported different approximation space for `inv_mass' form");

  check_macro (! base::get_first_space().get_numbering().is_continuous() &&
               ! base::get_first_space().get_numbering().is_continuous(),
	"unsupported continuous approximation space for `inv_mass' form");
 
  if (base::is_weighted()) {
    space_constant::valued_type weight_valued = base::_wh.valued_tag();
    if (weight_valued != space_constant::scalar) {
      // first case : vector space and tensor weight
      //     m(u,v) = int_Omega w_ij u_j v_i dx
      // second case : tensor space and tensor4 weight
      //     m(tau,gamma) = int_Omega w_ijkl tau_kj gamma_ij dx
      space_constant::valued_type space_valued = base::get_first_space().valued_tag();
      if (weight_valued == space_constant::tensor) {
	  check_macro (space_valued == space_constant::vector, "inv_mass form: unexpected tensorial weight and "
			  << base::get_first_space().valued() << " space");
      } else if (weight_valued == space_constant::tensor4) {
	  check_macro (space_valued == space_constant::tensor, "inv_mass form: unexpected tensorial weight and "
			  << base::get_first_space().valued() << " space");
      } else {
          error_macro ("inv_mass form: unexpected " << base::_wh.valued() << " weight and "
			  << base::get_first_space().valued() << " space");
      }
    }
  }
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class inv_mass<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class inv_mass<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
