///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
# include "rheolef/config.h"
#ifdef _RHEOLEF_HAVE_TAUCS
//
// taucs implementation of factorization
// for symmetric double out-of-core
//
// should support
//      - in-core/out-of-core
//	- symmetric/hermitian and unsymmetric]
// 	- simple, double and complex
//
#include "rheolef/ssk.h"
#include "rheolef/csr.h"
#include "rheolef/vec.h"
#include "rheolef/rheostream.h" // for itos()
using namespace rheolef;
using namespace std;

extern "C" {
#include <taucs.h>
extern void taucs_vec_permute (int n, int flags, void* v, void* pv, int p[]);
extern void taucs_vec_ipermute(int n, int flags, void* v, void* pv, int p[]);
extern void taucs_ccs_free    (taucs_ccs_matrix*);
};
struct ccs_taucs_rep {

    typedef unsigned int size_type;
    typedef double       element_type;

    ccs_taucs_rep();
    ~ccs_taucs_rep();
    void constructor();
    void destructor();
    void factorize (
        size_type                            n,
        vector<size_type>::const_iterator    ia,
        vector<size_type>::const_iterator    ja,
        vector<element_type>::const_iterator a);
  
    void solve (
        vector<element_type>::const_iterator b,
        vector<element_type>::iterator       x);

// data:
    taucs_ccs_matrix* _A;	// upper triang when symmetric
    taucs_io_handle*  _oocL;	// factorization
    int*              _perm;    // permutation
    double*           _b;       // tmp vectors
    double*           _x;
    double*           _b_perm;
    double*           _x_perm;
    static int        _file_count;	// unique count
};
int ccs_taucs_rep::_file_count = 0;

void
ccs_taucs_rep::constructor() {
	_A    = 0;
	_oocL = 0;
	_perm = 0;
	_b = 0;
	_x = 0;
	_b_perm = 0;
	_x_perm = 0;
}
void
ccs_taucs_rep::destructor()
{
    	if (_A)    taucs_dccs_free (_A);
	if (_oocL) taucs_io_delete (_oocL);
	if (_perm) free (_perm); // allocated in taucs C library
	if (_b)      delete_tab_macro (_b);
	if (_x)      delete_tab_macro (_x);
	if (_b_perm) delete_tab_macro (_b_perm);
	if (_x_perm) delete_tab_macro (_x_perm);
 	_A    = 0;
 	_oocL = 0;
	_perm = 0;
	_b = 0;
	_x = 0;
	_b_perm = 0;
	_x_perm = 0;
}
ccs_taucs_rep::ccs_taucs_rep() { constructor(); }
ccs_taucs_rep::~ccs_taucs_rep() { destructor(); }

void
ccs_taucs_rep::factorize (
    size_type                            n,
    vector<size_type>::const_iterator    ia,
    vector<size_type>::const_iterator    ja,
    vector<element_type>::const_iterator a)
{
    char*        ordering = "metis";
    string       matrix_file_name = string(tempnam(0,"tauc")) + "-M" + itos(_file_count);
    _file_count++;

    double       memory_mb   = -1.0;
    taucs_logfile ("none"); // "none" or "stderr"

    // ------------------------------------------------------------------
    // step 1
    // assume input data is symmetric
    // store only the upper triangular part in _A
    // since taucs use CSC instead of CSR, declare it lower triangular
    // ------------------------------------------------------------------
    int nrow = n;
    int ncol = n;
    size_type nent = 0;
    for (size_type i = 0 ; i < n; i++) {
        for (size_type p = ia[i] ; p < ia[i+1]; p++ ) {
          if (ja[p] >= i) nent++;
        }
    }
    _A = taucs_dccs_create (nrow,ncol,nent);
    _A->flags = TAUCS_DOUBLE | TAUCS_SYMMETRIC | TAUCS_LOWER;
    check_macro (_A, "taucs_ccs_create: creation failed");

    size_type inz = 0;
    int    *pind = _A -> rowind;
    double *pval = _A -> values.d;
    for (size_type i = 0 ; i < n; i++) {
	_A->colptr [i] = inz;
        for (size_type p = ia[i] ; p < ia[i+1]; p++ ) {
          if (ja[p] >= i) {
		*pind++ = int(ja [p]);
		*pval++ =  a [p];
		inz++;
	  }
        }
    }
    _A->colptr [n] = inz;
    size_type expect_nent = n+(ia[n]-n)/2;
    check_macro (nent == ia[n] || nent == expect_nent, 
	"invalid unsymmetric matrix for LDLt factorization: nent = "
	<< nent << " while " << ia[n] << " or "
	<< expect_nent << " was expected");
    // ------------------------------------------------------------------
    // step 2 : factor 
    //-----------------------------------------------------------
    int* invperm = 0;
    taucs_ccs_order (_A, &_perm, &invperm, ordering);
    check_macro (_perm, "taucs ordering failed\n");
    taucs_ccs_matrix*  PAPT = taucs_ccs_permute_symmetrically (_A, _perm, invperm);
    if (invperm) free (invperm); // allocated in taucs C library

    _oocL = taucs_io_create_multifile((char*)matrix_file_name.c_str());
    check_macro (_oocL, "taucs out-of-core creation failed");
    if (memory_mb == -1.0) {
	memory_mb = taucs_available_memory_size()/1048576.0;
    }
    int status = taucs_ooc_factor_llt (PAPT, _oocL, memory_mb*1048576.0);
    // if (status != 0) { warning_macro ("taucs out-of-core factorization failed"); }
    check_macro (status == 0, "taucs out-of-core factorization failed (status="<<status<<")");
    // ------------------------------------------------------------------
    // step 3 : allocate tmp vectors for solve() 
    //-----------------------------------------------------------
    // stupid g+-3.0.2 does not convert vector<T>::iterator in T*
    // => may allocate _x and _b !
    _x = new_tab_macro (double, n);
    _b = new_tab_macro (double, n);
    _x_perm = new_tab_macro (double, n);
    _b_perm = new_tab_macro (double, n);
}
void
ccs_taucs_rep::solve (
    vector<element_type>::const_iterator b0,
    vector<element_type>::iterator       x0)
{
    int n = _A->n;
    copy (b0, b0+n, _b);
    taucs_vec_permute (n, _A->flags, _b, _b_perm, _perm);
    int status = taucs_ooc_solve_llt (_oocL, _x_perm, _b_perm);
    check_macro (status == 0, "taucs out-of-core solve failed");
    taucs_vec_ipermute (_A->n, _A->flags, _x_perm, _x, _perm);
    copy (_x, _x+n, x0);
}
// --------------------------------------------------------------------------
// wrapper for ssk_rep<T>
// --------------------------------------------------------------------------
inline
ccs_taucs_rep& gerrep(void *p)
{
    return *((ccs_taucs_rep*)p);
}
template<class T>
taucs_rep<T>::taucs_rep ()
 : _p(new_macro(ccs_taucs_rep))
{
    gerrep(_p).constructor();
}
template<class T>
taucs_rep<T>::taucs_rep (const csr<T>& a)
 : _p(new_macro(ccs_taucs_rep))
{
    gerrep(_p).factorize (a.nrow(), a.ia().begin(), a.ja().begin(), a.a().begin());
}
template<class T>
taucs_rep<T>::taucs_rep (const taucs_rep<T>&)
{
    fatal_macro ("taucs_rep<T>::taucs_rep (const taucs_rep<T>&) may not be called");
}
template<class T>
taucs_rep<T>::~taucs_rep ()
{
    delete_macro ((ccs_taucs_rep*)_p);
}
template<class T>
void
taucs_rep<T>::factorize_ldlt()
{
    // already factorized: nothing to do
}
template<class T>
void
taucs_rep<T>::solve(const vec<T>& b, vec<T>& x) const
{
    if (x.size() == 0) return;
    gerrep(_p).solve(b.begin(),x.begin());
}
template<class T>
taucs_rep<T>::size_type
taucs_rep<T>::nnz () const
{
    fatal_macro("taucs_rep::nnz: not implemented");
}
template<class T>
taucs_rep<T>::size_type
taucs_rep<T>::nrow () const
{
    fatal_macro("taucs_rep::nrow: not implemented");
}
template<class T>
taucs_rep<T>::size_type
taucs_rep<T>::ncol () const
{
    fatal_macro("taucs_rep::ncol: not implemented");
}

// instanciation in library
template class ssk<double>;
template class taucs_rep<double>;
template ssk<double> ldlt (const csr<double>&);

#endif // _RHEOLEF_HAVE_TAUCS
