/* -*- Mode: C; c-file-style: "k&r"; -*-*/

/* token.c
 *    get tokens from a file
 *
 * $Id: token.c,v 1.7 2001/03/30 20:37:24 antoine Exp $
 *
 * Copyright (C) 2000,
 *     Antoine Lefebvre <antoine.lefebvre@polymtl.ca>
 *     Remi Lefebvre <remi@debian.org>
 *
 * gpcp is free software; you can redistribute them and/or modify them
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "err.h"
#include "token.h"


/* Idea to configure the parser
struct parser_config
{
     char  open_delimiter;
     char  close_delimiter;
     char  separator;
     char  comments;
     short equal;
} ParserConfig = {'(', ')', ',', '#', 1};
*/

int get_token(FILE *fd, char *buffer, int size, int *line)
{
     int pos = 0; /* position from beginning of token in long tokens */

     int character;

     static short open_quote = 0; /* in quote */
     static short lexer_state = STATE_NONE;
  
     while (1)
     {
          /* get a character */
          character = getc(fd);
          
          if (character == EOF)
               return EOF;

          if (pos >= size)
               return OVERFLOW;
          
          switch (lexer_state)
          {
          case STATE_NONE:
               buffer[0] = character;
               buffer[1] = '\0';

               switch (character)
               {
                    /* destroy spaces and tabs*/
               case '\t':
               case ' ':
                    break;

                    /* one char tokens: */
               case '(':
                    return TOK_OPENPAR;
                    
               case ')':
                    return TOK_CLOSEPAR;
                    
               case '=':
                    return TOK_EQUAL;

               case '"':
                    open_quote = (!open_quote);

                    if (open_quote)
                    {
                         lexer_state = STATE_STRING;
                    }
                    else
                    {
                         lexer_state = STATE_NONE;
                    }
                    break;

               case ',':
                    return TOK_COMMA;

               case '#':
                    lexer_state = STATE_IN_COMMENT;
                    break;

               case '\n':
                    (*line)++;
                    break;

                    /* any other character is part of a long token */
               default: 
                    buffer[pos] = character;
                    pos++;
                    lexer_state = STATE_LONG_TOKEN;
                    break;
               }
               break;

          case STATE_LONG_TOKEN:
          
               switch (character)
               {

                    /* in this case we get a function
                     * NOTE: the parenthesis must be right beside the function
                     * name */
               case '(':
                    lexer_state = STATE_NONE;
                    return TOK_FUNCTION;

                    /* if we get a single token char, push it back, it will
                     * be used next iteration. long token completed */
               case '\n':
               case '=':
               case ')':
               case '"':
               case ',':
                    if (ungetc(character, fd) == EOF)
                    {
                         /* error pushing the character back */
                         printf("Error pushing character back into stream.\n");
                    }

                    /* the token have been read */
               case ' ':
                    lexer_state = STATE_NONE; /* we are done */
                
                    /* what is the type */
                    buffer[pos] = '\0';

                    /* we parse for special keywords */
                    if (strncasecmp("true", buffer, size) == 0)
                    {
                         return TOK_TRUE;
                    }
                    else if (strncasecmp("false", buffer, size) == 0)
                    {
                         return TOK_FALSE;
                    }
                    else if (is_integer(buffer))
                    {
                         return TOK_INTEGER;
                    }
                    else if (is_float(buffer))
                    {
                         return TOK_FLOAT;
                    }
                    else
                    {
                         return TOK_KEYWORD; /* anything else */
                    }
                    break;

                    /* just push it back in buffer and continue looping.
                     * yahooo! */
               default:
                    buffer[pos] = character;
                    pos++;
                    break;
               }
               break;

          case STATE_STRING:
               switch (character)
               {
               case '"':
                    lexer_state = STATE_NONE;
                    buffer[pos] = '\0';

                    /* lets push it back, we are not char eaters =) */
                    if (ungetc(character, fd) == EOF)
                    {
                         /* error pushing the character back */
                         printf("Error pushing character back\n");
                    }

                    return TOK_STRING;

               case '\t':
                    break;
                
               case '\n':
                    (*line)++;

                    /* same pattern as above, push in buffer and loop */
               default:
                    buffer[pos] = character;
                    pos++;
                    break; 
               }
               break;

          case STATE_IN_COMMENT:
               switch (character)
               {
                    /* one line comments only */
               case '\n':
                    lexer_state = STATE_NONE;

                    /* push it back so we can count it next loop */
                    if (ungetc(character, fd) == EOF)
                    {
                         printf("Error pushing character back\n");
                    }
                    pos = 0;
                    break;
                    
                    /* eat up comment. its useless for a machine */
               default:
                    buffer[pos] = character;
                    pos++;
                    break;
               }
               break;
          }
     } 
}


int is_float(char *buffer)
{
     double tmp;
     char *ptr;

     tmp = strtod(buffer, &ptr);

     if ((tmp != 0) || (buffer != ptr))
          return 1;

     return 0;
}

int is_integer(char *buffer)
{
     int i;
     for(i = 0; i < strlen(buffer); i++)
     {
          if (!(isdigit(buffer[i]) || isspace(buffer[i])))
               return 0;
     }

     return 1;
}


