
/*
 * Copyright (C) 1999-2002, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>

/* The minimum size of a string chunk */
#define RBUF_MIN_SIZE (sizeof (void *))

/* The breaking point where we switch from using
 * memchunks, to using malloc/realloc/free */
#define RBUF_MEMCHUNK_SIZE 1024

/* Should have proper warning system for this type of thing.. */
#define RBUF_RETURN_IF_RDONLY(buf) \
if (rbuf_rdonly(buf)) { \
    printf ("ACK! Someone tried to mess with a rdonly rbuf!\n"); \
        return; \
}


#define rbuf_maybe_expand(buf, size) \
    do { \
        unsigned int roy_len_total = buf->len + size; \
        if ((roy_len_total >= buf->alloc) || roy_len_total < (size >> 3)) \
            rbuf_resize(buf, size); \
    } while (0);


static void
rbuf_resize (RBuf * buf, unsigned int size)
{
    unsigned int total;
    unsigned int new_size = RBUF_MIN_SIZE;
    char *tmpbuf;

    total = buf->len + size;

    /* Get the nearest power above the new size */
    while (new_size <= total)
        new_size <<= 1;

    /* If the new size is the same as the previous, just return */
    if (new_size == buf->alloc)
        return;

    /* Deal with memchunk allocated strings */
    if (total < RBUF_MEMCHUNK_SIZE) {

        /* create a new buffer with rchunk */
        tmpbuf = rchunk_alloc (new_size);

        /* Copy the old string into the new buffer */
        if (buf->str) {
            if (buf->len > 0) {
                memcpy (tmpbuf, buf->str, buf->len);
            }

            /* Free the old buffer if necessary */
            if (buf->alloc < RBUF_MEMCHUNK_SIZE)
                rchunk_free (buf->str, buf->alloc);
            else
                rmem_free (buf->str);
        }

        buf->str = tmpbuf;

    } else {
        /* Total is too big to use a memchunk */

        if (buf->alloc > RBUF_MEMCHUNK_SIZE) {
            /* if the old size was also from an malloc, try doing
             * realloc and hope the system can just expand the 
             * block */
            buf->str = rmem_realloc (buf->str, new_size);
        } else {
            /* we are moving from a previous rchunk_alloc to rmem_alloc
             * style */

            tmpbuf = rmem_alloc (new_size);\
                if (buf->str) {
                    if (buf->len > 0) {
                        memcpy (tmpbuf, buf->str, buf->len);
                    }
                    rchunk_free (buf->str, buf->alloc);
                }

            buf->str = tmpbuf;
        }
    }

    buf->alloc = new_size;
}


void
rbuf_free (RBuf *buf)
{
    if (!buf || RFLAG_ISSET (buf, RBUF_OWNED)) {
        return;
    }

    if (!RFLAG_ISSET (buf, RBUF_STATIC)) {
        if (buf->alloc < RBUF_MEMCHUNK_SIZE) {
            rchunk_free (buf->str, buf->alloc);
        } else {
            rmem_free (buf->str);
        }
    }

    rchunk_free (buf, sizeof (RBuf));
}

static RBuf *
rbuf_alloc (void)
{
    RBuf *newbuf;

    newbuf = rchunk_alloc (sizeof (RBuf));
    newbuf->len = 0;
    newbuf->alloc = 0;
    newbuf->str = NULL;
    newbuf->flags = 0;

    return (newbuf);
}


RBuf *
rbuf_new (void)
{
    RBuf *newbuf;

    newbuf = rbuf_alloc ();

    rbuf_maybe_expand (newbuf, 2);
    newbuf->str[newbuf->len] = '\0';

    return (newbuf);
}

RBuf *
rbuf_new_sized (unsigned int size)
{
    RBuf *newbuf;

    newbuf = rbuf_alloc ();
    rbuf_maybe_expand (newbuf, size);

    newbuf->str[newbuf->len] = '\0';

    return (newbuf);
}


RBuf *
rbuf_new_with_str (const char *str)
{
    RBuf *newbuf;

    newbuf = rbuf_alloc ();
    rbuf_append_str (newbuf, str);

    return (newbuf);
}

RBuf *
rbuf_new_with_static (char *str)
{
    RBuf *newbuf;

    newbuf = rbuf_alloc ();
    newbuf->str = str;
    newbuf->len = strlen (str);

    rbuf_set_rdonly (newbuf);
    RFLAG_SET(newbuf, RBUF_STATIC);

    return (newbuf);
}

RBuf *
rbuf_new_with_data (const char *str, unsigned int len)
{
    RBuf *newbuf;

    newbuf = rbuf_alloc ();
    rbuf_append_data (newbuf, str, len);

    return (newbuf);
}

RBuf *
rbuf_new_with_rbuf (const RBuf * buf)
{
    RBuf *newbuf;

    newbuf = rbuf_alloc ();
    rbuf_append_rbuf (newbuf, buf);

    return (newbuf);
}

RBuf *
rbuf_new_with_long (long int i)
{
    RBuf *newbuf;
    char convert[128];
    int place = 0;
    unsigned long uvalue;
    unsigned int negative;
    char *p;
    unsigned int length;

    if (i < 0) {
        negative = TRUE;
        uvalue = -i;
    } else {
        negative = FALSE;
        uvalue = i;
    }
   
    do {
        convert[place++] = ("0123456789") [uvalue % 10];
        uvalue = uvalue / 10;
    } while (uvalue);

    length = place + (negative ? 1 : 0);

    newbuf = rbuf_new_sized (length + 1);
    p = rbuf_str (newbuf);
    if (negative) {
        *p++ = '-';
    }

    while (place > 0) {
        *p++ = convert[--place];
    }

    *p = '\0';
    newbuf->len = length;

    return (newbuf);
}


void
rbuf_set_to_str (RBuf *buf, const char *str)
{
    unsigned int len;
    unsigned int total;

    if (!buf || !str)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    len = strlen (str);

    total = len - buf->len;
    rbuf_maybe_expand (buf, total);

    memcpy (buf->str, str, len);
    buf->len = len;

    buf->str[buf->len] = '\0';
}

void
rbuf_set_to_rbuf (RBuf *buf, const RBuf *val)
{
    unsigned int total;

    if (!buf || !val)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    total = val->len - buf->len;
    rbuf_maybe_expand (buf, total);

    memcpy (buf->str, val->str, val->len);
    buf->len = val->len;
    buf->str[buf->len] = '\0';
}

void
rbuf_set_to_data (RBuf *buf, const char *str, unsigned int len)
{
    unsigned int total;

    if (!buf || !str)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    total = len - buf->len;
    rbuf_maybe_expand (buf, total);

    memcpy (buf->str, str, len);
    buf->len = len;
    buf->str[buf->len] = '\0';
}


void
rbuf_prepend_str (RBuf *buf, const char *str)
{
    unsigned int len;

    if (!buf || !str)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    len = strlen (str);
    rbuf_maybe_expand (buf, len);

    memmove (buf->str + len, buf->str, buf->len);
    memcpy (buf->str, str, len);
    buf->len += len;
    buf->str[buf->len] = '\0';
}

void
rbuf_prepend_char (RBuf *buf, char c)
{
    if (!buf)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    rbuf_maybe_expand (buf, 1);

    memmove (buf->str + 1, buf->str, buf->len);
    buf->str[0] = c;
    buf->len += 1;
    buf->str[buf->len] = '\0';
}

void
rbuf_prepend_rbuf (RBuf *buf, const RBuf *newbuf)
{
    if (!buf || !newbuf)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    rbuf_maybe_expand (buf, rbuf_len (newbuf));

    memmove (buf->str + rbuf_len (newbuf), buf->str, buf->len);
    memcpy (buf->str, rbuf_str (newbuf), rbuf_len (newbuf));
    buf->len += rbuf_len (newbuf);
    buf->str[buf->len] = '\0';
}

void
rbuf_prepend_data (RBuf *buf, const char *str, unsigned int len)
{
    if (!buf || !str)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    rbuf_maybe_expand (buf, len);

    memmove (buf->str + len, buf->str, buf->len);
    memcpy (buf->str, str, len);
    buf->len += len;
    buf->str[buf->len] = '\0';
}

void
rbuf_append_str (RBuf *buf, const char *str)
{
    unsigned int len;

    if (!buf || !str)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    len = strlen (str);
    rbuf_maybe_expand (buf, len);

    memcpy (buf->str + buf->len, str, len);
    buf->len += len;
    buf->str[buf->len] = '\0';
}

void
rbuf_append_char (RBuf *buf, char c)
{
    if (!buf)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    rbuf_maybe_expand (buf, 1);

    buf->str[buf->len] = c;
    buf->len += 1;
    buf->str[buf->len] = '\0';
}

void
rbuf_append_data (RBuf *buf, const char *str, unsigned int len)
{
    if (!buf || !str)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    rbuf_maybe_expand (buf, len);

    memcpy (&buf->str[buf->len], str, len);
    buf->len += len;
    buf->str[buf->len] = '\0';
}

void
rbuf_append_rbuf (RBuf *buf, const RBuf *newbuf)
{
    if (!buf || !newbuf)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    rbuf_maybe_expand (buf, newbuf->len);

    memcpy (&buf->str[buf->len], newbuf->str, newbuf->len);
    buf->len += newbuf->len;
    buf->str[buf->len] = '\0';
}

void
rbuf_insert_str (RBuf *buf, unsigned int pos, const char *val)
{
    unsigned int len;

    if (!buf || !val)
        return;
    if (pos > buf->len)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    len = strlen (val);
    rbuf_maybe_expand (buf, len);

    memmove (buf->str + pos + len, buf->str + pos, buf->len - pos);
    memcpy (buf->str + pos, val, len);
    buf->len += len;
    buf->str[buf->len] = '\0';
}

void
rbuf_insert_data (RBuf *buf, unsigned int pos, const char *val, unsigned int len)
{
    if (!buf || !val)
        return;
    if (pos > buf->len)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    rbuf_maybe_expand (buf, len);

    memmove (buf->str + pos + len, buf->str + pos, buf->len - pos);
    memcpy (buf->str + pos, val, len);
    buf->len += len;
    buf->str[buf->len] = '\0';
}

void
rbuf_insert_rbuf (RBuf *buf, unsigned int pos, const RBuf *val)
{
    if (!buf || !val)
        return;
    if (pos > buf->len)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    rbuf_maybe_expand (buf, val->len);

    memmove (buf->str + pos + val->len, buf->str + pos, buf->len - pos);
    memcpy (buf->str + pos, val, val->len);
    buf->len += val->len;
    buf->str[buf->len] = '\0';
}


void
rbuf_erase (RBuf *buf, unsigned int pos, unsigned int len)
{
    if (!buf)
        return;
    if ((pos > buf->len) || (pos + len > buf->len))
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    memmove (buf->str + pos, buf->str + pos + len,
        buf->len - (pos + len));
    buf->len -= len;
    buf->str[buf->len] = '\0';
}

void
rbuf_truncate (RBuf *buf, unsigned int len)
{
    if (!buf)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    buf->len = len;

    /* This will shrink the buffer.. */
    rbuf_maybe_expand (buf, 0);

    buf->str[len] = '\0';
}

void
rbuf_down (RBuf *buf)
{
    unsigned char *c;

    if (!buf)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    c = (unsigned char *) buf->str;
    while (*c) {
        *c = tolower (*c);
        c++;
    }
}

void
rbuf_up (RBuf *buf)
{
    unsigned char *c;

    if (!buf)
        return;

    RBUF_RETURN_IF_RDONLY (buf);

    c = (unsigned char *) buf->str;
    while (*c) {
        *c = toupper (*c);
        c++;
    }
}

unsigned int
rbuf_is_whitespace (const RBuf *buf)
{
    char *str;
    unsigned int i;

    if (!buf)
        return (TRUE);

    str = buf->str;

    for (i = 0; i < rbuf_len (buf); i++) {
        if ((str[i] != ' ') &&
            (str[i] != '\t') &&
            (str[i] != '\r') && 
            (str[i] != '\n') && 
            (str[i] != '\0'))
            return (FALSE);
    }
    return (TRUE);
}

unsigned int
rbuf_equal_rbuf (const RBuf *buf1, const RBuf *buf2)
{
    unsigned char *c1;
    unsigned char *c2;
    unsigned int *i1;
    unsigned int *i2;
    unsigned int i;
    unsigned int num_ints;

    /* quick and easy check that might catch a lot of them without
     * having to walk the strings */
    if (buf1->len != buf2->len)
        return (FALSE);
   
    i1 = (unsigned int *) rbuf_str (buf1);
    i2 = (unsigned int *) rbuf_str (buf2);

    num_ints = rbuf_len (buf1) / sizeof (int);
    for (i = num_ints; i--; i1++, i2++) {
	if (*i1 != *i2)
	    return (FALSE);
    }

   
    c1 = (unsigned char *) i1;
    c2 = (unsigned char *) i2;

    /* c1 and c2 now point to the last bit of the string, 
     * so we can just assign our pointers to them, and compare
     * from there */
    i = rbuf_len (buf1) - (num_ints * sizeof (int));

    for (; i--; c1++, c2++) {
	if (*c1 != *c2)
	    return (FALSE);
    }

    return (TRUE);
}

unsigned int
rbuf_equal_str (const RBuf *buf1, const char *str)
{
    if (!buf1 || !str)
        return (FALSE);

    return (strcmp (buf1->str, str) == 0);
}

unsigned int
rbuf_equal_strcase (const RBuf *buf1, const char *str)
{
    if (!buf1 || !str)
        return (FALSE);

    return (strcasecmp (buf1->str, str) == 0);
}

/* A slightly faster, cheap knock off */
unsigned long
rbuf_hash2 (const RBuf *buf)
{
    unsigned long hashval = 0;

    if (rbuf_len(buf) >= sizeof (int)) {
	const unsigned int *i;
        unsigned int num_ints = rbuf_len(buf) / sizeof(int);

        for (i = ((unsigned int *) rbuf_str (buf)); num_ints--; i++) {
            hashval += (hashval << 3) + *i;
        }
    } else {
        const char *p;
        for (p = rbuf_str (buf); *p != '\0'; p++) 
            hashval = (hashval << 3) + *p;
    }

    return (hashval);
}


unsigned long
rbuf_hash (const RBuf *buf)
{
    unsigned long hashval = 0;
    const char *p;

    for (p = rbuf_str (buf); *p != '\0'; p++) 
        hashval = (hashval << 5) - hashval + *p;

    return (hashval);
}

RList *
rbuf_split (const RBuf *string, char *separators, int max_splits)
{
    RBuf *newbuf;
    RListOfRBuf *string_list;
    RListOfRBufEntry *newbufentry;
    char *str;
    char *c;
    int splits = 0;
    unsigned int index = 0;
    unsigned int last = 0;

    if (string == NULL)
        return NULL;

    if (max_splits < 0)
        max_splits = INT_MAX;

    string_list = rlist_new ();

    str = rbuf_str (string);

    if (max_splits == 0) {
        /* Create a list with the entire string in it */
        rlist_append (string_list, rlist_of_rbuf_entry_new (rbuf_new_with_rbuf (string)));
    } else {
        for (index = 0; index < rbuf_len (string); index++) {
            for (c = separators; *c != '\0'; c++) {
                if (*c == str[index]) {
                    newbuf = rbuf_new_with_data (&str[last], index - last);
                    newbufentry = rlist_of_rbuf_entry_new (newbuf);
                    rlist_append (string_list, newbufentry);

                    last = index + 1;

                    splits++;
                    if (splits >= max_splits) {
                        index = rbuf_len (string);
                        goto done_splitting;
                    }

                    break;
                }
            }
        }

done_splitting:
        newbuf = rbuf_new_with_data (&str[last], index - last);
        newbufentry = rlist_of_rbuf_entry_new (newbuf);
        rlist_append (string_list, newbufentry);
    }

    return (string_list);
}


typedef struct
{
    RLIST_HEADER;
    RBuf buf;
} RBufAutoListEntry;

static RList *rbuf_auto_list_entries = NULL;
static RBufAutoListEntry *rbuf_auto_list_entry = NULL;

#define RBUF_AUTO_NUM_BUFS 32

void
rbuf_auto_cleanup__P (void);


void
rbuf_auto_cleanup__P (void)
{

    /* We do it reverse because it's a circular list going forward */
    RLIST_FOREACH_REVERSE (rbuf_auto_list_entries, rbuf_auto_list_entry) {
        rchunk_free (rbuf_auto_list_entry, sizeof (RBufAutoListEntry));
    } RFOREACH_CLOSE;

    rlist_free (rbuf_auto_list_entries);
}

void
rbuf_auto_init__P (void)
{
    int i;
    
    /* We keep around a list of buf's that we can snatch
     * from.  Hopefully you never use so many automatic
     * bufs that this list tramples itself */
    
    if (!rbuf_auto_list_entries) {
        rbuf_auto_list_entries = rlist_new ();

        for (i = 0; i < RBUF_AUTO_NUM_BUFS; i++) {
            rbuf_auto_list_entry = rchunk_alloc (sizeof (RBufAutoListEntry));
            rlist_append (rbuf_auto_list_entries, rbuf_auto_list_entry);
        }

        /* Make the list circular, at least for the forward walking 
         * direction */
        rbuf_auto_list_entry = rlist_last (rbuf_auto_list_entries);
        rbuf_auto_list_entry->rlist_header.next = rlist_first (rbuf_auto_list_entries);

        rbuf_auto_list_entry = rlist_first (rbuf_auto_list_entries);
    }

    rcleanup_add_hook (rbuf_auto_cleanup__P);
}

RBuf *
rbuf_auto (char *str)
{
    RBuf *newbuf;

    newbuf = &rbuf_auto_list_entry->buf;
    rbuf_auto_list_entry = rlist_next (rbuf_auto_list_entry);

    newbuf->str = str;
    newbuf->len = strlen (str);
    newbuf->flags = RBUF_OWNED | RBUF_STATIC | RBUF_RDONLY;

    return (newbuf);
}



