
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#ifndef __RCHUNK_H__
#define __RCHUNK_H__

/* This is a chunk allocator.  It allocates blocks of memory from a pool
 * that it creates on demand.  Any free'd blocks are simply put back into
 * the pool, and are not given back to the system.  It is very fast.  In
 * my tests, it allocates approximately 2.5x faster than malloc for chunks
 * that are less than 4096 bytes in size.  I suspect it would be even
 * faster for larger chunks, but that's not really its purpose.
 *
 * The chunk allocator should be used instead of malloc whenever you want
 * to allocate and free a lot of items that are the same size.  Common
 * cases are structures for linked lists, structures found in your
 * applications etc.
 *
 * You'll notice that the size must be specified for the free'ing of the
 * chunk.  This is because we don't keep the extra integer around to
 * keep track of the size.  It is assumed that since you are doing
 * allocations of fixed sized chunks, that you will know the size of the
 * chunk you are free'ing.
 */

/* Public interface should be defined as:
 *
 *
 * Allocate a new block of memory of the specified size
 * 
 * void *
 * rchunk_alloc (unsigned long size);
 *
 *
 * Free memory allocated with rchunk_alloc.  Size must be specified for
 * the free, as it is not stored within the rchunk subsystem.
 * 
 * void
 * rchunk_free (void *chunk, unsigned long size);
 *
 */


/* Private functions */

void
rchunk_free (void *chunk, unsigned long size);

void *
rchunk_alloc (unsigned long size);

/* Same as above, but zero all memory before returning */
void *
rchunk_alloc0 (unsigned long size);

#endif /* __RCHUNK_H__ */


