
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>
 * and Jim Meier <fatjim@home.com>.
 *
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

/* -*- mode: C; c-basic-offset: 4  -*- */ 

#ifndef __THERML_H__
#define __THERML_H__

#include <roy.h>
#include <stdlib.h>
#include <ctype.h>

/*

  Actions:

  A - Add char to 'data' attribute of cur node
  B - NOP
  C - push a new node
  D - mark cur node pcmd
  E - mark cur node comment 
  F - mark cur node start
  G - Add char to cur node name
  H - Mark cur node end
  I - signal this node (if end node or comment or pcmd, pop cur node)
  J - add attr to cur node, name = char
  K - add char to cur attr name
  L - add char to cur attr val
  M - add char to count_str of cur attr
  N - turn count_str to count
  O - add char to end_str of cur_node
  P - remove len(end_str)-1 chars from cur attr val
	Q - ensure that the node name is a properly formed path name

*/

/*

  States:

  01 - (start node)   '<':C:02; else:A:01
  02 - '/':H:16; '?':D:13; '!':E:08; alpha:E:03
  03 - alpha:G:03, whitespace:B:04; '>':I:1; '/':H:15
  04 - whitespace:B:04; alpha:J:05; '>':I:1; '/':H:17
  05 - alpha:K:05; '=':B:-6; '(':B:18
  06 - '"':B:07
  07 - '"':B:04; anythingelse:L:07
  08 - '-':B:09
  09 - '-':B:10
  10 - '-":B:11; anythingelse:B:10
  11 - '-':B:12; anythingelse:B:10
  12 - '<':B:01; anythingelse:B:10
  13 - '?':B:14; anythingelse:A:13
  14 - '>':I:01
  15 - '>':I:01
  16 - '>':I:1; alpha:G:16
  17 - '>':I:01
  18 - digit:M:19; endchar:O:22
  19 - ')':N:20; digit:M:19
  20 - '=':B:21
  21 - **count chars have been read**:L:04;  **count chars have /not/ been read**:L:21
  22 - ')':B:23; endchar:O:22
  23 - '=':B:24
  24 - **end node read**:P:04; **end node not reached**:L:24

  obviously some of this could be trimmed; i see 14, 15, and 17 are identical
  and could be merged. other less obvious improvements. but this is what you
  get. MWAHAHAHA!

*/


/* Action symbolic names */
#define ERROR_ACT 0
#define ACT_A     1
#define ACT_B     2
#define ACT_C     3
#define ACT_D     4
#define ACT_E     5
#define ACT_F     6
#define ACT_G     7
#define ACT_H     8
#define ACT_I     9
#define ACT_J    10
#define ACT_K    11
#define ACT_L    12
#define ACT_M    13
#define ACT_N    14
#define ACT_O    15
#define ACT_P    16
#define ACT_Q    17
#define ACT_R    18

/* character classes and special checks */
/* note that trying to match -1 always results in an error match, error
 * action, and return to initial state (FIXME: improve error recovery) */
#define ANYTH 0
#define ALPHA 1
#define DIGIT 2
#define ENDCH 3
#define WHTSP 4
#define NWHSP 5
#define CNTR1 6
#define CNTR2 7
#define ENDT1 8
#define ENDT2 9
#define ENDAT 10

/* tag type flags */
#define NODE_TYPE_PCMD    (1<<0)
#define NODE_TYPE_COMMENT (1<<1)
#define NODE_TYPE_START   (1<<2)
#define NODE_TYPE_END     (1<<3)
#define NODE_TYPE_PARTIAL (1<<4)
#define NODE_TYPE_HASPATH (1<<5)

typedef struct _ThermlAttribute ThermlAttribute;
typedef struct _ThermlNode ThermlNode;
typedef struct _ThermlState ThermlState;
typedef struct _ThermlTransition ThermlTransition;
typedef struct _ThermlParser ThermlParser;


typedef void (*ThermlActFunc) (ThermlParser *parser, char c);

typedef struct _ThermlActFuncEntry ThermlActFuncEntry;

typedef void (*ThermlErrorHandler) (ThermlParser *parser, int severity, 
                                    int linenum, int colnum,
                                    const char *description);

typedef void (*ThermlPCommandHandler) (ThermlParser *parser, RBuf *data);

typedef void (*ThermlNodeStartHandler) (ThermlParser *parser, RBuf *type,
                                        RBHash *attribs);

typedef void (*ThermlNodeEndHandler) (ThermlParser *parser, RBuf *type,
                                      RBuf * data_attr);



struct _ThermlAttribute {
    RBHASH_HEADER;
    RBuf *value;
};


struct _ThermlNode {
    RBuf *type;
    int flags;
    RBHash *attribs;
    RBuf *data;
    ThermlNode *prev;
};



struct _ThermlParser {
    int state;

    int linenum;
    int colnum;

    int last_state_linenum;
    int last_state_colnum;

    int fatal_error;

    /* A linked list of nodes */
    ThermlNode *nodes;

    /* Storage for the current attribute, till we get it all */
    RBuf *current_attribute;
    RBuf *current_value;

    /* If a value has a length specifier, keep track of it here */
    RBuf *count_string;
    unsigned int value_length;

    /* Keep track of the quote char if applicable */
    char quote_char;

    /* Keep track of closing quote string if applicable */
    RBuf *end_of_attribute_string;

    /* user callbacks */
    ThermlErrorHandler error_handler;
    ThermlPCommandHandler pcommand_handler;
    ThermlNodeStartHandler node_start_handler;
    ThermlNodeEndHandler node_end_handler;
   
    /* User data */
    void *user_data;
};


struct _ThermlTransition {
    int match;
    int act;
    int newstate;
};


struct _ThermlState {
    int number;
    char *description;
    ThermlTransition t1;
    ThermlTransition t2;
    ThermlTransition t3;
    ThermlTransition t4;
    ThermlTransition t5;
};

struct _ThermlActFuncEntry {
    ThermlActFunc act;
    const char *description;
};



/* interface */

/* Parse a chunk of therml.  Returns 0 on success, -1 on error */
int 
therml_parse_chunk (ThermlParser *parser, char *chunk, int len,
		    int is_last_chunk);


/* Set a function to be called on a parse error */
void 
therml_set_error_handler (ThermlParser *parser,
                          ThermlErrorHandler handler);

/* Set the pcommand handler */
void
therml_set_pcommand_handler (ThermlParser *parser,
                             ThermlPCommandHandler handler);

/* Set a function to be called when a new node is found */
void
therml_set_node_start_handler (ThermlParser *parser,
                               ThermlNodeStartHandler handler);

/* Set function to be called when a node is closed */
void 
therml_set_node_end_handler (ThermlParser *parser,
                             ThermlNodeEndHandler handler);

/* Set/Get user data */
void
therml_set_user_data (ThermlParser *parser, 
                      void *userdata);

void *
therml_get_user_data (ThermlParser *parser);

ThermlParser *
therml_parser_new (void);

void
therml_parser_free (ThermlParser *ptr);


#endif /* __THERML_H__ */



