;;
;;  processing a user format
;;
;;  a "user format" is the specification of a function which
;;  takes as input a record and renders its desired representation
;;  onto the current output port
;;
;;  there is only one form supported so far:  (raw field...)
;;  which renders a |-seperated line consisting of the given fields,
;;  followed by a newline

;;
;;  takes a user format descriptor and returns a procedure of
;;  one argument which implements the formatting operation
;;

(define (compile-user-format target-class fmt)
  (let ((slots (user-format-accessors target-class)))
    (cond
     ((and (list? fmt)
	   (pair? fmt)
	   (eq? (car fmt) 'raw))
      (let ((grabs (map (lambda (x)
			  (let ((g (assq x slots)))
			    (if g
				(cdr g)
				(error "~s: field not defined for ~s" 
				       x target-class))))
			(cdr fmt))))
	(lambda (item)
	  (let loop ((g grabs)
		     (not-first? #f))
	    (if (null? g)
		(newline)
		(begin
		  (if not-first?
		      (write-char #\|))
		  (display ((car g) item))
		  (loop (cdr g) #t)))))))
     (else
      (error "~s: unrecognized format" fmt)))))

(define-method user-format-accessors ((self <<class>>))
  (append
   (map (lambda (gf)
	  (cons (generic-function-name gf)
		(lambda (rec)
		  (gf rec))))
	(select (curry gf-has-method-on-class? self)
		(user-format-gfs)))
   (map (lambda (sd)
	 (cons (name sd)
	       (lambda (rec)
		 (slot-value sd rec))))
       (slot-descriptors self))))

(define (user-format-gfs)
  (list state cr))

(define (gf-has-method-on-class? class gf)
  (any? (lambda ((meth <method>))
	  (subclass? class (car (function-specializers meth))))
	(generic-function-methods gf)))

;;;
;;;  build the appropriate rendition proc, using
;;;  `default-proc' if no `--format' is specified
;;;
;;;  valid formats look like:
;;;
;;;      --format field ...
;;;      --format "(raw field ...)"

(define (rendition-proc target-class req default-proc)
  (if (assq 'format req)
      (let ((f (cdr (assq 'format req))))
	(if (> (length f) 1)
	    (compile-user-format target-class 
				 (cons 'raw 
				       (map read-str f)))
	    (if (null? f)
		(service-error 497 "missing `--format' arguments")
		(let ((f (read-str (car f))))
		  (if (symbol? f)
		      (compile-user-format target-class (list 'raw f))
		      (compile-user-format target-class f))))))
      default-proc))
