
(define (extract-data png)
  (let ((o (open-output-string)))
    (vector-for-each
     (lambda (ch)
       (if (string=? (type ch) "IDAT")
	   (write-string o (data ch))))
     (chunks png))
     (uncompress (close-output-port o))))


;;; returns => samples/pixel bits/pixel indexed?

(define (bit-metrics png)
  (let ((ctype (get-property png 'color-type))
	(bits (get-property png 'bit-depth)))
    (if (eq? ctype 3)
	(values 1 bits #t)
	(let ((spp (case ctype
		     ((0) 1) ;; gray scale
		     ((2) 3) ;; rgb
		     ((4) 2) ;; gray scale + alpha
		     ((6) 4) ;; rgb + alpha
		     (else
		      (error "png: invalid color type ~s" ctype)))))
	  (values spp (* bits spp) #f)))))

(define (for-each-scanline png proc)
  (bind ((h (get-property png 'height))
	 (w (get-property png 'width))
	 (spp bits/pix (bit-metrics png))
	 (bytes/pix (max 1 (quotient bits/pix 8)))
	 (scanline-bytes (quotient (* bits/pix w) 8))
	 (d (extract-data png)))
    (assert (= (get-property png 'filter-method) 0))
    (let loop ((i 0)
	       (k 0)
	       (prev #f))
      (if (< i h)
	  (let* ((filt-type (bvec-ref d k))
		 (line (unfilter filt-type
				 d
				 (+ k 1) 
				 scanline-bytes
				 bytes/pix prev)))
	    ;(format #t "[~d] " filt-type)
	    (proc i line)
	    (loop (+ i 1)
		  (+ k scanline-bytes 1)
		  line))))))

;;; calls the given `proc' for each pixel in scanline (y-major) order
;;; the proc is called with three arguments:
;;;   `x'  x coordinate (fastest-varying)
;;;   `y'  y coordinate
;;;   `c'  <pixel> value

(define-method for-each-pixel ((png <png-image>) proc)
  (let ((w (get-property png 'width))
	(Bps (quotient (* 4 (get-property png 'bit-depth)) 8)))
    (define (getpix scan x)
      (make <pixel>
	red-component: (* 257 (read-u8 scan x))
	green-component: (* 257 (read-u8 scan (+ x 1)))
	blue-component: (* 257 (read-u8 scan (+ x 2)))
	alpha-component: (* 257 (read-u8 scan (+ x 3)))))
    (for-each-scanline
     png
     (lambda (y scanline)
       (let loop ((i 0)
		  (k 0))
	 (if (< i w)
	     (begin
	       (proc i y (getpix scanline k))
	       (loop (+ i 1) (+ k Bps)))))))))
