;; From some example code provided by
;; "Paul R. Wilson" <wilson@cs.utexas.edu>

;; Here's a simple example of classes and generic procedures.

;; We define class <point> as subclass of <object>, the universal 
;; superclass.  It has fields x and y, because we've chosen to use
;; Cartesian coordinates.  Defining a class with x and y fields will 
;; define appropriate "accessor" methods on the generic procedures
;; x and y, which will fetch the x and y values, as well as the generic
;; procedures and set-x! and set-y!.  It will also create those generic
;; procedures, if they don't already exist, so that it can put the methods 
;; there.

(define-class <point> (<object>)
   x
   y)

;; make one and give it a name
;;
;; (define my-point (make <point> x: 10.0 y: 20.0))
;;
;; x  ;; =>  #[<GP> x] 
;;
;; y  ;; =>  #[<GP> y] 
;;
;; (x my-point)  ;; => 10.0
;;
;; (y my-point)  ;; => 20.0
;;
;; (set-y! my-point 10)
;;
;; (y my-point)  ;; => 10

;; define class <colored-point> as being like <point>,
;; but with a color field

(define-class <colored-point> (<point>)
   color)

;; (define my-colored-point
;;        (make <colored-point> x: 1 y: 2 color: 'red))
;;
;; (color my-colored-point)  ;; => red

;; hypotenuse finds the length of the third side of a right triangle,
;; given the lengths of the two sides at right angles.

(define (hypotenuse a-length b-length)
   (sqrt (+ (expt a-length 2)
            (expt b-length 2))))

;; distance finds the distance between two points, which is the 
;; hypotenuse of the (absolute values of) the differences in their
;; x coordinates and y coordinates.  Note that this is just a plain
;; procedure, not a generic procedure, but it uses the generic 
;; procedures x and y.  It will work as long as the two objects
;; passed to it respond appropriately to the x and y generic
;; procedures; that is, if their classes have methods on those
;; GP's that return their x and y coordinates.

(define (distance pt1 pt2)
   (hypotenuse (abs (- (x pt1) (x pt2)))
               (abs (- (y pt1) (y pt2)))))

;; Now we define a new kind of object, <polar-point>, whose representation
;; is entirely different from the Cartesian coordinates used by <point>.
;; It represents a two-dimensional position as a distance from the origin
;; and an angle (in radians).  (Traditionally the distance and angle are
;; called rho and theta.)
;;
;; Since it's implemented entirely differently from <point>, <cartesian-point>
;; doesn't inherit from <point>.  We "start from scratch" by inheriting
;; only from <object>, the universal superclass.  This situation might
;; also arise if someone else implemented points, and used a different
;; representation, but we wanted to integrate their code into our program.

(define-class <polar-point> (<object>)
   rho
   theta)

;; We can make polar points usable in the same ways as Cartesian points
;; by ensuring that they convert their distance and angle into x and
;; y coordinates as necessary.  For example, to get the x and y coordinates
;; of a polar, point, we can define methods on x and y using (standard
;; Scheme) trigonometric functions.

(define-method x ((pt <polar-point>))
   (* (rho pt)
      (cos (theta pt))))

(define-method y ((pt <polar-point>))
   (* (rho pt)
      (sin (theta pt))))

;; (define my-polar-point (make <polar-point> rho: 10 theta: 0))
;;
;; now we can take the distance between two points, whether they're
;; cartesian or polar, because the distance procedure only depends
;; on its arguments supporting the x and y operations, not on how
;; they're implemented
;;
;; (distance my-point my-polar-point)         ;; => 10.0
;; (distance my-polar-point my-colored-point) ;; => 9.21954
