/*
	Copyright (C) 2006 Ivo van Doorn

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/rfkill.h>

#include <asm/atomic.h>

MODULE_AUTHOR("Ivo van Doorn <IvDoorn@gmail.com>");
MODULE_VERSION("1.0");
MODULE_DESCRIPTION("RF button support");
MODULE_LICENSE("GPL");

/*
 * List of all registered buttons.
 */
static struct list_head rfkill_list;
static spinlock_t rfkill_list_lock;

/*
 * Polling timer, poll_delay and use count.
 */
static struct timer_list poll_timer;
static atomic_t poll_required;

static void rfkill_toggle_radio(int new_status)
{
	struct list_head *entry;
	struct rfkill *rfkill;

	/*
	 * Go through the list of all radio's to toggle the radio state.
	 */
	list_for_each(entry, &rfkill_list) {
		rfkill =  list_entry(entry, struct rfkill, entry);

		rfkill->current_status = new_status;

		/*
		 * If the input_device has been opened
		 * all radio events should be send to user space.
		 */
		if (rfkill->input_dev->users) {
			input_report_key(rfkill->input_dev,
				KEY_RFKILL, new_status);
			input_sync(rfkill->input_dev);
			continue;
		}

		/*
		 * If the hardware does not toggle the radio status automaticly,
		 * we should take care of it.
		 */
		if (new_status && rfkill->enable_radio)
			rfkill->enable_radio(rfkill->data);
		else if (!new_status && rfkill->disable_radio)
			rfkill->disable_radio(rfkill->data);
	}
}

static void rfkill_poll_button(unsigned long data)
{
	struct list_head *entry;
	struct rfkill *rfkill;
	int status;

	spin_lock(&rfkill_list_lock);

	list_for_each(entry, &rfkill_list) {
		rfkill =  list_entry(entry, struct rfkill, entry);

		if (!rfkill->poll)
			continue;

		status = !!rfkill->poll(rfkill->data);

		if (status != rfkill->current_status) {
			rfkill_toggle_radio(status);
			break;
		}
	}

	spin_unlock(&rfkill_list_lock);

	/*
	 * Check if polling is still required.
	 */
	if (atomic_read(&poll_required)) {
		poll_timer.expires = jiffies + RFKILL_POLL_DELAY;
		add_timer(&poll_timer);
	}
}

void rfkill_button_event(struct rfkill *rfkill, int status)
{
	if (status != rfkill->current_status) {
		spin_lock(&rfkill_list_lock);
		rfkill_toggle_radio(status);
		spin_unlock(&rfkill_list_lock);
	}
}
EXPORT_SYMBOL(rfkill_button_event);

int rfkill_add_device(struct rfkill *rfkill)
{
	int status;

	/*
	 * Allocate, initialize and register input device.
	 */
	rfkill->input_dev = input_allocate_device();
	if (!rfkill->input_dev) {
		printk(KERN_ERR "Failed to allocate input device %s.\n",
			rfkill->dev_name);
		return -ENOMEM;
	}

	rfkill->input_dev->name = "Radio button";
	rfkill->input_dev->phys = rfkill->dev_name;
	rfkill->input_dev->id.bustype = BUS_HOST;
	set_bit(KEY_RFKILL, rfkill->input_dev->keybit);

	status = input_register_device(rfkill->input_dev);
	if (status) {
		printk(KERN_ERR "Failed to register input device %s.\n",
			rfkill->dev_name);
		input_free_device(rfkill->input_dev);
		return status;
	}

	INIT_LIST_HEAD(&rfkill->entry);

	spin_lock(&rfkill_list_lock);
	list_add(&rfkill->entry, &rfkill_list);
	spin_unlock(&rfkill_list_lock);

	/*
	 * If polling is required the poll_required counter should be
	 * increased. If it was previously 0 we should start the polling timer.
	 */
	if (rfkill->poll) {
		if (!atomic_read(&poll_required)) {
			poll_timer.expires = jiffies + RFKILL_POLL_DELAY;
			add_timer(&poll_timer);
		}
		atomic_inc(&poll_required);
	}

	return 0;
}
EXPORT_SYMBOL(rfkill_add_device);

void rfkill_del_device(struct rfkill *rfkill)
{
	spin_lock(&rfkill_list_lock);

	/*
	 * Check if this button required polling and if this
	 * was the last button that required polling.
	 */
	if (rfkill->poll && atomic_dec_and_test(&poll_required))
		del_timer(&poll_timer);

	list_del(&rfkill->entry);

	spin_unlock(&rfkill_list_lock);
}
EXPORT_SYMBOL(rfkill_del_device);

static int __init radiobtn_init(void)
{
	printk(KERN_INFO "Loading rfkill driver.\n");

	INIT_LIST_HEAD(&rfkill_list);
	spin_lock_init(&rfkill_list_lock);

	init_timer(&poll_timer);
	poll_timer.function = rfkill_poll_button;
	poll_timer.data = 0;
	atomic_set(&poll_required, 0);

	return 0;
}

static void __exit radiobtn_exit(void)
{
	printk(KERN_INFO "Unloading rfkill driver.\n");
}

module_init(radiobtn_init);
module_exit(radiobtn_exit);
