#include <rumba/arghandler.h>
#include <rumba/numerics.h>
#include <rumba/exception.h>
#include <iostream>
#include <string>
#include <fstream>
#include <cmath>

using namespace RUMBA;

void usage()
{
	std::cerr << "Usage: hrf [--shape1 sh1 --shape2 sh2]"
		"[--scale1 sc1 --scale2 sc2 -o outfile\n"
		"[--tr tr]\n"
		"[--npoints n]";
}
class HRF 
{
public:
	HRF(double shape1, double shape2, double scale1, double scale2)
	: a(shape1), b(shape2), s(scale1), t(scale2), Max(-1)
	{
		init_max();
	}

	double operator()(double x)
	{
		if (x>Last) 
			return 0;
		return f(x)/Max;
	}
private:
	double f(double x)
	{
		return incomplete_gamma( s * x, a ) - incomplete_gamma( t * x, b );
	}

	void init_max()
	{
		double tmp;
		int max_index = 0;
		double x = 1;
		if ( Max > 0 ) return ;

		while (  fabs(f(2*x)) > 0.00001  )
			x *= 2;
		Last = x;

		for ( int i = 0; i < 1000; ++i )
		{
			if ( Max < (tmp = f ( x/1000 * i )) )
			{
				Max = tmp;
				max_index = i;
			}
		}
	}


	double a,b,s,t, Max;
	double Last;
};

int main(int argc, char** argv)
{
	Argument myArgs [] = {
		Argument("shape1",RUMBA::NUMERIC,'\0',5,false),
		Argument("shape2",RUMBA::NUMERIC,'\0',14,false),
		Argument("scale1",RUMBA::NUMERIC,'\0',0.67,false),
		Argument("scale2",RUMBA::NUMERIC,'\0',1.7,false),
		Argument("tr",RUMBA::NUMERIC,'\0',1,false),
		Argument("npoints",RUMBA::NUMERIC,'\0',40,false),
		Argument()
	};


	std::ostream* out;
	std::ofstream fout;
	std::string outfile;
	double shape1, shape2, scale1, scale2, tr, npoints;


	try {
		ArgHandler argh ( argc , argv, myArgs );
		if ( argh.arg("help") )
		{
			usage();
			exit(0);
		}

		try { argh.arg( "outfile" , outfile ); } catch (...) {} 
		argh.arg( "shape1", shape1 );
		argh.arg( "shape2", shape2 );
		argh.arg( "scale1", scale1 );
		argh.arg( "scale2", scale2 );
		argh.arg( "tr", tr );
		argh.arg( "npoints", npoints );

		HRF hrf( shape1,shape2,scale1,scale2 );

		if ( outfile.empty() )
			out = &std::cout;
		else
		{
			fout.open(outfile.c_str());	
			if ( !fout )
				throw RUMBA::Exception ( std::string("couldn't open file") + outfile ) ;
			else
				out = &fout;
		}		

		for ( int i = 0; i < npoints; ++i )
			*out << hrf( i * tr ) << " ";
		*out << std::endl;

	
	}
	catch ( RUMBA::InvalidArgumentException& s)
	{
		std::cerr << "Invalid argument: " << s.error() << std::endl;
	}
    catch (RUMBA::DuplicateArgumentException& s)
    {
		std::cerr << "Duplicate argument: " << s.error() << std::endl;
	}
	catch (RUMBA::ArgHandlerException& s)
	{
		std::cerr << "Error: " << s.error() << std::endl;
	}
	catch (RUMBA::Exception& s)
	{
		std::cerr << "Fatal exception: " << s.error() << std::endl;
	}



}
