#include "mask.h"
const double ThresholdMask::value_false = 0;
const double ThresholdMask::value_true = 1;

using RUMBA::intPoint;

RUMBA::Manifold<char> mask ( MaskFunctor& f, const RUMBA::BaseManifold* M )
{
	intPoint dims(M->extent());
	dims.t()=1;
	RUMBA::Manifold<char> mask (dims);
	for ( int i = 0; i < M->pixels(); ++i )
		mask[i] =  f(i,M);
	return mask;
};



Threshold::Threshold 
( 
 const double* min, 
 const double* max, 
 const double* minreturn, 
 const double* midreturn, 
 const double* maxreturn
)
: 
min(min? new double(*min):0), 
max(max?new double(*max):0),
minreturn(minreturn?new double(*minreturn):0),
midreturn(midreturn?new double(*midreturn):0),
maxreturn(maxreturn?new double(*maxreturn) : 0 )
{
	assert ( !min || minreturn );
	assert ( !max || maxreturn );
}



Threshold::~Threshold()
{ 
	delete min; 
	delete max; 
	delete minreturn; 
	delete midreturn;
	delete maxreturn; 
}

double Threshold::operator() (double x)
{
	if ( min && x <= *min )
		return *minreturn;
	else if ( max && x >= *max )
		return *maxreturn;
	else if ( midreturn )
			return *midreturn;
	else return x;
};


intPoint squashSize (const RUMBA::Manifold<char>& M, RUMBA::BaseManifold* N)
{
	intPoint result(M.extent());
	int p = 0;
		
	// count nonzero mask points 
	for ( int i = 0; i < M.size(); ++i )
		if ( M[i] )
			p++;

	if ( M.timepoints() > 1 )
	{
		result = N->extent();
		result.t() = p;
	}
	else
	{ 
		result.z() = (p-1) / (M.width()*M.height()) + 1;
		result.t() = N->timepoints();
	}
	return result;
}




void t_squash
( const RUMBA::Manifold<char>& mask, const RUMBA::BaseManifold* in, RUMBA::BaseManifold* out)
{
	int j = 0;
	if (mask.timepoints()!=in->timepoints())
		throw RUMBA::Exception("mismatched dimensions in t_squash()");

	if (mask.timepoints() < out->timepoints())
		throw RUMBA::Exception("mask has insufficient timepoints" );
		
	for ( int i = 0; i < mask.size(); ++i )
	{
		if ( mask[i] )
		{
			for ( int k = 0; k < out->pixels(); ++k )
			{
				(*out)[j*out->pixels()+k] = 
					in->getElementDouble(i*in->pixels()+k);
			}
			++j;
		}
	}

	while (j<out->timepoints())
	{
		for ( int k = 0; k < out->pixels(); ++k )
			(*out)[k+j*out->pixels()] = 0;
		++j;
	}
}


void squash 
( 
 const RUMBA::Manifold<char>& mask, 
 const RUMBA::BaseManifold* in, 
 RUMBA::BaseManifold* out
)
{
	int j = 0;

	assert(in!=0);
	assert(out!=0);

	if (mask.timepoints() > 1 )
	{
		t_squash(mask,in,out);
		return;
	}

	if ((in->width()!=mask.width()) || 
   		(in->height()!=mask.height())||
		(in->depth()!=mask.depth()) 
		)
		throw RUMBA::Exception ( "mismatched mask dimensions in squash()" );

	if ( mask.pixels() < out->pixels() )
		throw RUMBA::Exception("mask has insufficient pixels" );

	for ( int i = 0; i < mask.size(); ++i )
	{
		if ( mask[i] )
		{
			for ( int k = 0; k < out->timepoints(); ++k )
			{
				(*out)[j+k*out->pixels()] = 
					in->getElementDouble(i+k*in->pixels());
			}
			++j;
		}
	}

	while (j<out->pixels())
	{
		for ( int k = 0; k < out->timepoints(); ++k )
			(*out)[j+k*out->pixels()] = 0;
		++j;
	}
}


void
unsquash
( 
 const RUMBA::Manifold<char>& mask, 
 const RUMBA::BaseManifold* in,
 RUMBA::BaseManifold* out
 )
{
	int j = 0;
	for ( int i = 0; i < mask.size(); ++i )
	{
		if ( mask[i] )
		{
			for ( int k = 0; k < in->timepoints(); ++k )
			{
				(*out)[i+k*out->pixels()] = 
					in->getElementDouble(j+k*in->pixels());
			}

			++j;
		}
		else
		{
			for ( int k = 0; k < in->timepoints(); ++k )
				(*out)[i+k*out->pixels()] = 0;
					
		}
	}

}

