#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc Bin {bx by} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin factor $bx $by"
	set bin(factor) "[$current(frame) get bin factor]"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc BinAbout {x y} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin about $x $y"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc BinCols {x y z} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) \
	    "$current(frame) bin cols \{$x\} \{$y\} \{$z\}"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc BinFilter {str} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin filter \{\{$str\}\}"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc BinToFit {} {
    global current
    global bin
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin to fit"
	set bin(factor) "[$current(frame) get bin factor]"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc ChangeBinFactor {} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin factor to $bin(factor)"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc ChangeBinDepth {} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin depth $bin(depth)"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc ChangeBinFunction {} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin function $bin(function)"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc ChangeBinBufferSize {} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin buffer size $bin(buffersize)"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc ChangeBinSmooth {} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin smooth $bin(smooth)"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc ChangeBinSmoothFunction {} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin smooth function $bin(smooth,function)"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc ChangeBinSmoothRadius {} {
    global bin
    global current
    global rgb

    if {$current(frame) != ""} {
	SetWatchCursor
	RGBEvalLock rgb(lock,bin) "$current(frame) bin smooth radius $bin(smooth,radius)"

	UpdateBinDialog
	UpdateContourMenu
	UpdateGraphXAxis
	UnsetWatchCursor
    }
}

proc BinDialog {} {
    global ds9
    global bindialog
    global menu

    # see if we already have a window visible

    if [winfo exist $bindialog(top)] {
	raise $bindialog(top)
	return
    }

    set w $bindialog(top)
    set title "Binning Parameters"

    # create the window

    toplevel $w -colormap $ds9(main)
    wm title $w $title
    wm iconname $w $title
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW BinDestroyDialog

    $w configure -menu $bindialog(mb)

    menu $bindialog(mb) -tearoff 0
    $bindialog(mb) add cascade -label File -menu $bindialog(mb).file
    $bindialog(mb) add cascade -label Edit -menu $bindialog(mb).edit
    $bindialog(mb) add cascade -label Method -menu $bindialog(mb).method
    $bindialog(mb) add cascade -label Block -menu $bindialog(mb).block
    $bindialog(mb) add cascade -label Buffer -menu $bindialog(mb).buffer
    $bindialog(mb) add cascade -label Smooth -menu $bindialog(mb).smooth

    menu $bindialog(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $bindialog(mb).file add command -label "Apply" -command BinApplyDialog
    $bindialog(mb).file add command -label "Clear Filter" \
	-command BinClearFilterDialog
    $bindialog(mb).file add separator
    $bindialog(mb).file add command -label "Close" -command BinDestroyDialog

    menu $bindialog(mb).edit -tearoff 0 -selectcolor $menu(selectcolor)
    $bindialog(mb).edit add command -label "Cut" -command BinCutDialog
    $bindialog(mb).edit add command -label "Copy" -command BinCopyDialog
    $bindialog(mb).edit add command -label "Paste" -command BinPasteDialog
    $bindialog(mb).edit add command -label "Clear" -command BinClearDialog

    menu $bindialog(mb).method -tearoff 0 -selectcolor $menu(selectcolor)
    $bindialog(mb).method add radiobutton -label "Average" \
	-variable bin(function) -value average -command ChangeBinFunction
    $bindialog(mb).method add radiobutton -label "Sum" \
	-variable bin(function) -value sum -command ChangeBinFunction

    menu $bindialog(mb).block -tearoff 0 -selectcolor $menu(selectcolor)
    $bindialog(mb).block add command -label "Block In" -command "Bin .5 .5"
    $bindialog(mb).block add command -label "Block Out" -command "Bin 2 2"
    $bindialog(mb).block add separator
    $bindialog(mb).block add command -label "Block to Fit Frame" \
	-command "BinToFit"
    $bindialog(mb).block add separator
    $bindialog(mb).block add radiobutton -label "Block 1" \
	-variable bin(factor) -value { 1 1 } -command ChangeBinFactor
    $bindialog(mb).block add radiobutton -label "Block 2" \
	-variable bin(factor) -value { 2 2 } -command ChangeBinFactor
    $bindialog(mb).block add radiobutton -label "Block 4" \
	-variable bin(factor) -value { 4 4 } -command ChangeBinFactor
    $bindialog(mb).block add radiobutton -label "Block 8" \
	-variable bin(factor) -value { 8 8 } -command ChangeBinFactor
    $bindialog(mb).block add radiobutton -label "Block 16" \
	-variable bin(factor) -value { 16 16 } -command ChangeBinFactor
    $bindialog(mb).block add radiobutton -label "Block 32" \
	-variable bin(factor) -value { 32 32 } -command ChangeBinFactor
    $bindialog(mb).block add radiobutton -label "Block 64" \
	-variable bin(factor) -value { 64 64 } -command ChangeBinFactor
    $bindialog(mb).block add radiobutton -label "Block 128" \
	-variable bin(factor) -value { 128 128 } -command ChangeBinFactor

    menu $bindialog(mb).buffer -tearoff 0 -selectcolor $menu(selectcolor)
    $bindialog(mb).buffer add radiobutton -label "128x128" \
	-variable bin(buffersize) -value 128 -command ChangeBinBufferSize 
    $bindialog(mb).buffer add radiobutton -label "256x256" \
	-variable bin(buffersize) -value 256 -command ChangeBinBufferSize 
    $bindialog(mb).buffer add radiobutton -label "512x512" \
	-variable bin(buffersize) -value 512 -command ChangeBinBufferSize 
    $bindialog(mb).buffer add radiobutton -label "1024x1024" \
	-variable bin(buffersize) -value 1024 -command ChangeBinBufferSize
    $bindialog(mb).buffer add radiobutton -label "2048x2048" \
	-variable bin(buffersize) -value 2048 -command ChangeBinBufferSize
    $bindialog(mb).buffer add radiobutton -label "4096x4096" \
	-variable bin(buffersize) -value 4096 -command ChangeBinBufferSize

    menu $bindialog(mb).smooth -tearoff 0 -selectcolor $menu(selectcolor)
    $bindialog(mb).smooth add checkbutton -label "Smooth" \
	-variable bin(smooth) -command ChangeBinSmooth
    $bindialog(mb).smooth add separator
    $bindialog(mb).smooth add radiobutton -label "BoxCar" \
	-variable bin(smooth,function) -value boxcar \
	-command ChangeBinSmoothFunction
    $bindialog(mb).smooth add radiobutton -label "TopHat" \
	-variable bin(smooth,function) -value tophat \
	-command ChangeBinSmoothFunction
   $bindialog(mb).smooth add  radiobutton -label "Gaussian" \
	-variable bin(smooth,function) -value gaussian \
	-command ChangeBinSmoothFunction

    frame $w.cols -relief groove -borderwidth 2
    frame $w.center -relief groove -borderwidth 2
    frame $w.filter -relief groove -borderwidth 2
    frame $w.z -relief groove -borderwidth 2
    frame $w.smooth -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2

    pack $w.cols $w.center $w.filter $w.z $w.smooth -fill x
    pack $w.buttons -fill x -ipadx 4 -ipady 4

    frame $w.cols.f
    frame $w.center.f
    frame $w.filter.f
    frame $w.z.f
    frame $w.smooth.f
    pack $w.cols.f $w.center.f $w.filter.f $w.z.f $w.smooth.f \
	-ipadx 4 -ipady 4 -anchor w

    # Columns

    label $w.cols.f.title -text "Bin Columns   " -width 13
    label $w.cols.f.titlefactor -text "Block"
    label $w.cols.f.titlemin -text "Min"
    label $w.cols.f.titlemax -text "Max"
    menubutton $w.cols.f.x -textvariable bindialog(xcol) -menu $w.cols.f.x.m \
	-relief raised -bd 2 -width 10
    entry $w.cols.f.xfactor -textvariable bindialog(factor,x) -width 8
    label $w.cols.f.xmin -textvariable bindialog(xcol,min) -width 12 \
	-relief groove
    label $w.cols.f.xmax -textvariable bindialog(xcol,max) -width 12 \
	-relief groove
    menubutton $w.cols.f.y -textvariable bindialog(ycol) -menu $w.cols.f.y.m \
	-relief raised -bd 2 -width 10
    entry $w.cols.f.yfactor -textvariable bindialog(factor,y) -width 8 
    label $w.cols.f.ymin -textvariable bindialog(ycol,min) -width 12 \
	-relief groove
    label $w.cols.f.ymax -textvariable bindialog(ycol,max) -width 12 \
	-relief groove

    grid rowconfigure $w.cols.f 0 -pad 4
    grid rowconfigure $w.cols.f 1 -pad 4
    grid rowconfigure $w.cols.f 2 -pad 4

    grid $w.cols.f.title $w.cols.f.titlefactor $w.cols.f.titlemin \
	$w.cols.f.titlemax -padx 4 -sticky ew
    grid $w.cols.f.x $w.cols.f.xfactor $w.cols.f.xmin $w.cols.f.xmax \
	-padx 4 -sticky w
    grid $w.cols.f.y $w.cols.f.yfactor $w.cols.f.ymin $w.cols.f.ymax \
	-padx 4 -sticky w

    # Center

    label $w.center.f.title -text "Bin Center      " -width 13
    entry $w.center.f.x -textvariable bindialog(x) -width 12 	 
    entry $w.center.f.y -textvariable bindialog(y) -width 12 
    checkbutton $w.center.f.auto -text "or center of data" \
	-selectcolor $menu(selectcolor) -variable bindialog(auto)

    grid rowconfigure $w.center.f 0 -pad 4
    grid rowconfigure $w.center.f 1 -pad 4

    grid $w.center.f.title $w.center.f.x $w.center.f.y -padx 4 -sticky w
    grid x $w.center.f.auto - -padx 4 -sticky w

    # Filter

    label $w.filter.f.title -text "Bin Filter        " -width 13
    entry $w.filter.f.filter -textvariable bindialog(filter) -width 36

    grid rowconfigure $w.filter.f 0 -pad 4
    grid $w.filter.f.title $w.filter.f.filter -padx 4 -sticky w

    # Bin 3rd Column

    label $w.z.f.title -text "Bin 3rd Column" -width 13
    label $w.z.f.titledepth -text "Depth"
    label $w.z.f.titlemin -text "Min"
    label $w.z.f.titlemax -text "Max"

    menubutton $w.z.f.z -textvariable bindialog(zcol) \
	-menu $w.z.f.z.m -relief raised -bd 2 -width 10
    entry $w.z.f.depth -textvariable bindialog(depth) -width 8
    entry $w.z.f.min -textvariable bindialog(zcol,min) -width 12 
    entry $w.z.f.max -textvariable bindialog(zcol,max) -width 12 

    grid rowconfigure $w.z.f 0 -pad 4
    grid rowconfigure $w.z.f 1 -pad 4

    grid $w.z.f.title $w.z.f.titledepth $w.z.f.titlemin $w.z.f.titlemax \
	-padx 4 -sticky ew
    grid $w.z.f.z $w.z.f.depth $w.z.f.min $w.z.f.max -padx 4 -sticky w

    # Smooth

    scale $w.smooth.f.slider  -from 1 -to 20 -length 300 \
	-label "Smooth Kernel Radius" \
	-variable bindialog(smooth,radius) \
	-orient horizontal \
	-tickinterval 2 -showvalue false -resolution 1
    entry $w.smooth.f.value -textvariable bindialog(smooth,radius) -width 7 

    pack $w.smooth.f.slider $w.smooth.f.value -side left -padx 4

    button $w.buttons.apply -text "Apply" -command BinApplyDialog
    button $w.buttons.clear -text "Clear Filter" -command BinClearFilterDialog
    button $w.buttons.close -text "Close" -command BinDestroyDialog
    pack $w.buttons.apply $w.buttons.clear $w.buttons.close \
	-side left -padx 10 -expand true

    $w.cols.f.xfactor select range 0 end

    set bindialog(auto) 0
    set bindialog(minmax) 1

    UpdateBinDialog
}

proc PopUp {b m l cmd} {
    destroy $m

    menu $m -tearoff 0
    for {set i 0} {$i<[llength $l]} {incr i} {
	$m add command -label [lindex $l $i] \
	    -command "global bindialog;set $b [lindex $l $i]; $cmd"
    }
}

proc BlankPopUp {m} {
    destroy $m
    menu $m -tearoff 0
}

proc UpdateBinDialog {} {
    global bin
    global bindialog
    global current

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateBinDialog"
    }

    if {![winfo exist $bindialog(top)]} {
	return
    }
    if {$current(frame) == ""} {
	return
    }

    set w $bindialog(top)

    if {[$current(frame) has bin]} {
	set bf "[$current(frame) get bin factor]"
	set bindialog(factor,x) [lindex $bf 0]
	set bindialog(factor,y) [lindex $bf 1]

	set cols [$current(frame) get bin cols]
	set colslist "[$current(frame) get bin list]"

	set bindialog(xcol) [lindex $cols 0]
	set bindialog(ycol) [lindex $cols 1]
	PopUp bindialog(xcol) $w.cols.f.x.m $colslist UpdateXCol
	PopUp bindialog(ycol) $w.cols.f.y.m $colslist UpdateYCol

	set mm [$current(frame) get bin cols minmax  \{$bindialog(xcol)\}]
	set bindialog(xcol,min) [lindex $mm 0]
	set bindialog(xcol,max) [lindex $mm 1]

	set mm [$current(frame) get bin cols minmax \{$bindialog(ycol)\}]
	set bindialog(ycol,min) [lindex $mm 0]
	set bindialog(ycol,max) [lindex $mm 1]

	set cursor [$current(frame) get bin cursor]
	set bindialog(x) [lindex $cursor 0]
	set bindialog(y) [lindex $cursor 1]

	set bindialog(filter) [$current(frame) get bin filter]

	set bindialog(depth) [$current(frame) get bin depth]
	set bindialog(zcol) [lindex $cols 2]
	PopUp bindialog(zcol) $w.z.f.z.m $colslist UpdateZCol
	set mm [$current(frame) get bin cols minmax \{$bindialog(zcol)\}]
	set bindialog(zcol,min) [lindex $mm 0]
	set bindialog(zcol,max) [lindex $mm 1]

	set bindialog(smooth,radius) [$current(frame) get bin smooth radius]
    } else {
	set bindialog(factor,x) {}
	set bindialog(factor,y) {}
	set bindialog(xcol) {}
	set bindialog(xcol,min) {}
	set bindialog(xcol,max) {}
	set bindialog(ycol) {}
	set bindialog(ycol,min) {}
	set bindialog(ycol,max) {}

	set bindialog(x) {}
	set bindialog(y) {}

	set bindialog(filter) {}

	set bindialog(depth) {}
	set bindialog(zcol) {}
	set bindialog(zcol,min) {}
	set bindialog(zcol,max) {}

	BlankPopUp $w.cols.f.x.m
	BlankPopUp $w.cols.f.y.m
	BlankPopUp $w.z.f.z.m

	set bindialog(smooth,radius) 0
    }
}

proc UpdateXCol {} {
    global current
    global bindialog

    if {$current(frame) != ""
	&& [$current(frame) has bin]
	&& $bindialog(xcol) != ""} {

	set mm [$current(frame) get bin cols minmax \{$bindialog(xcol)\}]
	set bindialog(xcol,min) [lindex $mm 0]
	set bindialog(xcol,max) [lindex $mm 1]
    } else {
	    set bindialog(xcol,min) {}
	    set bindialog(xcol,max) {}
    }
}

proc UpdateYCol {} {
    global current
    global bindialog

    if {$current(frame) != ""
	&& [$current(frame) has bin]
	&& $bindialog(ycol) != ""} {

	set mm [$current(frame) get bin cols minmax \{$bindialog(ycol)\}]
	set bindialog(ycol,min) [lindex $mm 0]
	set bindialog(ycol,max) [lindex $mm 1]
    } else {
	    set bindialog(ycol,min) {}
	    set bindialog(ycol,max) {}
    }
}

proc UpdateZCol {} {
    global current
    global bindialog

    if {$current(frame) != ""
	&& [$current(frame) has bin]
	&& $bindialog(zcol) != ""} {

	if {$bindialog(minmax)} {
	    set mm [$current(frame) get bin cols minmax \{$bindialog(zcol)\}]
	    set bindialog(zcol,min) [lindex $mm 0]
	    set bindialog(zcol,max) [lindex $mm 1]
	}
    } else {
	    set bindialog(zcol,min) {}
	    set bindialog(zcol,max) {}
    }
}

proc BinApplyDialog {} {
    global current
    global bin
    global bindialog

    if {$current(frame) == {}} {
	# reset
	set bindialog(auto) 0
	return
    }

    # clean up filter if needed
    set bindialog(filter) [string trimleft $bindialog(filter)]
    set bindialog(filter) [string trimright $bindialog(filter)]

    # delete any markers if needed
    if {[$current(frame) has bin]} {
	set foo [$current(frame) get bin cols]
	set xcol [lindex $foo 0]
	set ycol [lindex $foo 1]

	if {$xcol != $bindialog(xcol) || $ycol != $bindialog(ycol)} {
	    $current(frame) marker delete all
	}
    }

    SetWatchCursor

    if {$bindialog(depth)>1} {
	CubeDialog

	if {$bindialog(auto)} {
	    if {$bindialog(factor,x) != {}
		&& $bindialog(factor,y) != {}
		&& $bindialog(depth) != {}
		&& $bindialog(zcol,min) != {}
		&& $bindialog(zcol,max) != {}
		&& $bindialog(xcol) != {}
		&& $bindialog(ycol) != {}
		&& $bindialog(zcol) != {}} {

		$current(frame) bin to \
		    $bindialog(factor,x) $bindialog(factor,y)\
		    $bindialog(depth) \
		    $bindialog(zcol,min) $bindialog(zcol,max) \
		    about center \
		    \{$bindialog(xcol)\} \{$bindialog(ycol)\} \
		    \{$bindialog(zcol)\} \{$bindialog(filter)\} \
		    $bin(smooth) $bin(smooth,function) \
		    $bindialog(smooth,radius)
	    }
	} else {
	    if {$bindialog(factor,x) != {}
		&& $bindialog(factor,y) != {}
		&& $bindialog(depth) != {}
		&& $bindialog(zcol,min) != {}
		&& $bindialog(zcol,max) != {}
		&& $bindialog(x) != {}
		&& $bindialog(y) != {}
		&& $bindialog(xcol) != {}
		&& $bindialog(ycol) != {}
		&& $bindialog(zcol) != {}} {

		$current(frame) bin to \
		    $bindialog(factor,x) $bindialog(factor,y) \
		    $bindialog(depth) \
		    $bindialog(zcol,min) $bindialog(zcol,max) \
		    about $bindialog(x) $bindialog(y) \
		    \{$bindialog(xcol)\} \{$bindialog(ycol)\} \
		    \{$bindialog(zcol)\} \{$bindialog(filter)\} \
		    $bin(smooth) $bin(smooth,function) \
		    $bindialog(smooth,radius)
	    }
	}
    } else {
	if {$bindialog(auto)} {
	    if {$bindialog(factor,x) != {}
		&& $bindialog(factor,y) != {}
		&& $bindialog(xcol) != {}
		&& $bindialog(ycol) != {}} {

		$current(frame) bin to \
		    $bindialog(factor,x) $bindialog(factor,y) \
		    about center \
		    \{$bindialog(xcol)\} \{$bindialog(ycol)\} \
		    \{$bindialog(filter)\} \
		    $bin(smooth) $bin(smooth,function) \
		    $bindialog(smooth,radius)
	    }
	} else {
	    if {$bindialog(factor,x) != {}
		&& $bindialog(factor,y) != {}
		&& $bindialog(x) != {}
		&& $bindialog(y) != {}
		&& $bindialog(xcol) != {}
		&& $bindialog(ycol) != {}} {

		$current(frame) bin to \
		    $bindialog(factor,x) $bindialog(factor,y) \
		    about $bindialog(x) $bindialog(y) \
		    \{$bindialog(xcol)\} \{$bindialog(ycol)\} \
		    \{$bindialog(filter)\} \
		    $bin(smooth) $bin(smooth,function) \
		    $bindialog(smooth,radius)
	    }
	}
    }

    UpdateContourMenu
    UpdateCubeDialog
    UpdateGraphXAxis
    UpdateScaleDialog
    UnsetWatchCursor

    # the bining center will shift
    UpdateBinDialog
    UpdateBinMenu

    # reset
    set bindialog(auto) 0
}

proc BinClearFilterDialog {} {
    global bindialog

    $bindialog(top).filter.f.filter delete 0 end
}

proc BinDestroyDialog {} {
    global bindialog

    destroy $bindialog(top)
    destroy $bindialog(mb)
}

proc BinCutDialog {} {
    global bindialog

    set w [focus -displayof $bindialog(top)]

    if {![catch {set data [string range [$w get] [$w index sel.first] [expr {[$w index sel.last] - 1}]]}]} {
        clipboard clear -displayof $w
        clipboard append -displayof $w $data
        $w delete sel.first sel.last
    }
}

proc BinCopyDialog {} {
    global bindialog

    set w [focus -displayof $bindialog(top)]

    if {![catch {set data [string range [$w get] [$w index sel.first] [expr {[$w index sel.last] - 1}]]}]} {
        clipboard clear -displayof $w
        clipboard append -displayof $w $data
    }
}

proc BinPasteDialog {} {
    global tcl_platform
    global bindialog

    set w [focus -displayof $bindialog(top)]

    catch {
        if {"$tcl_platform(platform)" != "unix"} {
            catch {
                $w delete sel.first sel.last
            }
        }
        $w insert insert [selection get -displayof $w -selection CLIPBOARD]
        tkEntrySeeInsert $w
    }
}

proc BinClearDialog {} {
    global bindialog

    set w [focus -displayof $bindialog(top)]

    if {![catch {selection get -displayof $bindialog(top)}]} {
	$w delete sel.first sel.last
    }
}

proc UpdateBinMenu {} {
    global current
    global bin
    global buttons
    global ds9
    global bin
    global menu

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateBinMenu"
    }

    if {$current(frame) != ""} {
	if {[$current(frame) has fits]} {
	    if {[$current(frame) has bin]} {
		$ds9(mb) entryconfig $menu(bin) -state normal
		$buttons(name).major.bin1 configure -state normal
	    } else {
		$ds9(mb) entryconfig $menu(bin) -state disabled
		$buttons(name).major.bin1 configure -state disabled
	    }
	} else {
	    $ds9(mb) entryconfig $menu(bin) -state normal
	    $buttons(name).major.bin1 configure -state normal
	}

	set bin(function) [$current(frame) get bin function]
	set bin(factor) "[$current(frame) get bin factor]"
	set bin(depth) [$current(frame) get bin depth]
	set bin(buffersize) [$current(frame) get bin buffer size]
	set bin(smooth) [$current(frame) get bin smooth]
	set bin(smooth,function) [$current(frame) get bin smooth function]
	set bin(smooth,radius) [$current(frame) get bin smooth radius]
    } else {
	$ds9(mb) entryconfig $menu(bin) -state disabled
	$buttons(name).major.bin1 configure -state disabled
    }
}

proc ProcessBinCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global bin

    switch -- [string tolower [lindex $var $i]] {
	about {
	    BinAbout [lindex $var [expr $i+1]] [lindex $var [expr $i+2]]
	    incr i 2
	}
	buffersize {
	    incr i
	    set bin(buffersize) [lindex $var $i]
	    ChangeBinBufferSize
	}
	cols {
	    BinCols \"[lindex $var [expr $i+1]]\" \"[lindex $var [expr $i+2]]\" \"[lindex $var [expr $i+3]]\"
	    incr i 3
	}
	factor {
	    incr i
	    set bx [lindex $var $i]
	    incr i
	    set by [lindex $var $i]
	    # note: the spaces are needed so that the menus are in sync
	    if {$by != {}} {
		set bin(factor) " $bx $by "
	    } else {
		set bin(factor) " $bx $bx "
	    }
	    ChangeBinFactor
	}
	depth {
	    incr i
	    set bin(depth) [lindex $var $i]
	    ChangeBinDepth
	}
	filter {
	    incr i
	    BinFilter [lindex $var $i]
	}
	function {
	    incr i
	    set bin(function) [string tolower [lindex $var $i]]
	    ChangeBinFunction
	}
	to {
	    # eat the 'fit'
	    incr i
	    BinToFit
	}
	smooth {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		radius {
		    incr i
		    set bin(smooth,radius) [lindex $var $i]
		    ChangeBinSmoothRadius
		}
		function {
		    incr i
		    set bin(smooth,function) [lindex $var $i]
		    ChangeBinSmoothFunction
		}
		default {
		    set bin(smooth) [FromYesNo [lindex $var $i]]
		    ChangeBinSmooth
		}
	    }
	}
    }
}

