#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc DSSDialog {} {
    global dss
    global nsvr
    global ds9
    global menu
    global current

    if [winfo exists $dss(top)] {
	raise $dss(top)
	return
    }

    set w $dss(top)
    set title "DSS Server"

    # create the window

    toplevel $w -colormap $ds9(main)
    wm title $w $title
    wm iconname $w $title
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW DSSDestroyDialog

    # menu

    $w configure -menu $dss(mb)
    menu $dss(mb) -tearoff 0
    $dss(mb) add cascade -label File -menu $dss(mb).file
    $dss(mb) add cascade -label Edit -menu $dss(mb).edit
    $dss(mb) add cascade -label "DSS Server" -menu $dss(mb).server
    $dss(mb) add cascade -label "Name Server" -menu $dss(mb).name
    $dss(mb) add cascade -label Survey -menu $dss(mb).survey
    $dss(mb) add cascade -label Prefs -menu $dss(mb).prefs

    menu $dss(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $dss(mb).file add command -label "Update from Current Frame" \
	-command UpdateDSSDialog
    $dss(mb).file add command -label Acknowledgement -command DSSAck
    $dss(mb).file add separator
    $dss(mb).file add command -label Close -command DSSDestroyDialog

    menu $dss(mb).edit -tearoff 0 -selectcolor $menu(selectcolor)
    $dss(mb).edit add command -label Cut -state disabled
    $dss(mb).edit add command -label Copy -state disabled
    $dss(mb).edit add command -label Paste -state disabled
    $dss(mb).edit add command -label Clear -command DSSClearDialog

    menu $dss(mb).server -tearoff 0 -selectcolor $menu(selectcolor)
    $dss(mb).server add radiobutton -label "DSS@SAO" \
	-variable dss(server) -value sao -command DSSServerDialog
    $dss(mb).server add radiobutton -label "DSS@STSCI" \
	-variable dss(server) -value stsci -command DSSServerDialog
    $dss(mb).server add radiobutton -label "DSS@ESO" \
	-variable dss(server) -value eso -command DSSServerDialog

    menu $dss(mb).name -tearoff 0 -selectcolor $menu(selectcolor)
    $dss(mb).name add radiobutton -label "NED@SAO" \
	-variable nsvr(server) -value ned-sao
    $dss(mb).name add radiobutton -label "NED@ESO" \
	-variable nsvr(server) -value ned-eso
    $dss(mb).name add radiobutton -label "SIMBAD@SAO" \
	-variable nsvr(server) -value simbad-sao
    $dss(mb).name add radiobutton -label "SIMBAD@ESO" \
	-variable nsvr(server) -value simbad-eso

    menu $dss(mb).survey -tearoff 0 -selectcolor $menu(selectcolor)
    $dss(mb).survey add radiobutton -label "DSS1" \
	-variable dss(survey) -value dss
    $dss(mb).survey add radiobutton -label "DSS2-red" \
	-variable dss(survey) -value dss2red
    $dss(mb).survey add radiobutton -label "DSS2-blue" \
	-variable dss(survey) -value dss2blue

    menu $dss(mb).prefs -tearoff 0 -selectcolor $menu(selectcolor)
    $dss(mb).prefs add checkbutton -label "Save FITS on download" \
	-variable dss(save)
    $dss(mb).prefs add separator
    $dss(mb).prefs add radiobutton -label "New Frame" \
	-variable dss(frame) -value new
    $dss(mb).prefs add radiobutton -label "Current Frame" \
	-variable dss(frame) -value current

    frame $w.param -relief groove -borderwidth 2
    frame $w.status -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.status $w.buttons -fill x -expand true -ipadx 4 -ipady 4

    label $w.param.nametitle -text "Name"
    entry $w.param.name -textvariable dss(name) -width 38

    set dss(xname) [label $w.param.xtitle -text "" -font {symbol 12} -width 1]
    entry $w.param.x -textvariable dss(x) -width 14
    set dss(yname) [label $w.param.ytitle -text "" -font {symbol 12} -width 1]
    entry $w.param.y -textvariable dss(y) -width 14
    label $w.param.system -textvariable dss(sky) -width 10 -relief groove

    label $w.param.wtitle -text "Width"
    entry $w.param.w -textvariable dss(width) -width 14
    label $w.param.htitle -text "Height"
    entry $w.param.h -textvariable dss(height) -width 14
    label $w.param.format -textvariable dss(skyformat) -width 10 -relief groove

    grid rowconfigure $w.param 0 -pad 4
    grid rowconfigure $w.param 1 -pad 4
    grid rowconfigure $w.param 2 -pad 4

    grid $w.param.nametitle -row 0 -column 0 -padx 4 -pady 1 -sticky w
    grid $w.param.name -row 0 -column 1 -columnspan 4 -padx 4 -pady 1 -sticky w

    grid $w.param.xtitle -row 1 -column 0 -padx 4 -pady 1 -sticky w
    grid $w.param.x -row 1 -column 1 -padx 4 -pady 1 -sticky w
    grid $w.param.ytitle -row 1 -column 2 -padx 4 -pady 1 -sticky w
    grid $w.param.y -row 1 -column 3 -padx 4 -pady 1 -sticky w
    grid $w.param.system -row 1 -column 4 -padx 4 -pady 1 -sticky w

    grid $w.param.wtitle -row 2 -column 0 -padx 4 -pady 1 -sticky w
    grid $w.param.w -row 2 -column 1 -padx 4 -pady 1 -sticky w
    grid $w.param.htitle -row 2 -column 2 -padx 4 -pady 1 -sticky w
    grid $w.param.h -row 2 -column 3 -padx 4 -pady 1 -sticky w
    grid $w.param.format -row 2 -column 4 -padx 4 -pady 1 -sticky w

    label $w.status.item -textvariable dss(status)
    pack $w.status.item -anchor w -pady 4

    set dss(apply) \
	[button $w.buttons.apply -text "Retrieve" -command DSSApplyDialog]
    set dss(cancel) \
	[button $w.buttons.cancel -text "Cancel" -command DSSCancelDialog \
	     -state disabled]
    button $w.buttons.close -text "Close" -command DSSDestroyDialog
    pack $w.buttons.apply $w.buttons.cancel $w.buttons.close \
	-side left -padx 10 -expand true

    set dss(current) {}

    UpdateDSSDialog
    DSSCoordDialog
    DSSServerDialog
    DSSStatus {}
}

proc DSSDestroyDialog {} {
    global dss

    if {[info exists dss(token)]} {
	set dss(state) 0
	http::reset $dss(token)
    }

    destroy $dss(top)
    destroy $dss(mb)

    unset dss(xname)
    unset dss(yname)
    unset dss(cancel)
    unset dss(apply)
    unset dss(status)
    unset dss(current)
}

proc DSSApplyDialog {} {
    global dss
    global ds9

    DSSStatus {}
    $dss(apply) configure -state disabled
    $dss(cancel) configure -state normal

    if {($dss(name) != "")} {
	DSSNSVR
    } else {
	DSSServer
    }
}

proc DSSCancelDialog {} {
    global dss

    if {[info exists dss(token)]} {
	set dss(state) 0
	http::reset $dss(token)
    }
}

# NSVR

proc DSSNSVR {} {
    global dss
    global nsvr
    global http

    DSSStatus "Looking up $dss(name)"
    set dss(x) {}
    set dss(y) {}
    set dss(system) wcs
    set dss(sky) fk5
    DSSCoordDialog

    switch -- $nsvr(server) {
	ned-sao {
	    set query "[regsub -all {\ } $dss(name) {%20}]"
	    set url "http://cfa-www.harvard.edu/catalog/ned"
	    set cmd DSSNEDSAO
	}
	ned-eso {
	    set query "&o=[regsub -all {\ } $dss(name) {%20}]"
	    set url "http://archive.eso.org/skycat/servers/ned-server"
	    set cmd DSSNEDESO
	}
	simbad-sao {
	    set query "[regsub -all {\ } $dss(name) {%20}]"
	    set url "http://cfa-www.harvard.edu/catalog/simbad-cfa"
	    set cmd DSSSIMBADSAO
	}
	simbad-eso {
	    set query "&o=[regsub -all {\ } $dss(name) {%20}]"
	    set url "http://archive.eso.org/skycat/servers/sim-server"
	    set cmd DSSSIMBADESO
	}
    }

    set dss(token) [eval "http::geturl $url?$query -command $cmd -headers \{[ProxyHTTP]\}"]
    set dss(state) 1
}

proc DSSNSVRParse {data} {
    global dss

    set dss(x) [lindex $data 0]
    set dss(y) [lindex $data 1]

    if {($dss(x) == "") || ($dss(y) == "")} {
	set dss(x) {}
	set dss(y) {}
	DSSStatus "$dss(name) not found"
	DSSReset
    } else {
	DSSServer
    }
}

proc DSSReset {} {
    global dss

    set dss(state) 0
    catch {unset dss(token)}

    $dss(apply) configure -state normal
    $dss(cancel) configure -state disabled
}

proc DSSError {code} {
    DSSStatus "Error code $code was returned"
    DSSReset
}

proc DSSNEDSAO {token} {
    global dss

    HTTPLog $token
    if {$dss(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    DSSError $code
	    return
	}

	# data is in sexagesmal
	set data [http::data $token]
	set r [lindex $data 0]
	set d  [lindex $data 1]
	if {($r != "") && ($d != "")} {
	    DSSNSVRParse "$r $d"
	} else {
	    DSSNSVRParse {}
	}
    } else {
	DSSStatus {Cancelled}
	DSSReset
    }
}

proc DSSNEDESO {token} {
    global dss

    HTTPLog $token
    if {$dss(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    DSSError $code
	    return
	}

	# data is in degrees
	set data [http::data $token]
	set l [llength $data]
	set r [lindex $data [expr $l-3]]
	set d [lindex $data [expr $l-2]]
	if {[string is double -strict $r] && [string is double -strict $d]} {
	    DSSNSVRParse "[format %.3@ [expr d2h($r)]] [format %.3@ $d]"
	} else {
	    DSSNSVRParse {}
	}
    } else {
	DSSStatus {Cancelled}
	DSSReset
    }
}

proc DSSSIMBADSAO {token} {
    global dss

    HTTPLog $token
    if {$dss(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    DSSError $code
	    return
	}

	# data is in sexagesmal
	set data [http::data $token]
	set r [lindex $data 0]
	set d  [lindex $data 1]
	if {($r != "") && ($d != "")} {
	    DSSNSVRParse "$r $d"
	} else {
	    DSSNSVRParse {}
	}
    } else {
	DSSStatus {Cancelled}
	DSSReset
    }
}

proc DSSSIMBADESO {token} {
    global dss

    HTTPLog $token
    if {$dss(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    DSSError $code
	    return
	}

	# data is in degrees
	set data [http::data $token]
	set l [llength $data]
	set r [lindex $data [expr $l-3]]
	set d [lindex $data [expr $l-2]]
	if {[string is double -strict $r] && [string is double -strict $d]} {
	    DSSNSVRParse "[format %.3@ [expr d2h($r)]] [format %.3@ $d]"
	} else {
	    DSSNSVRParse {}
	}
    } else {
	DSSStatus {Cancelled}
	DSSReset
    }
}

# DSS Server

proc DSSServer {} {
    global dss
    global ds9

    if {($dss(x) != "") && ($dss(y) != "")} {
	switch -- $dss(frame) {
	    new {
		set ds9(display,user) tile
		DisplayMode
		CreateFrame
	    }
	    current {}
	}

	DSSStatus "Contacting DSS Image Server"
	switch -- $dss(server) {
	    sao {DSSSAO}
	    stsci {DSSSTSCI}
	    eso {DSSESO}
	}
    } else {
	DSSStatus "Please specify Coordinates"

	$dss(apply) configure -state normal
	$dss(cancel) configure -state disabled
    }
}

proc DSSSAO {} {
    global dss
    global ds9

    if {$dss(save)} {
	set compress no
	set dss(fn) [SaveFileDialog dssfbox]
	if {$dss(fn) == {}} {
	    return
	}

    } else {
	set compress gzip
	set dss(fn) [tmpnam dss ".fits.gz"]
    }

    if {$dss(width)>60} {
	set dss(width) 60
    }
    if {$dss(height)>60} {
	set dss(height) 60
    }
    set query [http::formatQuery r $dss(x) d $dss(y) e J2000 \
		   w $dss(width) h $dss(height) \
		   c $compress]

    # Load image

    LoadDSS "http://archive.harvard.edu/archive/dss" $query
}

proc DSSESO {} {
    global dss
    global ds9

    switch -- $dss(survey) {
	dss {set survey "DSS1"}
	dss2red {set survey "DSS2-red"}
	dss2blue {set survey "DSS2-blue"}
    }

    if {$dss(save)} {
	set mime "application/x-fits"
	set dss(fn) [SaveFileDialog dssfbox]
	if {$dss(fn) == {}} {
	    return
	}
    } else {
	set mime "display/gz-fits"
	set dss(fn) [tmpnam dss ".fits.gz"]
    }

    if {$dss(width)>40} {
	set dss(width) 40
    }
    if {$dss(height)>40} {
	set dss(height) 40
    }
    set query [http::formatQuery ra $dss(x) dec $dss(y) equinox J2000\
		   x $dss(width) y $dss(height) \
		   mime-type "$mime" \
		   Sky-Survey $survey]

    # Load image
    # we can't use -query because eso needs a GET not a POST

    LoadDSS "http://archive.eso.org/dss/dss?$query" {}
}

proc DSSSTSCI {} {
    global dss
    global ds9

    switch -- $dss(survey) {
	dss {set survey "1"}
	dss2red {set survey "2r"}
	dss2blue {set survey "2b"}
    }

    if {$dss(save)} {
	set compress none
	set dss(fn) [SaveFileDialog dssfbox]
	if {$dss(fn) == {}} {
	    return
	}

    } else {
	set compress gz
	set dss(fn) [tmpnam dss ".fits.gz"]
    }

    if {$dss(width)>60} {
	set dss(width) 60
    }
    if {$dss(height)>60} {
	set dss(height) 60
    }
    set query [http::formatQuery r $dss(x) d $dss(y) e J2000\
		   w $dss(width) h $dss(height) \
		   f fits c $compress \
		   v $survey]

    # Load image

    LoadDSS "http://stdatu.stsci.edu/cgi-bin/dss_search" $query
}

proc LoadDSS {url query} {
    global dss
    global http

    StartLoad

    set dss(ch) [open "$dss(fn)" w]
    set dss(state) 1

    set cmd "http::geturl $url -channel $dss(ch) -command LoadDSSFinish	-progress DSSProgress -binary 1 -headers \{[ProxyHTTP]\}"

    if {$query != ""} {
	append cmd " -query $query"
    }

    set dss(token) [eval $cmd]
}

proc LoadDSSFinish {token} {
    global dss
    global loadParam

    HTTPLog $token
    if {$dss(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    DSSError $code
	    return
	}

	# alloc it because we are going to delete it after load
	set loadParam(load,type) allocgz
	set loadParam(file,type) fits
	set loadParam(file,mode) {}
	ConvertFile $dss(fn)
	ProcessLoad

	catch {close $dss(ch)}

	if {!$dss(save)} {
	    if {[file exists $dss(fn)]} {
		catch {file delete -force $dss(fn)}
	    }
	}
	FinishLoad
	DSSStatus {Done}
    } else {
	catch {close $dss(ch)}

	UnsetWatchCursor
	DSSStatus {Cancelled}
    }
    set dss(state) 0

    catch {unset dss(ch)}
    catch {unset dss(token)}

    $dss(apply) configure -state normal
    $dss(cancel) configure -state disabled
}

# Other procedures

proc UpdateDSSDialog {} {
    global current
    global dss

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateDSSDialog"
    }

    if {[winfo exist $dss(top)]} {
	if {$current(frame) != "" } {
	    set fn [$current(frame) get fits file name]
	    if {$fn != $dss(current)} {
		set dss(current) $fn
		if {[$current(frame) has wcs equatorial $dss(system)]} {
		    set coord [$current(frame) get cursor \
				   $dss(system) $dss(sky) sexagesimal]
		    set dss(x) [lindex $coord 0]
		    set dss(y) [lindex $coord 1]

		    if {$dss(updatesize)} {
			set size [$current(frame) get fits size \
				      $dss(system) arcmin]
			set dss(width) [lindex $size 0]
			set dss(height) [lindex $size 1]
			set dss(name) {}
		    }
		}
	    }
	}
    }
}

proc DSSAck {} {
set msg {Acknowledgements for the DSS 

The Digitized Sky Surveys were produced at the Space Telescope Science
Institute under U.S.  Government grant NAG W-2166. The images of these
surveys are based on photographic data obtained using the Oschin
Schmidt Telescope on Palomar Mountain and the UK Schmidt Telescope.
The plates were processed into the present compressed digital form
with the permission of these institutions.

The National Geographic Society - Palomar Observatory Sky Atlas
(POSS-I) was made by the California Institute of Technology with
grants from the National Geographic Society.

The Second Palomar Observatory Sky Survey (POSS-II) was made by the
California Institute of Technology with funds from the National
Science Foundation, the National Geographic Society, the Sloan
Foundation, the Samuel Oschin Foundation, and the Eastman Kodak
Corporation.

The Oschin Schmidt Telescope is operated by the California Institute
of Technology and Palomar Observatory.

The UK Schmidt Telescope was operated by the Royal Observatory
Edinburgh, with funding from the UK Science and Engineering Research
Council (later the UK Particle Physics and Astronomy Research
Council), until 1988 June, and thereafter by the Anglo-Australian
Observatory. The blue plates of the southern Sky Atlas and its
Equatorial Extension (together known as the SERC-J), as well as the
Equatorial Red (ER), and the Second Epoch [red] Survey (SES) were all
taken with the UK Schmidt.
}

    SimpleTextDialog dssack Acknowledgement 80 40 insert top $msg
}

proc DSSStatus {message} {
    global dss

    set dss(status) "Status: $message"
}

proc DSSProgress {token totalsize currentsize} {
    if {$totalsize != 0} {
	DSSStatus "$currentsize bytes of $totalsize bytes [expr int(double($currentsize)/$totalsize*100)]%"
    } else {
	DSSStatus "$currentsize bytes"
    }
}

proc DSSClearDialog {} {
    global dss

    set dss(name) ""
    set dss(x) ""
    set dss(y) ""
    DSSStatus {}
}

proc DSSServerDialog {} {
    global dss

    switch -- $dss(server) {
	sao {$dss(mb) entryconfig "Survey" -state disabled}
	stsci {$dss(mb) entryconfig "Survey" -state normal}
	eso {$dss(mb) entryconfig "Survey" -state normal}
    }
}

proc DSSCoordDialog {} {
    global dss

    switch -- $dss(sky) {
	fk4 -
	fk5 -
	icrs {
	    $dss(xname) configure -text "a" -font {symbol 12}
	    $dss(yname) configure -text "d" -font {symbol 12}
	}
	galactic {
	    $dss(xname) configure -text "l" -font {-family times -slant italic}
	    $dss(yname) configure -text "b" -font {-family times -slant italic}
	}
	ecliptic {
	    $dss(xname) configure -text "l" -font {symbol 12}
	    $dss(yname) configure -text "b" -font {symbol 12}
	}
    }
}

proc ProcessDSSCmd {varname iname flag fcName} {
    upvar $varname var
    upvar $iname i
    upvar $fcName fc

    global current
    global dss

    # do we have a rgb frame
    if {[$current(frame) get type] == "rgb"} {
	set base 0
    } else {
	set base 1
    }

    switch -- [string tolower [lindex $var $i]] {
	server {
	    incr i
	    set dss(server) [lindex $var $i]
	}
	survey {
	    incr i
	    set dss(survey) [lindex $var $i]
	}
	size {
	    incr i
	    set dss(width) [lindex $var $i]
	    incr i
	    set dss(height) [lindex $var $i]
	}
	width {
	    incr i
	    set dss(width) [lindex $var $i]
	}
	height {
	    incr i
	    set dss(height) [lindex $var $i]
	}
	coord {
	    if {$flag} {
		MultiLoad fc
	    }
	    incr i
	    set dss(x) [lindex $var $i]
	    incr i
	    set dss(y) [lindex $var $i]
	    DSSDoItCmd
	}
	x {
	    incr i
	    set dss(x) [lindex $var $i]
	}
	y {
	    incr i
	    set dss(y) [lindex $var $i]
	}
	name {
	    if {$flag && $base} {
		MultiLoad fc
	    }
	    incr i
	    set dss(name) [lindex $var $i]
	    DSSDoItCmd
	}
	default {
	    if {$flag && $base} {
		MultiLoad fc
	    }
	    set dss(name) [lindex $var $i]
	    DSSDoItCmd
	}
    }
}

proc DSSDoItCmd {} {
    global dss

    if {$dss(width) == ""} {
	set dss(width) 15
    }
    if {$dss(height) == ""} {
	set dss(height) 15
    }
    set dss(frame) current
    
    DSSDialog
    DSSApplyDialog
}
