// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <string.h>
#include <math.h>

#if __GNUC__ >= 3
#include <iostream>
#include <sstream>
using namespace std;
#else
#include <iostream.h>
#include <strstream.h>
#endif

#include "widget.h"

// Tk Canvas Widget Functions Declaration

int WidgetConfigProc(Tcl_Interp* interp, Tk_Canvas canvas, Tk_Item* item, 
			   int argc, Tcl_Obj *const argv[], int flags)
{
  return WIDGET(item).configure(argc, (const char**)argv, flags);
}

int WidgetCoordProc(Tcl_Interp* interp, Tk_Canvas canvas, Tk_Item* item, 
		      int argc, Tcl_Obj *const argv[])
{
  return WIDGET(item).coordProc(argc, (char**)argv);
}

void WidgetDeleteProc(Tk_Canvas canvas, Tk_Item* item, Display* display)
{
  delete ((WidgetOptions*)item)->widget;
}

void WidgetDisplayProc(Tk_Canvas canvas, Tk_Item* item, Display* display,
			     Drawable draw, int x, int y, int width, int height)
{
  WIDGET(item).displayProc(draw, x, y, width, height);
}

double WidgetPointProc(Tk_Canvas canvas, Tk_Item* item, double* point)
{
  return WIDGET(item).pointProc(point);
}

int WidgetAreaProc(Tk_Canvas canvas, Tk_Item* item, double* bbox)
{
  return WIDGET(item).areaProc(bbox);
}

int WidgetPostscriptProc(Tcl_Interp* interp, Tk_Canvas canvas,
			       Tk_Item* item, int prepass)
{
  return WIDGET(item).postscriptProc(prepass);
}

void WidgetScaleProc(Tk_Canvas canvas, Tk_Item* item, 
			   double Ox, double Oy , double Sx, double Sy)
{
  WIDGET(item).scaleProc(Ox, Oy, Sx, Sy);
}

void WidgetTranslateProc(Tk_Canvas canvas, Tk_Item* item, double x, double y)
{
  WIDGET(item).translateProc(x, y);
}

int WidgetIndexProc(Tcl_Interp* interp, Tk_Canvas canvas, Tk_Item* item,
		    char indexString, int* indexPtr)
{
  WIDGET(item).indexProc(indexString, indexPtr);
  return 1;
}

void WidgetICursorProc(Tk_Canvas canvas, Tk_Item* item, int index)
{
  WIDGET(item).icursorProc(index);
}

int WidgetSelectionProc(Tk_Canvas canvas, Tk_Item* item,
			int offset, char* buffer, int maxBytes)
{
  WIDGET(item).selectionProc(offset, buffer, maxBytes);
  return 1;
}

void WidgetInsertProc(Tk_Canvas canvas, Tk_Item* item, int index, char* string)
{
  WIDGET(item).insertProc(index, string);
}

void WidgetDCharsProc(Tk_Canvas canvas, Tk_Item* item, int first, int last)
{
  WIDGET(item).dcharsProc(first, last);
}

#if __GNUC__ >= 3
int WidgetParse(ClientData widget, Tcl_Interp* interp, int argc, 
		const char** argv)
{
  int result;
  Tcl_Preserve(widget);

  if (argc >= 2 && !strcmp(argv[1],"configure"))
    result = ((Widget*)widget)->configCmd(argc-2, argv+2);

  else {
    istringstream istr(ios::in|ios::out);
    ostream ostr(istr.rdbuf());

    for (int i=1; i<argc; i++)
      ostr << argv[i] << " ";
    ostr << ends;
    result = ((Widget*)widget)->parse(istr);
  }

  Tcl_Release(widget);
  return result;
}
#else
int WidgetParse(ClientData widget, Tcl_Interp* interp, int argc, 
		const char** argv)
{
  int result;
  Tcl_Preserve(widget);

  if (argc >= 2 && !strcmp(argv[1],"configure"))
    result = ((Widget*)widget)->configCmd(argc-2, argv+2);

  else {
    char buf[1024];
    ostrstream ostr(buf,1024);

    for (int i=1; i<argc; i++)
      ostr << argv[i] << " ";
    ostr << ends;

    istrstream istr(buf,1024);
    result = ((Widget*)widget)->parse(istr);
  }

  Tcl_Release(widget);
  return result;
}
#endif

// Member Functions

Widget::Widget(Tcl_Interp* interp_, Tk_Canvas canvas_, Tk_Item* item) :
  interp(interp_), canvas(canvas_)
{
  // initialize item ptr to this. This is the method the canvas widget procs
  // know how to call member functions.

  ((WidgetOptions*)item)->widget = this;

  // init members

  options = (WidgetOptions*)item;
  configSpecs = NULL;
  tkwin = Tk_CanvasTkwin(canvas);
  display = Tk_Display(tkwin);
  screenNumber = Tk_ScreenNumber(tkwin);
  visual = Tk_Visual(tkwin);
  depth = Tk_Depth(tkwin);

  pixmap = 0;
  visible = True;

  originX = 0;
  originY = 0;

  blackColor = Tk_GetColor(interp, tkwin, "black");
  whiteColor = Tk_GetColor(interp, tkwin, "white");
  redColor = Tk_GetColor(interp, tkwin, "red");
  greenColor = Tk_GetColor(interp, tkwin, "green");
  blueColor = Tk_GetColor(interp, tkwin, "blue");
  cyanColor = Tk_GetColor(interp, tkwin, "cyan");
  magentaColor = Tk_GetColor(interp, tkwin, "magenta");
  yellowColor = Tk_GetColor(interp, tkwin, "yellow");

  // this is needed because of a problem with Tk_ConfigureWidget
  options->cmdName = NULL;

  cmd = NULL;
  result = TCL_OK;

  // create GC for pixmap

  XGCValues values;
  gc = Tk_GetGC(tkwin, 0, &values);
}

Widget::~Widget()
{
  // free the options

  Tk_FreeOptions(configSpecs, (char*)this->options, display, 0);

  // clean up tcl command

  if (cmd) {
    Tcl_DeleteCommand(interp, cmd);
    delete [] cmd;
  }

  // free up colors

  if (blackColor)
    Tk_FreeColor(blackColor);

  if (whiteColor)
    Tk_FreeColor(whiteColor);

  if (redColor)
    Tk_FreeColor(redColor);

  if (greenColor)
    Tk_FreeColor(greenColor);

  if (blueColor)
    Tk_FreeColor(blueColor);

  if (cyanColor)
    Tk_FreeColor(cyanColor);

  if (magentaColor)
    Tk_FreeColor(magentaColor);

  if (yellowColor)
    Tk_FreeColor(yellowColor);

  // clean up pixmap resources

  if (pixmap)
    Tk_FreePixmap(display, pixmap);

  if (gc)
    Tk_FreeGC(display, gc);

}

void Widget::error(const char* m)
{
  Tcl_AppendResult(interp, m, NULL);
  result = TCL_ERROR;
}

void Widget::msg(const char* m)
{
  Tcl_AppendResult(interp, m, NULL);
}

int Widget::configure(int argc, const char** argv, int flags)
{
  if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv, 
			 (char*)this->options, flags) != TCL_OK)
    return TCL_ERROR;

  // test which resources have been changed and act accordingly

  if (flags != TK_CONFIG_ARGV_ONLY) {
    createCommand();
    updateBBox();
    invalidPixmap();
  }
  else {
    if (configSpecs[CONFIGCOMMAND].specFlags & TK_CONFIG_OPTION_SPECIFIED)
      createCommand();

    if ((configSpecs[CONFIGX].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||
	(configSpecs[CONFIGY].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||
	(configSpecs[CONFIGWIDTH].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||
	(configSpecs[CONFIGHEIGHT].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||
	(configSpecs[CONFIGANCHOR].specFlags & TK_CONFIG_OPTION_SPECIFIED))
      updateBBox();

    if ((configSpecs[CONFIGWIDTH].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||
	(configSpecs[CONFIGHEIGHT].specFlags & TK_CONFIG_OPTION_SPECIFIED))
      invalidPixmap();

    if ((configSpecs[CONFIGX].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||
	(configSpecs[CONFIGY].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||
	(configSpecs[CONFIGWIDTH].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||
	(configSpecs[CONFIGHEIGHT].specFlags & TK_CONFIG_OPTION_SPECIFIED) ||
	(configSpecs[CONFIGANCHOR].specFlags & TK_CONFIG_OPTION_SPECIFIED))
      redraw();
  }

  return TCL_OK;
}

// Required Canvas Functions

int Widget::coordProc(int argc, char** argv)
{
  char xStr[TCL_DOUBLE_SPACE], yStr[TCL_DOUBLE_SPACE];

  switch (argc) {

    // print the current values

  case 0:
    Tcl_PrintDouble(interp, options->x, xStr);
    Tcl_PrintDouble(interp, options->y, yStr);
    Tcl_AppendResult(interp, xStr, " ", yStr, NULL);
    return TCL_OK;

    // set current x&y to new values

  case 2: {
    double x0, y0;
    if ((Tk_CanvasGetCoord(interp, canvas, argv[0], &x0) != TCL_OK) ||
	(Tk_CanvasGetCoord(interp, canvas, argv[1], &y0) != TCL_OK))
	    return TCL_ERROR;
    options->x = (int)x0;
    options->y = (int)y0;
    updateBBox();
    return TCL_OK;
  }

 // else, error message

  default:
    Tcl_AppendResult(interp, "wrong # coordinates: expected 0 or 2", NULL);
    return TCL_ERROR;
  }
}

// Widget Display Procedure. It takes the contents of the pixmap and
// after clipping, copies it into the drawable.

void Widget::displayProc(Drawable draw, int clipX, int clipY, 
			int clipWidth, int clipHeight)
{
  if (visible == False)
    return;

  // create bbox

  BBox bb(clipX, clipY, clipX+clipWidth, clipY+clipHeight);

  // update the pixmap from ximage/graphics

  if (updatePixmap(bb) != TCL_OK)
    return; // something is wrong, bail out

  // define pixmap clip region
  // NOTE: it appears that the canvas coord system is 1 to n, width/height = n
  // with the original of value 1,1 located at upper left corner

  int pmX, pmY, pmWidth, pmHeight;

  if (clipX > options->item.x1) {
    pmX = clipX - options->item.x1;
    pmWidth = options->item.x2 - clipX;
  }
  else {
    pmX = 0;
    if ((clipX + clipWidth) < options->item.x2)
      pmWidth = clipX + clipWidth - options->item.x1;
    else
      pmWidth = options->item.x2 - options->item.x1;
  }

  if (clipY > options->item.y1) {
    pmY = clipY - options->item.y1;
    pmHeight = options->item.y2 - clipY;
  } 
  else {
    pmY = 0;
    if ((clipY + clipHeight) < options->item.y2)
      pmHeight = clipY + clipHeight - options->item.y1;
    else
      pmHeight = options->item.y2 - options->item.y1;
  }

  // convert to canvas coords

  short drawX, drawY;
  Tk_CanvasDrawableCoords(canvas, (double)(options->item.x1 + pmX),
			  (double)(options->item.y1 + pmY), &drawX, &drawY);

  // set the clip region and copy the pixmap into the drawable

  XSetClipOrigin(display, gc, drawX - pmX, drawY - pmY);
  XCopyArea(display, pixmap, draw, gc, pmX, pmY, (unsigned int) pmWidth,
	    (unsigned int) pmHeight, drawX, drawY);
}

double Widget::pointProc(double* point)
{
  double xdiff, ydiff;

  if (point[0] < options->item.x1)
    xdiff = options->item.x1 - point[0];
  else if (point[0] > options->item.x2)
    xdiff = point[0] - options->item.x2;
  else
    xdiff = 0;

  if (point[1] < options->item.y1)
    ydiff = options->item.y1 - point[1];
  else if (point[1] > options->item.y2)
    ydiff = point[1] - options->item.y2;
  else
    ydiff = 0;

  return hypot(xdiff, ydiff);
}

int Widget::areaProc(double* bbox)
{
  if ((bbox[2] <= options->item.x1) || (bbox[0] >= options->item.x2) ||
      (bbox[3] <= options->item.y1) || (bbox[1] >= options->item.y2))
    return -1; // item is outside bbox

  if ((bbox[0] <= options->item.x1) && (bbox[1] <= options->item.y1) &&
      (bbox[2] >= options->item.x2) && (bbox[3] >= options->item.y2))
    return 1; // item is inside bbox

  return 0;   // item overlaps bbox
}

int Widget::postscriptProc(int prepass)
{
  return TCL_OK;
}

void Widget::scaleProc(double Ox, double Oy, double Sx, double Sy)
{
  // translate to (Ox,Oy), scale by (Sx,Sy), translate back from (Ox,Oy)

  options->x = (int)(Ox + (Sx * (options->x - Ox)));
  options->y = (int)(Oy + (Sy * (options->y - Oy)));
  options->width = (int)(options->width * Sx);
  options->height = (int)(options->height * Sy);

  updateBBox();
  invalidPixmap();  // width and height have changed
}

void Widget::translateProc(double deltaX, double deltaY)
{
  options->x += (int)deltaX;
  options->y += (int)deltaY;

  updateBBox();
}

// Subcommand Functions

int Widget::configCmd(int argc, const char** argv)
{
  switch (argc) {
  case 0:
    return Tk_ConfigureInfo(interp, tkwin, configSpecs, (char*)this->options, NULL, 0);
  case 1:
    return Tk_ConfigureInfo(interp, tkwin, configSpecs, 
			    (char*)this->options, argv[0],0);
  default:
    return configure(argc, argv, TK_CONFIG_ARGV_ONLY);
  }
}

#if __GNUC__ >= 3
void Widget::getHeightCmd()
{
  ostringstream str;
  str << options->height << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}
#else
void Widget::getHeightCmd()
{
  char buf[8];
  ostrstream str(buf,8);
  str << options->height << ends;
  Tcl_AppendResult(interp, buf, NULL);
}
#endif

#if __GNUC__ >= 3
void Widget::getWidthCmd()
{
  ostringstream str;
  str << options->width << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}
#else
void Widget::getWidthCmd()
{
  char buf[8];
  ostrstream str(buf,8);
  str << options->width << ends;
  Tcl_AppendResult(interp, buf, NULL);
}
#endif

void Widget::hideCmd()
{
  visible = False;
  redraw();
}

void Widget::resetCmd()
{
  reset();
  invalidPixmap(); 
  redraw();
}

void Widget::showCmd()
{
  visible = True;
  redraw();
}

// Private Member Functions

void Widget::createCommand()
{
  if (cmd) {
    Tcl_DeleteCommand(interp, cmd);
    if (cmd)
      delete cmd;
  }

  cmd = new char[strlen(options->cmdName)+1];
  strcpy(cmd, options->cmdName);
  Tcl_CreateCommand(interp, cmd, WidgetParse, (ClientData)this, NULL);
}

#if __GNUC__ >= 3
int Widget::checkArgs(int should, int argc, char** argv)
{
  // if should is negative, don't check

  if (should >= 0) {
    if (should != argc) {
      ostringstream str;
      str << should << '\0';
      Tcl_AppendResult(interp, "wrong # args: requires ", str.str().c_str(),
		       " args.", NULL);
      return TCL_ERROR;
    }
    else
      return TCL_OK;
  }

  return TCL_OK;
}
#else
int Widget::checkArgs(int should, int argc, char** argv)
{
  // if should is negative, don't check

  if (should >= 0) {
    if (should != argc) {
      char buf[4];
      ostrstream str(buf,4);
      str << should << '\0';
      Tcl_AppendResult(interp, "wrong # args: requires ", buf, " args.", NULL);
      return TCL_ERROR;
    }
    else
      return TCL_OK;
  }

  return TCL_OK;
}
#endif

void Widget::invalidPixmap()
{
  if (pixmap)
    Tk_FreePixmap(display, pixmap);
  pixmap = 0;
}

void Widget::redraw()
{
  Tk_CanvasEventuallyRedraw(canvas, options->item.x1, options->item.y1, 
			    options->item.x2, options->item.y2);
}

void Widget::redraw(BBox bb)
{
  // bb in canvas coords
  // we need to expand by 1, i don't know why

  Tk_CanvasEventuallyRedraw(canvas, (int)bb.ll[0], (int)bb.ll[1], 
			    (int)bb.ur[0]+1, (int)bb.ur[1]+1);
}

void Widget::redrawNow()
{
  Tk_CanvasEventuallyRedraw(canvas, options->item.x1, options->item.y1, 
			    options->item.x2, options->item.y2);
  Tcl_DoOneEvent(TCL_IDLE_EVENTS);
}

void Widget::redrawNow(BBox bb)
{
  // bb in canvas coords
  // we need to expand by 1, i don't know why

  Tk_CanvasEventuallyRedraw(canvas, (int)bb.ll[0], (int)bb.ll[1], 
			    (int)bb.ur[0]+1, (int)bb.ur[1]+1);
  Tcl_DoOneEvent(TCL_IDLE_EVENTS);
}

void Widget::forceUpdate()
{
  Tcl_DoOneEvent(TCL_IDLE_EVENTS);
}

// Update Bounding Box. For image at (n,n) with size (m,m), 
// bbox is defined from (n,n) to (n+m,n+m)

void Widget::updateBBox()
{
  originX = options->x;
  originY = options->y;

  // Modify position point using anchor information.

  switch (options->anchor) {
  case TK_ANCHOR_N:
    originX -= options->width/2;
    break;
  case TK_ANCHOR_NE:
    originX -= options->width;
    break;
  case TK_ANCHOR_E:
    originX -= options->width;
    originY -= options->height/2;
    break;
  case TK_ANCHOR_SE:
    originX -= options->width;
    originY -= options->height;
    break;
  case TK_ANCHOR_S:
    originX -= options->width/2;
    originY -= options->height;
    break;
  case TK_ANCHOR_SW:
    originY -= options->height;
    break;
  case TK_ANCHOR_W:
    originY -= options->height/2;
    break;
  case TK_ANCHOR_NW:
    break;
  case TK_ANCHOR_CENTER:
    originX -= options->width/2;
    originY -= options->height/2;
    break;
  }

  // Update item item

  options->item.x1 = originX;
  options->item.y1 = originY;
  options->item.x2 = originX + options->width;
  options->item.y2 = originY + options->height;
}

int Widget::getColor(const char* str)
{
  if (!strcmp(str,"black"))
    return blackColor->pixel;
  else if (!strcmp(str,"white"))
    return whiteColor->pixel;
  else if (!strcmp(str,"red"))
    return redColor->pixel;
  else if (!strcmp(str,"green"))
    return greenColor->pixel;
  else if (!strcmp(str,"blue"))
    return blueColor->pixel;
  else if (!strcmp(str,"cyan"))
    return cyanColor->pixel;
  else if (!strcmp(str,"magenta"))
    return magentaColor->pixel;
  else if (!strcmp(str,"yellow"))
    return yellowColor->pixel;
  else {
    cerr << "Widget Internal Error: Unable to Parser color: "<< str << endl;
    return blackColor->pixel;
  }
}

XColor* Widget::getXColor(const char* str)
{
  if (!strcmp(str,"black"))
    return blackColor;
  else if (!strcmp(str,"white"))
    return whiteColor;
  else if (!strcmp(str,"red"))
    return redColor;
  else if (!strcmp(str,"green"))
    return greenColor;
  else if (!strcmp(str,"blue"))
    return blueColor;
  else if (!strcmp(str,"cyan"))
    return cyanColor;
  else if (!strcmp(str,"magenta"))
    return magentaColor;
  else if (!strcmp(str,"yellow"))
    return yellowColor;
  else {
    cerr << "Widget Internal Error: Unable to Parser color: "<< str << endl;
    return blackColor;
  }
}
