#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc HVLoadURL {varname url query} {
    upvar #0 $varname var
    global $varname

    global debug
    global message

    # this assumes the url has been already resolved
    if {$debug(tcl,hv)} {
	puts "HVLoadURL $varname $url $query"
    }

    # do we have anything?
    if {$url == ""} {
	return
    }

    # set the cursor to busy
    $var(widget) configure -cursor watch

    # parse url
    ParseURL $url r
    if {$debug(tcl,hv)} {
	puts "HVLoadURL parse url ;$r(scheme);$r(authority);$r(path);$r(query);$r(fragment);$query"
    }

    # check if have we been here before
    set prev ";$r(scheme);$r(authority);$r(path);$r(query);$query"
    if {$var(previous) == $prev} {
	if {$debug(tcl,hv)} {
	    puts "HVLoadURL found prev $prev"
	}
	HVSet $varname $url $query $r(fragment)
	HVGotoHTML $varname
	return
    }
    set var(previous) $prev

    switch -- $r(scheme) {
	file -
	{} {HVProcessURLFile $varname $url $query r}
	ftp {HVProcessURLFTP $varname $url $query r}
	http {HVProcessURLHTTP $varname $url $query r}
	default {
	    $var(widget) configure -cursor {}
	    Error "$message(error,hvsup,scheme) $r(scheme)"
	}
    }
}

proc HVProcessURLFile {varname url query rr} {
    upvar #0 $varname var
    global $varname

    upvar $rr r
    
    global debug
    global ds9

    if {$debug(tcl,hv)} {
	puts "HVProcessURLFile"
    }

    if [file exists $r(path)] {
	if [file isdirectory $r(path)] {
	    HVSetAll $varname $url {} {} {} \
		[HVFileHtmlList $r(path) [HVDirList $r(path)]] {} \
		"text/html" {} 200 {}

	    set var(active) 1
	    set var(delete) 0
	    HVParse $varname
	} else {
	    HVSet $varname $url {} $r(fragment)
	    set var(active) 1
	    set var(delete) 0
	    HVLoadFile $varname $r(path)
	}
    }
}

proc HVProcessURLFTP {varname url query rr} {
    upvar #0 $varname var
    global $varname

    upvar $rr r
    
    global debug
    global ds9

    if {$debug(tcl,hv)} {
	puts "HVProcessURLFTP"
    }

    set fn "$ds9(tmpdir)/[file tail $r(path)]"
    set ftp [ftp::Open $r(authority) "ftp" "ds9@" -mode passive]
    if {$ftp > -1} {
	# first try to get as file
	set ftp::VERBOSE $debug(tcl,ftp)
	set "ftp::ftp${ftp}(Output)" FTPLog
	ftp::Type $ftp binary
	if [ftp::Get $ftp $r(path) "$fn"] {
	    ftp::Close $ftp

	    HVSet $varname $url {} $r(fragment)

	    set var(active) 1
	    set var(delete) 1
	    HVLoadFile $varname "$fn"
	} else {
	    # from the prev attempt
	    catch {file delete -force "$fn"}

	    # now as a directory
	    set list [ftp::List $ftp $r(path)]
	    ftp::Close $ftp

	    HVSetAll $varname $url {} {} {} \
		[HVFTPHtmlList $r(authority) $r(path) $list] {} \
		"text/html" {} 200 {}

	    set var(active) 1
	    set var(delete) 0
	    HVParse $varname
	}
    }
}

proc HVProcessURLHTTP {varname url query rr} {
    upvar #0 $varname var
    global $varname

    upvar $rr r
    
    global debug
    global ds9
    global message

    if {$debug(tcl,hv)} {
	puts "HVProcessURLHTTP"
    }

    HVSet $varname $url $query $r(fragment)

    # stop any refresh
    if [info exists ${varname}(refresh,id)] {
	if {$var(refresh,id)>0} {
	    after cancel $var(refresh,id)
	}
    }
    set var(html) {}
    set var(meta) {}
    set var(refresh,time) 0
    set var(refresh,url) {}
    set var(refresh,id) 0
    set var(fn) {}
    set var(token) {}
    set var(ch) {}

    # do we have html? if so, use a var
    ParseURL $url r
    switch -- [file extension $r(path)] {
	".html" -
	".htm" {
	    # geturl in var
	    if {![catch {set var(token) [http::geturl $url \
		    -progress [list HVProgress $varname] \
		    -binary 1 \
		    -headers "[ProxyHTTP]" \
		    -query "$query" \
		    -command [list HVProcessURLHTTPVarFinish $varname]]}]} {
		set var(active) 1
		set var(delete) 0

		$var(mb).view entryconfig "Stop" -state normal
	    } else {
		Error "$message(error,hvsup,url) $url"
	    }
	}
	default {
	    # geturl as file
	    set var(fn) [tmpnam ds9 .http]
	    if [catch {open "$var(fn)" w} ${varname}(ch)] {
		Error "Unable to open tmp file $var(fn) for writing"
		return
	    }
	    if {![catch {set var(token) [http::geturl $url \
		    -progress [list HVProgress $varname] \
		    -binary 1 \
		    -headers "[ProxyHTTP]" \
		    -query "$query" \
		    -channel $var(ch) \
		    -command [list HVProcessURLHTTPFileFinish $varname]]}]} {
		set var(active) 1
		set var(delete) 1

		$var(mb).view entryconfig "Stop" -state normal
	    } else {
		catch {close $var(ch)}
		Error "$message(error,hvsup,url) $url"
		return
	    }
	}
    }
}

proc HVProcessURLHTTPVarFinish {varname token} {
    upvar #0 $varname var
    global $varname

    global debug
    global message

    if {$debug(tcl,hv)} {
	puts "HVLoadURLVarFinish"
    }

    upvar #0 $token t

    # Code
    set var(code) [http::ncode $token]
    if {$debug(tcl,hv)} {
	puts "HVLoadURLVarFinish code: $var(code)"
    }

    # Meta
    set var(meta) $t(meta)
    if {$debug(tcl,hv)} {
	puts "HVLoadURLVarFinish meta: $var(meta)"
    }

    # Mime-type
    # we want to strip and extra info after ';'
    regexp -nocase {([^;])*} $t(type) ${varname}(mime)
    if {$debug(tcl,hv)} {
	puts "HVLoadURLVarFinish mime: $var(mime)"
    }

    # Content-Encoding / Refresh
    foreach {name value} $var(meta) {
	if {[regexp -nocase ^content-encoding $name]} {
	    switch -- $value {
		gzip -
		x-gzip {set var(encoding) gzip}
		compress -
		Z {set var(encoding) compress}
		pack -
		z {set var(encoding) pack}
		default {}
	    }
	}

	if {[regexp -nocase ^Refresh $name]} {
	    set f [split $value \;]
	    set var(refresh,time) [lindex $f 0]
	    set var(refresh,url) [string [lindex $f 1] 4 end]
	    if {$var(refresh,url) != {} & $var(refresh,time) != {}} {
		if {$debug(tcl,hv)} {
		    puts "HVLoadURLVarFinish Refresh $var(refresh,time) $var(refresh,url)"
		}
		set var(previous) {}
		set var(refresh,id) [after [expr $var(refresh,time)*1000] "HVLoadURL $varname \{$var(refresh,url)\} {}"]

	    } else {
		set var(refresh,id) 0
	    }
	}
    }
	   
    # html
    set var(html) [http::data $token]

    HTTPLog $token
    http::cleanup $token
    
    $var(mb).view entryconfig "Stop" -state disabled

    # do this last, since we may reenter
    HVParse $varname
}

proc HVProcessURLHTTPFileFinish {varname token} {
    upvar #0 $varname var
    global $varname

    global debug
    global message

    if {$debug(tcl,hv)} {
	puts "HVLoadURLFileFinish"
    }

    upvar #0 $token t

    catch {close $var(ch)}

    # Code
    set var(code) [http::ncode $token]
    if {$debug(tcl,hv)} {
	puts "HVLoadURLFileFinish code: $var(code)"
    }

    # Meta
    set var(meta) $t(meta)
    if {$debug(tcl,hv)} {
	puts "HVLoadURLFileFinish meta: $var(meta)"
    }

    # Mime-type
    # we want to strip and extra info after ';'
    regexp -nocase {([^;])*} $t(type) ${varname}(mime)
    if {$debug(tcl,hv)} {
	puts "HVLoadURLFileFinish mime: $var(mime)"
    }

    # Content-Encoding
    foreach {name value} $var(meta) {
	if {[regexp -nocase ^content-encoding $name]} {
	    switch -- $value {
		gzip -
		x-gzip {set var(encoding) gzip}
		compress -
		Z {set var(encoding) compress}
		pack -
		z {set var(encoding) pack}
		default {}
	    }
	}

	if {[regexp -nocase ^Refresh $name]} {
	    set f [split $value \;]
	    set var(refresh,time) [lindex $f 0]
	    set var(refresh,url) [string range [lindex $f 1] 4 end]
	    if {$var(refresh,url) != {} & $var(refresh,time) != {}} {
		if {$debug(tcl,hv)} {
		    puts "HVLoadURLVarFinish Refresh $var(refresh,time) $var(refresh,url)"
		}
		set var(previous) {}
		set var(refresh,id) [after [expr $var(refresh,time)*1000] "HVLoadURL $varname \{$var(refresh,url)\} {}"]

	    } else {
		set var(refresh,id) 0
	    }
	}
    }

    HTTPLog $token
    http::cleanup $token
    
    $var(mb).view entryconfig "Stop" -state disabled

    # do this last, since we may reenter
    HVParse $varname
}

proc HVCancel {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVCancel"
    }

    # stop any refresh
    if {$var(refresh,id)>0} {
	after cancel $var(refresh,id)
	set var(refresh,id) 0
    }

    if {$var(active)} {
	# clean up
	HVClearTmpFile $varname
	set var(active) 0

	# reset willl call FinishURL and we can't feed extra params
	http::reset var(token)

	$var(widget) configure -cursor {}
    }
}

proc HVLoadFile {varname path} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVLoadFile $path"
    }

    set var(html) {}
    set var(fn) $path
    set var(code) 200
    set var(meta) {}
    set var(refresh,time) 0
    set var(refresh,url) {}
    set var(refresh,id) 0

    # set content-encoding
    switch -- [string tolower [file extension $path]] {
	.gz {
	    set path [file rootname $path]
	    set var(encoding) gzip
	}
	.Z {
	    set path [file rootname $path]
	    set var(encoding) compress
	}
	.z {
	    set path [file rootname $path]
	    set var(encoding) pack
	}
	default {set var(encoding) {}}
    }

    # set mime-type
    switch -- [string tolower [file extension $path]] {
	.html -
	.htm {set var(mime) "text/html"}
	.gif {set var(mime) "image/gif"}
	.jpeg -
	.jpg {set var(mime) "image/jpeg"}
	.tiff -
	.tif {set var(mime) "image/tiff"}
	.png {set var(mime) "image/png"}
	.bmp {set var(mime) "image/bmp"}
	.ppm {set var(mime) "image/ppm"}
	.xbm {set var(mime) "image/xbm"}

	.fits -
	.fit -
	.fts {set var(mime) "image/fits"}

	.ftz -
	.fits.gz -
	.fgz {
	    set var(mime) "image/fits"
	    set var(encoding) "gzip"
	}

	.text -
	.txt {set var(mime) "text/plain"}
	default {
	    switch -- $var(encoding) {
		gzip -
		compress -
		pack {set var(mime) "application/octet-stream"}
		default {set var(mime) "text/plain"}
	    }
	}
    }

    HVParse $varname
}

proc HVParse {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVParse mime-type:$var(mime) content-encoding:$var(encoding)"
    }

    if {!$var(active)} {
	if {$debug(tcl,hv)} {
	    puts "HVParse not active- abort"
	}

	$var(widget) configure -cursor {}
	return
    }

    set result 1
    switch -- "$var(mime)" {
	"multipart/x-mixed-replace" -
	"text/html" {set result [HVParseHTML $varname]}

	"text/plain" {HVParseText $varname}

	"image/gif" -
	"image/jpeg" -
	"image/tiff" -
	"image/png" -
	"image/bmp" -
	"image/ppm" -
	"image/x-portable-pixmap" -
	"image/xbm" -
	"image/x-xbitmap" {HVParseImg $varname}

	"image/fits" -
	"application/fits" {HVParseFITS $varname}

	"application/fits-image" -
	"application/fits-table" -
	"application/fits-group" {HVParseFITS $varname}

	"image/x-fits" -
	"binary/x-fits" -
	"application/x-fits" {HVParseFITS $varname}

	"image/x-gfits" -
	"binary/x-gfits" -
	"image/gz-fits" -
	"application/x-gzip" -
	"display/gz-fits" {
	    set var(encoding) gzip
	    HVParseFITS $varname
	}

	"image/x-cfits" -
	"binary/x-cfits" {
	    set var(encoding) compress
	    HVParseFITS $varname
	}

	"image/fits-hcompress" -
	"image/x-fits-h" {HVParseSave $varname}

	"application/octet-stream" {
	    # one last chance to grap it as a fits file

	    ParseURL $var(url) r
	    set path [file tail $r(path)]  

	    # set content-encoding
	    switch -- [file extension $path] {
		.gz {
		    set path [file rootname $path]
		    set var(encoding) gzip
		}
		.Z {
		    set path [file rootname $path]
		    set var(encoding) compress
		}
		.z {
		    set path [file rootname $path]
		    set var(encoding) pack
		}
		default {set var(encoding) {}}
	    }

	    # set mime-type
	    switch -- [file extension $path] {
		.fits -
		.fit -
		.fts {
		    set var(mime) "image/fits"
		    HVParseFITS $varname
		}

		.ftz -
		.fgz {
		    set var(mime) "image/fits"
		    set var(encoding) "gzip"
		    HVParseFITS $varname
		}
		default {HVParseSave $varname}
	    }
	}

	default {HVParseSave $varname}
    }

    # something wrong?
    if {!$result} {
	return
    }

    HVClearStatus $varname
    set var(active) 0
    HVClearTmpFile $varname
}

proc HVParseHTML {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVParseHTML"
    }

    if {[string length $var(html)] == 0} {
	if [file exists "$var(fn)"] {
	    if [catch {open "$var(fn)" r} ch] {
		Error "Unable to open file $var(fn) for reading"
		return
	    }
	    set var(html) [read $ch]
	    close $ch
	}
    }

    # figure out the base
    # we don't want any query or fragments
    ParseURL $var(url) r

    set base {}
    # scheme
    if {[string length $r(scheme)] != 0} {
	append base "$r(scheme)://"
    }
    # authority
    if {[string length $r(authority)] != 0} {
	append base "$r(authority)"
    }
    # path
    if {[string length $r(path)] != 0} {
	append base "$r(path)"
    } else {
	append base "/"
    }
    # query
    if {[string length $r(query)] != 0} {
	append base "?$r(query)"
    }
    $var(widget) config -base $base

    if {$debug(tcl,hv)} {
	DumpURL r
	puts "HVParseHTML base [$var(widget) cget -base]"
    }

    # is it a redirection?
    switch -- $var(code) {
	301 -
	302 -
	303 -
	304 {
	    # look for Location in meta data
	    foreach {name value} $var(meta) {
		if {[regexp -nocase ^location$ $name]} {
		    if {$debug(tcl,hv)} {
		      puts "HVParse redirect $var(code) from meta to $value"
		    }
		    HVClearTmpFile $varname
		    HVLoadURL $varname [$var(widget) resolve $value] {}
		    return 0
		}
	    }
	    # check html page
	    if {[regexp -nocase {.*<a href=\"([^\"]+)} $var(html) x url]} {
		if {$debug(tcl,hv)} {
		    puts "HVParse redirect $var(code) from html to $url"
		}
		HVClearTmpFile $varname
		HVLoadURL $varname [$var(widget) resolve $url] {}
		return 0
	    }
	}
    }

    # we have a valid html
    HVClearWidget $varname

    # fix forms with no action
    HVFixHTMLForm $varname

    # and now, parse it
    $var(widget) parse $var(html)

    HVGotoHTML $varname

    # success
    return 1
}

proc HVParseImg {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVParseImg"
    }

    # fake html 
    set html \
	"<html>\n<body>\n<img src=\"$var(fn)\" border=0>\n</body>\n</html>"
    HVSetAll $varname "$var(fn)" {} {} {} $html "$var(fn)" "text/html" {} 200 {}

    HVClearWidget $varname

    $var(widget) config -base {}
    $var(widget) parse $var(html)
    HVGotoHTML $varname
}

proc HVParseFITS {varname} {
    upvar #0 $varname var
    global $varname

    global debug
    global ds9

    if {$debug(tcl,hv)} {
	puts "HVParseFITS mime-type:$var(mime) content-encoding:$var(encoding)"
    }

    if {$var(delete) && $var(save)} {
	switch -- $var(encoding) {
	    gzip {FileLast hvfitsfbox "ds9.fits.gz"}
	    compress {FileLast hvfitsfbox "ds9.fits.Z"}
	    pack {FileLast hvfitsfbox "ds9.fits.z"}
	    default {FileLast hvfitsfbox "ds9.fits"}
	}

	set fn [SaveFileDialog hvfitsfbox]
	if {[string length "$fn"] != 0} {
	    if {![catch {file rename -force "$var(fn)" "$fn"}]} {
		set var(fn) "$fn"
		set var(delete) 0
	    }
	}
    }

    switch -- $var(frame) {
	new {
	    set ds9(display,user) tile
	    DisplayMode
	    CreateFrame
	}
	current {}
    }

    StartLoad
    global loadParam
    set loadParam(load,type) allocgz
    set loadParam(file,type) fits
    set loadParam(file,mode) "$var(file,mode)"
    set loadParam(file,name) "$var(fn)"

    # may have to convert the file, based on content-encoding
    switch -- "$var(encoding)" {
	compress {
	    catch {set ch [open "| uncompress < $var(fn) " r]}
	    set loadParam(load,type) channel
	    set loadParam(channel,name) $ch
	}
	pack {
	    catch {set ch [open "| pcat $var(fn) " r]}
	    set loadParam(load,type) channel
	    set loadParam(channel,name) $ch
	}
    }

    ProcessLoad
    FinishLoad

    HVClearTmpFile $varname
    HVClearAll $varname
    HVUpdateDialog $varname
}

proc HVParseText {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVParseText"
    }

    if {[string length $var(html)] == 0} {
	if [file exists "$var(fn)"] {
	    catch {
		if [catch {open "$var(fn)" r} ch] {
		    Error "Unable to open file $var(fn) for reading"
		    return
		}
		set var(html) [read $ch]
		close $ch
	    }
	}
    }

    set var(html) \
	"<html>\n<body>\n<pre>\n$var(html)\n</pre>\n</body>\n</html>"
    set var(mime) "text/html"
    set var(encoding) {}
    set var(code) 200
    set var(meta) {}
    set var(refresh,time) 0
    set var(refresh,url) {}
    set var(refresh,id) 0
    HVParseHTML $varname
}

proc HVParseSave {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVParseSave"
    }

    ParseURL $var(url) r
    FileLast hvfitsfbox [file tail $r(path)]
    set fn [SaveFileDialog hvfitsfbox]
    if {[string length "$fn"] != 0} {
	if {![catch {file rename -force "$var(fn)" "$fn"}]} {
	    set var(delete) 0
	}
    }

    HVClearAll $varname
    HVUpdateDialog $varname
}

proc HVGotoHTML {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    incr ${varname}(index)
    set var(index,$var(index)) "$var(url) $var(query)"

    if {$debug(tcl,hv)} {
	puts "HVGotoHTML $var(index) $var(fragment)"
    }

    if {[string length $var(fragment)] != 0} {
	if {$debug(tcl,idletasks)} {
	    puts "HVGotoHTML"
	}
	update idletasks

	$var(widget) yview $var(fragment)
    } else {
	$var(widget) yview moveto 0
    }

    HVUpdateDialog $varname
}

proc HVClearWidget {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVClearWidget"
    }

    $var(widget) clear
    foreach x [array names $varname] {
	set f [split $x ,]
	if {[lindex $f 0] == "images"} {
	    image delete $var($x)
	    unset ${varname}($x)
	}
    }
}

proc HVClearIndex {varname n} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVClearIndex $varname $n"
    }

    foreach x [array names $varname] {
	set f [split $x ,]
	if {[lindex $f 0] == "index" && [lindex $f 1] > $n} {
	    unset ${varname}($x)
	}
    }
    set var(index) $n
}

proc HVClearTmpFile {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVClearTmpFile"
    }

    if {$var(delete) && [string length "$var(fn)"] != 0} {
	if [file exists "$var(fn)"] {
	    if {$debug(tcl,hv)} {
		puts "HVClearTmpFile delete $var(fn)"
	    }
	    file delete "$var(fn)"
	}
	set var(fn) {}
	set var(delete) 0
    }
}

proc HVClearStatus {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    $var(top).s.status configure -text {}
}

proc HVUpdateDialog {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    # in case we've set the cursor
    $var(widget) configure -cursor {}

    if {$debug(tcl,hv)} {
	puts "HVUpdateDialog\n"
    }

    set id $var(index)
    set id [incr id -1]
    if {[info exists ${varname}(index,$id)]} {
	$var(mb).view entryconfig "Back" -state normal    
    } else {
	$var(mb).view entryconfig "Back" -state disabled
    }

    set id $var(index)
    set id [incr id 1]
    if {[info exists ${varname}(index,$id)]} {
	$var(mb).view entryconfig "Forward" -state normal    
    } else {
	$var(mb).view entryconfig "Forward" -state disabled
    }
}

proc HVRefresh {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVRefresh"
    }

    set var(active) 1
    set var(delete) 0
    HVParse $varname
}

proc HVStatus {varname str} {
    upvar #0 $varname var
    global $varname

    global debug

    if {[string length $str] > 0} {
	$var(top).s.status configure -text $str
    } else {
	$var(top).s.status configure -text {}
    }
}

proc HVProgress {varname token totalsize currentsize} {
    upvar #0 $varname var
    global $varname

    global debug
    
    # HVProgress can be called, even after the window has been destroyed
    if {![info exist ${varname}(top)]} {
	return
    }

    if {!$var(active)} {
	return
    }

    if {$totalsize != 0} {
	HVStatus $varname "$currentsize bytes of $totalsize bytes [expr int(double($currentsize)/$totalsize*100)]%"
    } else {
	HVStatus $varname "$currentsize bytes"
    }
}

proc HVFTPHtmlList {host path list} {
    global debug

    if {$debug(tcl,hv)} {
	puts "HVFTPHtmlList $host $path"
    }
    if {[string range $path end end] != "/"} {
	append path {/}
    }

    set html {}
    append html "<html>\n"
    append html "<head>\n"
    append html "<title>Index of $path on $host</title>\n"
    append html "</head>\n"
    append html "<body>\n"
    append html "<h1>Index on $path on $host</h1>\n"
    append html "<hr>\n"
    append html "<pre>\n"
    foreach l $list {
	switch -- [llength $l] {
	    8 {set offset 4}
	    9 {set offset 5}
	    10 {set offset 4}
	    11 {set offset 5}
	    default {set offset 5}
	}

	set ii [lindex $l [expr $offset+3]]
	switch -- [string range $l 0 0] {
	    d {
		set new "<a href=\"ftp://$host$path$ii/\">$ii</A>"
	    }
	    l {
		set new "<a href=\"ftp://$host$path$ii\">$ii</A>"
	    }
	    default {
		set new "<a href=\"ftp://$host$path$ii\">$ii</A>"
	    }
	}

	regsub $ii $l $new l
	append html "$l\n"
    }
    append html "</pre>\n"
    append html "</hr>\n"
    append html "</body>\n"

    return $html
}

proc HVFileHtmlList {path list} {
    global debug

    if {$debug(tcl,hv)} {
	puts "HVFileHtmlList $path"
    }

    if {[string range $path end end] != "/"} {
	append path {/}
    }

    set html {}
    append html "<html>\n"
    append html "<head>\n"
    append html "<title>Index of $path</title>\n"
    append html "</head>\n"
    append html "<body>\n"
    append html "<h1>Index on $path</h1>\n"
    append html "<hr>\n"
    append html "<pre>\n"
    foreach l $list {
	switch -- [llength $l] {
	    8 {set offset 4}
	    9 {set offset 5}
	    10 {set offset 4}
	    11 {set offset 5}
	    default {set offset 5}
	}

	set ii [lindex $l [expr $offset+3]]
	switch -- [string range $l 0 0] {
	    d {
		set new "<a href=\"file:$path$ii/\">$ii</A>"
	    }
	    l {
		set new "<a href=\"file:$path$ii\">$ii</A>"
	    }
	    default {
		set new "<a href=\"file:$path$ii\">$ii</A>"
	    }
	}

	regsub $ii $l $new l
	append html "$l\n"
    }
    append html "</pre>\n"
    append html "</hr>\n"
    append html "</body>\n"

    return $html
}

proc HVDirList {path} {
    global debug

    if {$debug(tcl,hv)} {
	puts "HVDirList $path"
    }
    return [split [exec ls -l $path] \n]
}

proc HVSet {varname url query fragment} {
    upvar #0 $varname var
    global $varname

    global debug

    set var(url) $url
    set var(query) $query
    set var(fragment) $fragment
}

proc HVSetAll {varname url query frag prev html fn mime encoding code meta} {
    upvar #0 $varname var
    global $varname

    global debug

    set var(url) $url
    set var(query) $query
    set var(fragment) $frag
    set var(previous) $prev
    set var(html) $html
    set var(fn) "$fn"
    set var(mime) $mime
    set var(encoding) $encoding
    set var(code) $code
    set var(meta) $meta
    set var(refresh,time) 0
    set var(refresh,url) {}
    set var(refresh,id) 0
}

proc HVClearAll {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    set var(url) {}
    set var(query) {}
    set var(fragment) {}
    set var(previous) {}
    set var(html) {}
    set var(fn) {}
    set var(mime) {}
    set var(encoding) {}
    set var(code) {}
    set var(meta) {}
    set var(refresh,time) 0
    set var(refresh,url) {}
    set var(refresh,id) 0
}

# Bindings

proc HVMotionBind {varname x y} {
    upvar #0 $varname var
    global $varname

    set url [$var(widget) href $x $y] 

    if {!$var(active)} {
	if {[string length $url] > 0} {
	    $var(widget) configure -cursor hand2
	} else {
	    $var(widget) configure -cursor {}
	}
    }

    HVStatus $varname $url
}

proc HVLinkBind {varname x y} {
    upvar #0 $varname var
    global $varname

    global debug
    global xpa

    if {$debug(tcl,hv)} {
	puts "HVLinkBind"
    }

    HVClearIndex $varname $var(index)
    HVClearStatus $varname
    set url [$var(widget) href $x $y]
    if {[string length $url] != 0} {
	# sub xpa method
	set exp {%40%40XPA_METHOD%40%40|@@XPA_METHOD@@}
	if {[regexp $exp $url]} {
	    regsub -all $exp $url "[xparec $xpa method]" url
	    if {$debug(tcl,hv)} {
		puts "HVLinkBind XPA_METHOD $url"
	    }
	}

	# already resolved
	HVLoadURL $varname $url {}
    }
}

proc HVLinkNewBind {varname x y} {
    upvar #0 $varname var
    global $varname

    global hv
    global debug
    global xpa

    if {$debug(tcl,hv)} {
	puts "HVLinkNewBind"
    }

    set url [$var(widget) href $x $y]
    if {[string length $url] != 0} {
	# sub xpa method
	set exp {%40%40XPA_METHOD%40%40|@@XPA_METHOD@@}
	if {[regexp $exp $url]} {
	    regsub -all $exp $url "[xparec $xpa method]" url
	}

	incr hv(incr)
	HV "${varname}$hv(incr)" "$var(title)" $url 2 0 {}
    }
}

# CallBacks

proc HVImageCB {varname args} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVImageCB args: $varname $args"
    }

    set url [lindex $args 0]
    if {$debug(tcl,hv)} {
	puts "HVImageCB $url"
    }

    # do we have anything?
    if {[string length $url] == 0} {
	return
    }

    # do we have a width/height?
    set aa [lindex $args 3]
    set width [HVattrs width $aa 0]
    set height [HVattrs height $aa 0]

    # check for percent (100%) in width/height
    if {![string is integer $width]} {
	set width 0
    }
    if {![string is integer $height]} {
	set height 0
    }

    if {$debug(tcl,hv)} {
	puts "HVImageCB width $width height $height"
    }

    set img [HVImageURL $varname $url $width $height]

    if {[string length $img] != 0} {
	return $img
    } else {
	if {$debug(tcl,hv)} {
	    puts "HVImageCB FAIL $url"
	}
	return $var(img,gray)
    }
}

proc HVImageURL {varname url width height} {
    upvar #0 $varname var
    global $varname

    global hv
    global debug
    global http

    if {$debug(tcl,hv)} {
	puts "HVImageURL $varname $url $width $height"
    }

    # do we already have the image?
    if {[info exists ${varname}(images,$url)]} {
	if {$debug(tcl,hv)} {
	    puts "HVImageURL found image a $url"
	}
	return $var(images,$url)
    }

    ParseURL $url r

    set fn {}
    switch -- $r(scheme) {
	{} -
	file {
	    if [file exists $r(path)] {
		catch {image create photo -file $r(path)} img
	    }
	}
	ftp {
	    set fn [tmpnam ds9 [file extension $r(path)]]
	    set ftp [ftp::Open $r(authority) "ftp" "ds9@" -mode passive]
	    if {$ftp > -1} {
		set ftp::VERBOSE $debug(tcl,ftp)
		set "ftp::ftp${ftp}(Output)" FTPLog
		ftp::Type $ftp binary
		if [ftp::Get $ftp $r(path) "$fn"] {
		    ftp::Close $ftp

		    if {[file size "$fn"] == 0} {
			catch {file delete -force "$fn"}
			return {}
		    }
		    if {[catch {image create photo -file "$fn"} img]} {
			catch {file delete -force "$fn"}
			return {}
		    }
		}
	    }
	}
	http {
	    set fn [tmpnam ds9 [file extension $r(path)]]
	    if [catch {open "$fn" w} ch] {
		Error "Unable to open tmp file $fn for writing"
		return {}
	    }
	    catch {
		set token [http::geturl $url \
			       -progress [list HVProgress $varname] \
			       -channel $ch \
			       -timeout $hv(timeout) \
			       -binary 1 \
			       -headers "[ProxyHTTP]"]
		close $ch
		http::cleanup $token
	    }

	    if {[file size "$fn"] == 0} {
		catch {file delete -force "$fn"}
		return {}
	    }
	    if {[catch {image create photo -file "$fn"} img]} {
		catch {file delete -force "$fn"}
		return {}
	    }
	}
    }

    # do we have an img?
    if {![info exists img]} {
	return {}
    }
    if {$debug(tcl,hv)} {
	puts "HVImageURL got image $img"
    }

    # adjust image size if needed
    if {$width != 0 || $height != 0} {
	set iw [image width $img]
	set ih [image height $img]

	set doit 1
	# check for one dimension of 0. calculate to maintain aspect ratio
	if {$width == 0} {
	    set width [expr $iw*$height/$ih]

	    # see if we have a bad resample dimension
	    set wf [expr double($iw)*$height/$ih]
	    if {$width != $wf} {
		if {$debug(tcl,hv)} {
		    puts "HVImageURL abort resample image $img width $wf"
		}
		set doit 0
	    }
	}
	if {$height == 0} {
	    set height [expr $ih*$width/$iw]

	    # see if we have a bad resample dimension
	    set hf [expr double($ih)*$width/$iw]
	    if {$height != $hf} {
		if {$debug(tcl,hv)} {
		    puts "HVImageURL abort resample image $img height $hf"
		}
		set doit 0
	    }
	}

	# check to resample
	if {$doit && ($width != $iw || $height != $ih)} {
	    if {$debug(tcl,hv)} {
		puts "HVImageURL resample image $iw->$width $ih->$height"
	    }

	    set img2 \
		[image create photo -width $width -height $height]
	    if {[catch {blt::winop image resample $img $img2 box} ]} {
		# just use existing img
		if {$debug(tcl,hv)} {
		    puts "HVImageURL error resample image $img"
		}
	    } else {
		set tmp $img
		set img $img2
		catch {image delete $tmp}
	    }
	}
    }

    # delete any tmp files
    if {"$fn" != {}} {
	catch {file delete -force "$fn"}
    }

    set var(images,$url) $img
    return $img
}

proc HVFontCB {varname sz args} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVFont $varname $sz $args"
    }

    set family helvetica
    set param {}
    set size 12

    foreach f [concat [lindex $args 0]] {
	switch -- $f {
	    fixed {set family courier}
	    default {append param "$f "} 
	}
    }
    
    switch -- $sz {
	1 {set size 6}
	2 {set size 8}
	3 {set size 10}
	4 {set size 12}
	5 {set size 16}
	6 {set size 20}
	7 {set size 24}
	default {set size 12}
    }

#	1 {set size 8}
#	2 {set size 10}
#	3 {set size 12}
#	4 {set size 16}
#	5 {set size 20}
#	6 {set size 24}
#	7 {set size 36}
#	default {set size 16}

    incr size $var(font,size)

    return "$family $size $param"
}

proc HVNoScriptCB {varname n tag args} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVNoScript $varname $n $tag $args"
    }
}

proc HVScriptCB {varname args} {
    upvar #0 $varname var

    global debug

    if {$debug(tcl,hv)} {
	puts "HVScriptCB $varname $args"
    }
}

proc HVFrameCB {varname args} {
    upvar #0 $varname var
    global $varname

    if {$debug(tcl,hv)} {
	puts "HVFrameCB $varname $args"
    }
}

proc HVAppletCB {varname w args} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVAppletCB $varname $w $args"
    }
}

