#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc InitXPA {} {
    global xpa
    global ds9
    global help

    set xpa [xpacmdnew "DS9" $ds9(title)]

    xpacmdadd $xpa 2mass \
	"$help(2mass)" \
	XPASend2MASS {} {} \
	XPARcvd2MASS {} "fillbuf=false"

    xpacmdadd $xpa about \
	{} \
	XPASendAbout {} {} \
	{} {} {}

    xpacmdadd $xpa analysis \
	"\[task number\]\n\t\t\[clear\]\n\t\t\[clear\]\[load <filename>\]\n\t\t\[message ok|okcancel|yesno \{<message>\}\]\n\t\t\[entry \{<message>\}\]\n\t\t\[text\]" \
	XPASendAnalysis {} {} \
	XPARcvdAnalysis {} {}

    xpacmdadd $xpa array \
	"\t\[new\]\[\[xdim=<x>,ydim=<y>|dim=<dim>\],zdim=<z>,bitpix=<b>,\[skip=<s>\]\]\n\t\t\[new\] rgb \[\[xdim=<x>,ydim=<y>|dim=<dim>\],zdim=<z>,bitpix=<b>,\[skip=<s>\]\]"\
	XPASendArray {} {} \
	XPARcvdArray {} "fillbuf=false"

    xpacmdadd $xpa blink \
	{}\
	XPASendBlink {} {} \
	XPARcvdBlink {} "fillbuf=false"

    xpacmdadd $xpa bin \
	"$help(bin)" \
	XPASendBin {} {} \
	XPARcvdBin {} "fillbuf=false"

    xpacmdadd $xpa catalog \
	"$help(cat)" \
	XPASendCAT {} {} \
	XPARcvdCAT {} "fillbuf=false"

    xpacmdadd $xpa cd \
	"\t\[<directory>\]" \
	XPASendCD {} {} \
	XPARcvdCD {} "fillbuf=false"

    xpacmdadd $xpa cmap \
	"$help(cmap)" \
	XPASendColormap {} {} \
	XPARcvdColormap {} "fillbuf=false"

    xpacmdadd $xpa contour \
	"$help(contour)" \
	XPASendContour {} {} \
	XPARcvdContour {} "fillbuf=false"

    xpacmdadd $xpa crosshair \
	"\[x y <coordinate system> \[<sky frame>\]\[<sky format>\]\]" \
	XPASendCrosshair {} {} \
	XPARcvdCrosshair {} "fillbuf=false"

    xpacmdadd $xpa cursor \
	"\t\[x y\] -- move pointer or crosshair" \
	{} {} {} \
	XPARcvdCursor {} "fillbuf=false"

    xpacmdadd $xpa data \
	"$help(data)" \
	XPASendData {} {} \
	{} {} {}

    xpacmdadd $xpa datacube \
	"$help(datacube)" \
	XPASendDataCube {} {} \
	XPARcvdDataCube {} "fillbuf=false"

    xpacmdadd $xpa dss \
	"$help(dss)" \
	XPASendDSS {} {} \
	XPARcvdDSS {} "fillbuf=false"

    xpacmdadd $xpa exit \
	{} \
	{} {} {} \
	XPARcvdExit {} "fillbuf=false"

    xpacmdadd $xpa file \
	"$help(file)" \
	XPASendFile {} {} \
	XPARcvdFile {} "fillbuf=false"

    xpacmdadd $xpa fits \
	"$help(fits)" \
	XPASendFits {} "fillbuf=false" \
	XPARcvdFits {} "fillbuf=false"

    xpacmdadd $xpa frame \
	"$help(frame)" \
	XPASendFrame {} {} \
	XPARcvdFrame {} "fillbuf=false"

    xpacmdadd $xpa grid \
	"$help(grid)" \
	XPASendGrid {} {} \
	XPARcvdGrid {} "fillbuf=false"

    xpacmdadd $xpa height \
	"\t$help(height)" \
	XPASendHeight {} {} \
	XPARcvdHeight {} "fillbuf=false"

    xpacmdadd $xpa iconify \
	"$help(iconify)" \
	XPASendIconify {} {} \
	XPARcvdIconify {} "fillbuf=false"

    xpacmdadd $xpa iis \
	"\t\[filename <filename> \[\#\]\]" \
	XPASendIIS {} {} \
	XPARcvdIIS {} "fillbuf=false"

    xpacmdadd $xpa imexam \
	"\t\[coordinate <coordinate system> \[<sky frame>\] \[<sky format>\]\]\n\t\t\[data \[width\]\[height\]\]" \
	XPASendImexam {} {} \
	{} {} {}

    xpacmdadd $xpa lock \
	"$help(lock)" \
	{} {} {} \
	XPARcvdLock {} "fillbuf=false"

    xpacmdadd $xpa lower \
	{} \
	{} {} {} \
	XPARcvdLower {} "fillbuf=false"

    xpacmdadd $xpa match \
	"$help(match)" \
	{} {} {} \
	XPARcvdMatch {} "fillbuf=false"

    xpacmdadd $xpa minmax \
	"$help(minmax)" \
	XPASendMinMax {} {} \
	XPARcvdMinMax {} "fillbuf=false"

    xpacmdadd $xpa mode \
	"$help(mode)" \
	XPASendMode {} {} \
	XPARcvdMode {} "fillbuf=false"

    xpacmdadd $xpa nameserver \
	"$help(nres)" \
	XPASendNRES {} {} \
	XPARcvdNRES {} "fillbuf=false"

    xpacmdadd $xpa orient \
	"$help(orient)" \
	XPASendOrient {} {} \
	XPARcvdOrient {} "fillbuf=false"

    xpacmdadd $xpa {page setup} \
	"$help(pagesetup)" \
	XPASendPageSetup {} {} \
	XPARcvdPageSetup {} "fillbuf=false"

    xpacmdadd $xpa pan \
	"$help(pan)" \
	XPASendPan {} {} \
	XPARcvdPan {} "fillbuf=false"

    xpacmdadd $xpa pixeltable \
	"$help(pixeltable)" \
	XPASendPixelTable {} {} \
	XPARcvdPixelTable {} "fillbuf=false"

    xpacmdadd $xpa prefs \
	"$help(prefs)" \
	XPASendPrefs {} {} \
	XPARcvdPrefs {} "fillbuf=false"

    xpacmdadd $xpa preserve \
	"$help(preserve)" \
	XPASendPreserve {} {} \
	XPARcvdPreserve {} "fillbuf=false"

    xpacmdadd $xpa print \
	"$help(print)" \
	XPASendPrint {} {} \
	XPARcvdPrint {} "fillbuf=false"

    xpacmdadd $xpa plot \
	"$help(plot)" \
	XPASendPlot {} {} \
	XPARcvdPlot {} {}

    xpacmdadd $xpa quit \
	{} \
	{} {} {} \
	XPARcvdExit {} "fillbuf=false"

    xpacmdadd $xpa raise \
	{} \
	{} {} {} \
	XPARcvdRaise {} "fillbuf=false"

    xpacmdadd $xpa regions \
	"$help(regions)" \
	XPASendRegions {} {} \
	XPARcvdRegions {} "fillbuf=false"

    xpacmdadd $xpa rgb \
	"$help(rgb)" \
	XPASendRGB {} {} \
	XPARcvdRGB {} "fillbuf=false"

    xpacmdadd $xpa rotate \
	"$help(rotate)" \
	XPASendRotate {} {} \
	XPARcvdRotate {} "fillbuf=false"

    xpacmdadd $xpa saveimage \
	"$help(save,image)" \
	{} {} {} \
	XPARcvdSaveImage {} "fillbuf=false"

    xpacmdadd $xpa savefits \
	"$help(save,fits)" \
	{} {} {} \
	XPARcvdSaveFits {} "fillbuf=false"

    xpacmdadd $xpa savempeg \
	"$help(save,mpeg)" \
	{} {} {} \
	XPARcvdSaveMPEG {} "fillbuf=false"

    xpacmdadd $xpa scale \
	"$help(scale)" \
	XPASendScale {} {} \
	XPARcvdScale {} "fillbuf=false"

    xpacmdadd $xpa single \
	{} \
	XPASendSingle {} {} \
	XPARcvdSingle {} "fillbuf=false"

    xpacmdadd $xpa shm \
	"$help(shm)" \
	XPASendShm {} {} \
	XPARcvdShm {} "fillbuf=false"

    xpacmdadd $xpa smooth \
	"$help(smooth)" \
	XPASendSmooth {} {} \
	XPARcvdSmooth {} "fillbuf=false"

    xpacmdadd $xpa source \
	"\t\[filename\]" \
	{} {} {} \
	XPARcvdSource {} "fillbuf=false"

    xpacmdadd $xpa tcl \
	"\t\[<tcl command>\]" \
	{} {} {} \
	XPARcvdTcl {} {}

    xpacmdadd $xpa tile \
	"$help(tile)" \
	XPASendTile {} {} \
	XPARcvdTile {} "fillbuf=false"

    xpacmdadd $xpa update \
	"\t\[\]\n\t\t\[\# x1 y1 x2 y2\]\n\t\t\[now\]\n\t\t\[now \# x1 y1 x2 y2\]\n\t\t\[on\]\n\t\t\[off\]" \
	{} {} {} \
	XPARcvdUpdate {} "fillbuf=false"

    xpacmdadd $xpa view \
	"$help(view)" \
	XPASendView {} {} \
	XPARcvdView {} "fillbuf=false"

    xpacmdadd $xpa version \
	{} \
	XPASendVersion {} {} \
	{} {} {}

    xpacmdadd $xpa vo \
	"$help(vo)" \
	XPASendVO {} {} \
	XPARcvdVO {} "fillbuf=false"

    xpacmdadd $xpa wcs \
	"$help(wcs)" \
	XPASendWCS {} {} \
	XPARcvdWCS {} "fillbuf=false"

    xpacmdadd $xpa web \
	"$help(web)" \
	XPASendWeb {} {} \
	XPARcvdWeb {} "fillbuf=false"

    xpacmdadd $xpa width \
	"\t$help(width)" \
	XPASendWidth {} {} \
	XPARcvdWidth {} "fillbuf=false"

    xpacmdadd $xpa zoom \
	"$help(zoom)" \
	XPASendZoom {} {} \
	XPARcvdZoom {} "fillbuf=false"
}

proc XPASend2MASS {xpa cdata param} {
    global 2mass

    InitXPAError $xpa
    catch {
	switch -- [string tolower [lindex $param 0]] {
	    survey {xpasetbuf $xpa "$2mass(survey)\n"}
	    coord {xpasetbuf $xpa "$2mass(x) $2mass(y)\n"}
	    x {xpasetbuf $xpa "$2mass(x)\n"}
	    y {xpasetbuf $xpa "$2mass(y)\n"}
	    size {xpasetbuf $xpa "$2mass(radius)\n"}
	    width {xpasetbuf $xpa "$2mass(width)\n"}
	    height {xpasetbuf $xpa "$2mass(height)\n"}
	    name -
	    default {xpasetbuf $xpa "$2mass(name)\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvd2MASS {xpa cdata param buf len} {
    set i 0
    catch {Process2MASSCmd param i 0 foobar}
    # http has an error, so just clear
    InitXPAError $xpa
}

proc XPASendAbout {xpa cdata param} {
    global help

    InitXPAError $xpa
    catch {xpasetbuf $xpa "$help(about)\n"}
    CatchXPAError $xpa
}

proc XPASendAnalysis {xpa cdata param} {
    global analysis

    InitXPAError $xpa
    catch {
	set result {}
	switch -- [string tolower [lindex $param 0]] {
	    entry {
		AnalysisEntry [lrange $param 1 end] result
		append result "\n"
	    }
	    default {
		for {set i 0} {$i<$analysis(menu,count)} {incr i} {
		    set result "$result\#$i menu"
		    set result "$result\n$analysis(menu,$i,item)"
		    set result "$result\n$analysis(menu,$i,template)"
		    set result "$result\n$analysis(menu,$i,cmd)"
		    set result "$result\n\n"
		}
		for {set i 0} {$i<$analysis(bind,count)} {incr i} {
		    set result "$result\#$i bind"
		    set result "$result\n$analysis(bind,$i,item)"
		    set result "$result\n$analysis(bind,$i,template)"
		    set result "$result\n$analysis(bind,$i,cmd)"
		    set result "$result\n\n"
		}
	    }
	}
	xpasetbuf $xpa $result
    }
    CatchXPAError $xpa
}

proc XPARcvdAnalysis {xpa cdata param buf len} {
    global ap

    InitXPAError $xpa
    catch {
	switch -- [string tolower [lindex $param 0]] {
	    message {
		switch -- [llength $param] {
		    0 -
		    1 {}
		    2 {AnalysisMessage ok [lindex $param 1]}
		    3 {AnalysisMessage [lindex $param 1] [lindex $param 2]}
		}
	    }
	    text {AnalysisText apXPA Analysis $buf append}
	    plot {
		# for backward compatibility
		# use xpa plot instead

		set tt $ap(tt)
		if {[string tolower [lindex $param 1]] == "stdin"} {
		    lappend ap(xpa) [AnalysisPlotStdin  $tt {} buf]
		} elseif {[string tolower [lindex $param 1]] == "close"} {
		    foreach f $ap(xpa) {
			APDestroy $f
		    }
		    set ap(xpa) {}
		} elseif {[llength $param] == 1} {
		    lappend ap(xpa) [AnalysisPlot $tt Plot {} {} {} xy $buf]
		} elseif {[llength $param] == 5}  {
		    lappend ap(xpa) \
			[AnalysisPlot $tt Plot \
			     [lindex $param 1] \
			     [lindex $param 2] \
			     [lindex $param 3] \
			     [lindex $param 4] \
			     $buf]
		}
	    }
	    load {XPASendAnalysisLoad $xpa buf [lindex $param 1]}
	    clear {
		ClearAnalysis
		switch -- [lindex $param 1] {
		    load {XPASendAnalysisLoad $xpa buf [lindex $param 2]}
		}
	    }
	    default {
		AnalysisTask [expr $param-1] menu {} 0 0
		# don't know why
		global errorInfo
		set errorInfo {}
	    }
	}
    }
    CatchXPAError $xpa
}

proc XPASendAnalysisLoad {xpa bufname fn} {
    upvar $bufname buf

    InitXPAError $xpa
    catch {
	if {$fn != {}} {
	    ProcessAnalysisFile $fn
	} else {
	    ProcessAnalysis buf
	}
	LayoutFrames
    }
    CatchXPAError $xpa
}

proc XPASendArray {xpa cdata param} {
    global current
    global message

    InitXPAError $xpa
    catch {
	if {$current(frame) != {}} {
	    if {[$current(frame) has fits]} {
		$current(frame) save array socket [xparec $xpa datafd]
	    }
	} else {
	    Error "$message(error,xpa,file)"
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdArray {xpa cdata param buf len} {
    global current
    global message

    InitXPAError $xpa
    catch {
	switch -- [string tolower [lindex $param 0]] {
	    new {
		switch -- [string tolower [lindex $param 1]] {
		    rgb {CreateRGBFrame}
		    default {CreateFrame}
		}
		set param [lreplace $param 0 0]
	    }
	}

	if {$current(frame) != {}} {
	    global loadParam

	    set loadParam(load,type) socketgz
	    set loadParam(socket,id) [xparec $xpa datafd]
	    set loadParam(file,type) array
	    set loadParam(file,mode) {}
	    set loadParam(file,name) [lindex $param 0]
	    if {$loadParam(file,name) == {}} {
		set loadParam(file,name) stdin
	    }

	    StartLoad
	    switch -- [lindex $param 0] {
		rgb {
		    if {[$current(frame) get type] == "rgb"} {
			set loadParam(file,mode) {rgb cube}
			set loadParam(file,name) [lindex $param 1]
			ProcessLoad
		    } else {
			Error "$message(error,fits,rgb)"
		    }
		}
		default {ProcessLoad}
	    }
	    FinishLoad
	}
    }
    CatchXPAError $xpa
}

proc XPASendBlink {xpa cdata param} {
    global ds9

    InitXPAError $xpa
    catch {
	if {$ds9(display,user) == "blink"} {
	    xpasetbuf $xpa [ToYesNo 1]
	} else {
	    xpasetbuf $xpa [ToYesNo 0]
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdBlink {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0;ProcessBlinkCmd param i}
    CatchXPAError $xpa
}

proc XPASendBin {xpa cdata param} {
    global bin
    global current

    InitXPAError $xpa
    catch {
	switch -- [string tolower [lindex $param 0]] {
	    about {
		if {$current(frame) != {}} {
		    xpasetbuf $xpa "[$current(frame) get bin cursor]\n"
		}
	    }
	    buffersize {xpasetbuf $xpa "$bin(buffersize)\n"}
	    cols {
		if {$current(frame) != {}} {
		    xpasetbuf $xpa "[$current(frame) get bin cols]\n"
		}
	    }
	    factor {xpasetbuf $xpa "$bin(factor)\n"}
	    depth {xpasetbuf $xpa "$bin(depth)\n"}
	    filter {
		if {$current(frame) != {}} {
		    xpasetbuf $xpa "[$current(frame) get bin filter]\n"
		}
	    }
	    function {xpasetbuf $xpa "$bin(function)\n"}
	    # back compatibility
	    smooth {
		global smooth
		switch -- [string tolower [lindex $param 1]] {
		    function {xpasetbuf $xpa "$smooth(function)\n"}
		    radius {xpasetbuf $xpa "$smooth(radius)\n"}
		    default {xpasetbuf $xpa "$smooth(view)\n"}
		}
	    }
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdBin {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessBinCmd param i}
    CatchXPAError $xpa
}

proc XPASendCAT {xpa cdata param} {
    global current
    global cat

    InitXPAError $xpa
    catch {
	switch -- [string tolower [lindex $param 0]] {
	    {} {xpasetbuf $xpa "$cat(current)\n"}
	    header {
		set cc [lindex $cat(current) end]
		xpasetbuf $xpa "[CATGetHeader $cc]\n"
	    }
	    default {
		set cc [lindex $param 0]
		xpasetbuf $xpa "[CATGetHeader $cc]\n"
	    }
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdCAT {xpa cdata param buf len} {
    catch {set i 0; ProcessCatalogCmd param i}
    # http has an error, so just clear
    InitXPAError $xpa
}

proc XPASendCD {xpa cdata param} {
    InitXPAError $xpa
    catch {xpasetbuf $xpa "[pwd]\n"}
    CatchXPAError $xpa
}

proc XPARcvdCD {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {cd $param}
    CatchXPAError $xpa
}

proc XPASendContour {xpa cdata param} {
    global contour
    global current

    InitXPAError $xpa
    catch {
	if {$param != {}} {
	    if {$current(frame) != {}} {
		xpasetbuf $xpa [$current(frame) get contour [lindex $param 0] [lindex $param 1]]
	    }
	} else {
	    xpasetbuf $xpa [ToYesNo $contour(view)]
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdContour {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessContourCmd param i}
    CatchXPAError $xpa
}

proc XPASendColormap {xpa cdata param} {
    global current
    global colorbar

    InitXPAError $xpa
    catch {
	switch -- [string tolower $param] {
	    file {xpasetbuf $xpa "[$current(colorbar) get file name]\n"}
	    invert {xpasetbuf $xpa [ToYesNo $colorbar(invert)]}
	    value {xpasetbuf $xpa "[$current(colorbar) get contrast] [$current(colorbar) get bias]\n"}
	    {} {xpasetbuf $xpa "[$current(colorbar) get name]\n"}
	    default {xpaerror $xpa "Illegal option: $param"}
	}
	CatchXPAError $xpa
    }
}

proc XPARcvdColormap {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessCmapCmd param i}
    CatchXPAError $xpa
}

proc XPASendCrosshair {xpa cdata param} {
    global current

    InitXPAError $xpa
    catch {
	set sys [lindex $param 0]
	set sky [lindex $param 1]
	set skyformat [lindex $param 2]
	if {$current(frame) != {}} {
	    switch -- $sys {
		{} {set sys physical}
		fk4 -
		fk5 -
		icrs -
		galactic -
		ecliptic {set skyformat $sky; set sky $sys; set sys wcs}
	    }

	    xpasetbuf $xpa \
		"[$current(frame) get crosshair $sys $sky $skyformat]\n"
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdCrosshair {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessCrosshairCmd param i}
    CatchXPAError $xpa
}

proc XPARcvdCursor {xpa cdata param buf len} {
    global current

    InitXPAError $xpa
    catch {
	if {$current(frame) != {}} {
	    set x [lindex $param 0]
	    set y [lindex $param 1]
	    ArrowKeyFrame $current(frame) $x $y
	}
    }
    CatchXPAError $xpa
}

proc XPASendDSS {xpa cdata param} {
    global dss

    InitXPAError $xpa
    catch {
	switch -- [string tolower [lindex $param 0]] {
	    server {xpasetbuf $xpa "$dss(server)\n"}
	    survey {xpasetbuf $xpa "$dss(survey)\n"}
	    coord {xpasetbuf $xpa "$dss(x) $dss(y)\n"}
	    x {xpasetbuf $xpa "$dss(x)\n"}
	    y {xpasetbuf $xpa "$dss(y)\n"}
	    size {xpasetbuf $xpa "$dss(width) $dss(height)\n"}
	    width {xpasetbuf $xpa "$dss(width)\n"}
	    height {xpasetbuf $xpa "$dss(height)\n"}
	    name -
	    default {xpasetbuf $xpa "$dss(name)\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdDSS {xpa cdata param buf len} {
    catch {set i 0; ProcessDSSCmd param i 0 foobar}
    # http has an error, so just clear
    InitXPAError $xpa
}

proc XPASendData {xpa cdata param} {
    global cube
    global blink
    global current

    InitXPAError $xpa
    catch {
	if {$current(frame) != {}} {
	    set sys [lindex $param 0]
	    set sky [lindex $param 1]
	    set x [lindex $param 2]
	    set y [lindex $param 3]
	    set w [lindex $param 4]
	    set h [lindex $param 5]
	    set strip [lindex $param 6]
	    switch -- $sys {
		image -
		physical -
		detector -
		amplifier {
		    set strip $h
		    set h $w
		    set w $y
		    set y $x
		    set x $sky
		    set sky fk5
		}
		fk5 -
		fk4 -
		icrs -
		galactic -
		ecliptic {
		    set strip $h
		    set h $w
		    set w $y
		    set y $x
		    set x $sky
		    set sky $sys
		    set sys wcs
		}
	    }
	    set strip [FromYesNo $strip]

	    $current(frame) get data $sys $sky $x $y $w $h rr
	    set ss {}
	    foreach id [array names rr] {
		if {$strip} {
		    append ss "$rr($id)\n"
		} else {
		    append ss "$id = $rr($id)\n"
		}
	    }
	    xpasetbuf $xpa "$ss"
	}
    }
    CatchXPAError $xpa
}

proc XPASendDataCube {xpa cdata param} {
    global cube
    global blink

    InitXPAError $xpa
    catch {
	switch -- [string tolower [lindex $param 0]] {
	    interval {xpasetbuf $xpa "$blink(interval)\n"}
	    default {xpasetbuf $xpa "$cube(slice)\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdDataCube {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessDataCubeCmd param i}
    CatchXPAError $xpa
}

proc XPARcvdExit {xpa cdata param buf len} {
    QuitDS9
}

proc XPASendFile {xpa cdata param} {
    global current

    InitXPAError $xpa
    catch {
	if {$current(frame) != {}} {
	    xpasetbuf $xpa "[$current(frame) get fits file name full]\n"
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdFile {xpa cdata param buf len} {
    global current

    InitXPAError $xpa
    catch {
	switch -- [string tolower [lindex $param 0]] {
	    new {
		switch -- [string tolower [lindex $param 1]] {
		    rgbimage -
		    rgbcube -
		    rgbarray {CreateRGBFrame}
		    default {CreateFrame}
		}
		set param [lreplace $param 0 0]
	    }
	}

	set fn  [file normalize [lindex $param 1]]
	set fn2 [file normalize [lindex $param 2]]
	set fn3 [file normalize [lindex $param 3]]

	switch -- [lindex $param 0] {
	    save {
		switch -- [lindex $param 1] {
		    gz {
			$current(frame) save fits image file $fn2 gz
		    }
		    resample {
			switch -- [lindex $param 2] {
			    gz {$current(frame) save fits resample file \
				    $fn3 gz}
			    default {$current(frame) save fits resample file \
					 $fn2}
			}
		    }
		    default {$current(frame) save fits image file $fn}
		}
	    }
	    default {
		StartLoad
		switch -- [lindex $param 0] {
		    fits {LoadFits $fn}
		    sfits {LoadSFits $fn $fn2}

		    datacube -
		    medatacube {LoadDataCubeFits $fn}

		    mosaicimage {
			if {[lindex $param 2] == {}} {
			    LoadMosaicImageIRAFFits $fn
			} else {
			    switch -- [lindex $param 1] {
				iraf {LoadMosaicImageIRAFFits $fn2}
				wfpc2 {LoadMosaicImageWFPC2Fits $fn2}
				default {LoadMosaicImageWCSFits \
					     [lindex $param 1] $fn2}
			    }
			}
		    }
		    mosaicimagenext {
			if {[lindex $param 2] == {}} {
			    LoadMosaicImageNextWCSFits wcs $fn
			} else {
			    LoadMosaicImageNextWCSFits [lindex $param 1] $fn2
			}
		    }
		    mosaic {
			if {[lindex $param 2] == {}} {
			    LoadMosaicIRAFFits $fn
			} else {
			    switch -- [lindex $param 1] {
				iraf {LoadMosaicIRAFFits $fn2}
				default {LoadMosaicWCSFits \
					     [lindex $param 1] $fn2}
			    }
			}
		    }
		    smosaic {
			switch -- [lindex $param 1] {
			    iraf {LoadMosaicIRAFSFits $fn2 $fn3}
			    default {LoadMosaicWCSSFits [lindex $param 1] \
					 $fn2 $fn3}
			}
		    }

		    mosaicimageiraf {
			# backward compatibility
			LoadMosaicImageIRAFFits $fn
		    }
		    mosaiciraf {
			# backward compatibility
			LoadMosaicIRAFFits $fn
		    }
		    mosaicimagewcs {
			# backward compatibility
			LoadMosaicImageWCSFits wcs $fn
		    }
		    mosaicimagenextwcs {
			# backward compatibility
			LoadMosaicImageNextWCSFits wcs $fn
		    }
		    mosaicwcs {
			# backward compatibility
			LoadMosaicWCSFits wcs $fn
		    }
		    mosaicwfpc2 -
		    mosaicimagewfpc2 {
			# backward compatibility
			LoadMosaicImageWFPC2Fits $fn
		    }

		    rgbcube {LoadRGBCubeFits $fn}
		    srgbcube {LoadRGBCubeSFits $fn $fn2}
		    rgbimage {LoadRGBImageFits $fn}
		    rgbarray {LoadRGBArray $fn}

		    array {LoadArray $fn}

		    url {LoadURL [lindex $param 1]}
		    default {LoadFits [file normalize [lindex $param 0]]}
		}
		FinishLoad
	    }
	}
    }
    CatchXPAError $xpa
}

proc XPASendFits {xpa cdata param} {
    global current
    global message

    InitXPAError $xpa
    catch {
	if {$current(frame) != {}} {
	    if {[$current(frame) has fits]} {
		switch -- [string tolower [lindex $param 0]] {
		    size {
			xpasetbuf $xpa "[$current(frame) get fits size [lindex $param 1] [lindex $param 2]]\n"
		    }
		    depth {
			xpasetbuf $xpa "[$current(frame) get fits depth]\n"
		    }
		    header {
			switch [llength $param] {
			    1 {xpasetbuf $xpa "[$current(frame) get fits header 1]\n"}
			    2 {xpasetbuf $xpa "[$current(frame) get fits header [lindex $param 1]]\n"}
			    3 {xpasetbuf $xpa "[$current(frame) get fits header 1 keyword [lindex $param 2]]\n"}
			    4 {xpasetbuf $xpa "[$current(frame) get fits header [lindex $param 1] keyword [lindex $param 3]]\n"}
			}
		    }
		    type {
			if {[$current(frame) has fits image]} {
			    xpasetbuf $xpa "image\n"
			} elseif {[$current(frame) has fits table]} {
			    xpasetbuf $xpa "table\n"
			}
		    }
		    table {
			switch -- [lindex $param 1] {
			    gz {
				$current(frame) save fits table socket \
				    [xparec $xpa datafd] gz
			    }
			    default {
				$current(frame) save fits table socket \
				    [xparec $xpa datafd]
			    }
			}
		    }
		    image {
			switch -- [lindex $param 1] {
			    gz {
				$current(frame) save fits image socket \
				    [xparec $xpa datafd] gz
			    }
			    default {
				$current(frame) save fits image socket \
				    [xparec $xpa datafd]
			    }
			}
		    }
		    resample {
			switch -- [lindex $param 1] {
			    gz {
				$current(frame) save fits resample socket \
				    [xparec $xpa datafd] gz
			    }
			    default {
				$current(frame) save fits resample socket \
				    [xparec $xpa datafd]
			    }
			}
		    }
		    gz {
			$current(frame) save fits image socket \
			    [xparec $xpa datafd] gz
		    }
		    default {
			$current(frame) save fits image socket \
			    [xparec $xpa datafd]
		    }
		}
	    } else {
		Error "$message(error,xpa,file)"
	    }
	} else {
	    Error "$message(error,xpa,file)"
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdFits {xpa cdata param buf len} {
    global current
    global ds9
    global message

    InitXPAError $xpa
    catch {
	switch -- [string tolower [lindex $param 0]] {
	    new {
		CreateFrame
		set param [lreplace $param 0 0]
	    }
	}

	if {$current(frame) != {}} {
	    global loadParam
	    set loadParam(load,type) socketgz
	    set loadParam(socket,id) [xparec $xpa datafd]
	    set loadParam(file,type) fits
	    set loadParam(file,mode) {}
	    set loadParam(file,name) [lindex $param 1]
	    if {$loadParam(file,name) == {}} {
		set loadParam(file,name) stdin
	    }

	    StartLoad
	    switch -- [lindex $param 0] {
		datacube -
		medatacube {
		    set loadParam(file,mode) {data cube}
		    ProcessLoad
		}

		mosaicimage {
		    if {[lindex $param 2] == {}} {
			set loadParam(file,mode) [list mosaic image iraf]
		    } else {
			set loadParam(file,mode) \
			    [list mosaic image [lindex $param 1]]
			set loadParam(file,name) [lindex $param 2]
		    }
		    ProcessLoad
		}
		mosaicimagenext {
		    if {[lindex $param 2] == {}} {
			set loadParam(file,mode) [list mosaic image next wcs]
		    } else {
			set loadParam(file,mode) \
			    [list mosaic image next [lindex $param 1]]
			set loadParam(file,name) [lindex $param 2]
		    }
		    ProcessLoad
		}
		mosaic {
		    if {[lindex $param 2] == {}} {
			set loadParam(file,mode) [list mosaic iraf]
		    } else {
			set loadParam(file,mode) \
			    [list mosaic [lindex $param 1]]
			set loadParam(file,name) [lindex $param 2]
		    }
		    ProcessLoad
		}

		mosaicimageiraf {
		    # backward compatibility
		    set loadParam(file,mode) {mosaic image iraf}
		    ProcessLoad
		}
		mosaiciraf {
		    # backward compatibility
		    set loadParam(file,mode) {mosaic iraf}
		    ProcessLoad
		}
		mosaicimagewcs {
		    # backward compatibility
		    set loadParam(file,mode) {mosaic image wcs}
		    ProcessLoad
		}
		mosaicimagenextwcs {
		    # backward compatibility
		    set loadParam(file,mode) {mosaic image next wcs}
		    ProcessLoad
		}
		mosaicwcs {
		    # backward compatibility
		    set loadParam(file,mode) {mosaic wcs}
		    ProcessLoad
		}
		mosaicwfpc2 -
		mosaicimagewfpc2 {
		    # backward compatibility
		    set loadParam(file,mode) {mosaic image wfpc2}
		    ProcessLoad
		}

		rgbimage {
		    if {[$current(frame) get type] == "rgb"} {
			set loadParam(file,mode) {rgb image}
			ProcessLoad
		    } else {
			Error "$message(error,fits,rgb)"
		    }
		}
		rgbcube {
		    if {[$current(frame) get type] == "rgb"} {
			set loadParam(file,mode) {rgb cube}
			ProcessLoad
		    } else {
			Error "$message(error,fits,rgb)"
		    }
		}
		default {
		    set loadParam(file,name) [lindex $param 0] 
		    if {$loadParam(file,name) == {}} {
			set loadParam(file,name) stdin
		    }
		    ProcessLoad
		}
	    }
	    FinishLoad
	}
    }
    CatchXPAError $xpa
}

proc XPASendFrame {xpa cdata param} {
    global ds9
    global current
    global rgb

    InitXPAError $xpa
    catch {
	switch -- [lindex $param 0] {
	    active {
		set r {}
		foreach f $ds9(active) {
		    append r "[string range $f 5 end] "
		}
		xpasetbuf $xpa "$r\n"
	    }
	    all {
		set r {}
		foreach f $ds9(frames) {
		    append r "[string range $f 5 end] "
		}
		xpasetbuf $xpa "$r\n"
	    }
	    default {xpasetbuf $xpa "[string range $current(frame) 5 end]\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdFrame {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessFrameCmd param i}
    CatchXPAError $xpa
}

proc XPASendGrid {xpa cdata param} {
    global grid

    InitXPAError $xpa
    catch {
	switch -- [lindex $param 0] {
	    system {xpasetbuf $xpa "$grid(system)\n"}
	    sky {xpasetbuf $xpa "$grid(sky)\n"}
	    skyformat {xpasetbuf $xpa "$grid(skyformat)\n"}
	    type {
		switch -- [lindex $param 1] {
		    axes {xpasetbuf $xpa "$grid(labelling)\n"}
		    numerics {xpasetbuf $xpa "$grid(type,numlab)\n"}
		    default {xpasetbuf $xpa "$grid(type)\n"}
		}
	    }
	    view {
		switch -- [lindex $param 1] {
		    grid {xpasetbuf $xpa [ToYesNo $grid(grid)]}
		    axes {
			switch -- [lindex $param 2] {
			    numbers {xpasetbuf $xpa [ToYesNo $grid(numlab)]}
			    tickmarks {xpasetbuf $xpa [ToYesNo $grid(tick)]}
			    label {xpasetbuf $xpa [ToYesNo $grid(textlab)]}
			    default {xpasetbuf $xpa [ToYesNo $grid(axes)]}
			}
		    }
		    title {xpasetbuf $xpa [ToYesNo $grid(title)]}
		    border {xpasetbuf $xpa [ToYesNo $grid(border)]}
		    vertical {xpasetbuf $xpa [ToYesNo $grid(labelup)]}
		}
	    }
	    default {xpasetbuf $xpa [ToYesNo $grid(view)]}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdGrid {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessGridCmd param i}
    CatchXPAError $xpa
}

proc XPASendHeight {xpa cdata param} {
    InitXPAError $xpa
    catch {xpasetbuf $xpa "[CanvasTrueHeight]\n"}
    CatchXPAError $xpa
}

proc XPARcvdHeight {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessHeightCmd param i}
    CatchXPAError $xpa
}

proc XPASendIconify {xpa cdata param} {
    global ds9

    InitXPAError $xpa
    catch {
	if {[wm state $ds9(top)] == "normal"} {
	    xpasetbuf $xpa "no\n"
	} else {
	    xpasetbuf $xpa "yes\n"
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdIconify {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessIconifyCmd param i}
    CatchXPAError $xpa
}

proc XPASendIIS {xpa cdata param} {
    global current

    InitXPAError $xpa
    catch {
	if {$current(frame) != {}} {
	    switch -- [string tolower [lindex $param 0]] {
		filename {
		    xpasetbuf $xpa \
		    "[$current(frame) get iis file name [lindex $param 1]]\n"
		}
	    }
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdIIS {xpa cdata param buf len} {
    global current

    InitXPAError $xpa
    catch {
	if {$current(frame) != {}} {
	    switch -- [string tolower [lindex $param 0]] {
		filename {
		    $current(frame) iis set file name \
			[lindex $param 1] [lindex $param 2]
		}
	    }
	}
    }
    CatchXPAError $xpa
}

proc XPASendImexam {xpa cdata param} {
    global ds9
    global cursor

    InitXPAError $xpa
    catch {
	set ds9(imexam,x) {}
	set ds9(imexam,y) {}
	set ds9(imexam,which) {}
	set ds9(imexam,mode) $ds9(mode)

	set ds9(mode) imexam
	set ds9(imexam) 1

	# turn on blinking cursor
	set cursor(timer) 1
	CursorTimer

	switch -- [string tolower [lindex $param 0]] {
	    data {
		vwait ds9(imexam)
		set w [lindex $param 1]
		set h [lindex $param 2]
		if {$w == {}} {
		    set w 1
		}
		if {$h == {}} {
		    set h 1
		}
		xpasetbuf $xpa "[$ds9(imexam,which) get data canvas $ds9(imexam,x) $ds9(imexam,y) $w $h]\n"
	    }
	    coordinate {
		set sys [lindex $param 1]
		set sky [lindex $param 2]
		set skyformat [lindex $param 3]
		switch -- $skyformat {
		    {} {set skyformat degrees}
		}
		switch -- $sky {
		    {} {set sky fk5}
		}
		switch -- $sys {
		    {} {set sys physical}
		    fk4 -
		    fk5 -
		    icrs -
		    galactic -
		    ecliptic {set sky $sys; set sys wcs}
		}

		vwait ds9(imexam)
		xpasetbuf $xpa "[$ds9(imexam,which) get coordinates canvas $ds9(imexam,x) $ds9(imexam,y) $sys $sky $skyformat]\n"
	    }
	}
	# turn off blinking cursor
	set cursor(timer) 0

	set ds9(imexam) 0
	set ds9(mode) $ds9(imexam,mode)

	set ds9(imexam,x) {}
	set ds9(imexam,y) {}
	set ds9(imexam,which) {}
	set ds9(imexam,mode) {}
    }
    CatchXPAError $xpa
}

proc XPARcvdLock {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessLockCmd param i}
    CatchXPAError $xpa
}

proc XPARcvdLower {xpa cdata param buf len} {
    global ds9

    InitXPAError $xpa
    catch {set i 0; ProcessLowerCmd param i}
    CatchXPAError $xpa
}

proc XPARcvdMatch {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessMatchCmd param i}
    CatchXPAError $xpa
}

proc XPASendMinMax {xpa cdata param} {
    global current
    global minmax

    InitXPAError $xpa
    catch {
	switch -- [string tolower $param] {
	    mode {xpasetbuf $xpa "$minmax(mode)\n"}
	    interval {xpasetbuf $xpa "$minmax(sample)\n"}
	    default {xpasetbuf $xpa "$minmax(mode)\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdMinMax {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessMinMaxCmd param i}
    CatchXPAError $xpa
}

proc XPASendMode {xpa cdata param} {
    global ds9

    InitXPAError $xpa
    catch {xpasetbuf $xpa "$ds9(mode)\n"}
    CatchXPAError $xpa
}

proc XPARcvdMode {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessModeCmd param i}
    CatchXPAError $xpa
}

proc XPASendNRES {xpa cdata param} {
    global nres

    InitXPAError $xpa
    catch {
	switch -- [string tolower [lindex $param 0]] {
	    server {xpasetbuf $xpa "$nres(nameserver)\n"}
	    format -
	    skyformat {xpasetbuf $xpa "$nres(skyformat)\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdNRES {xpa cdata param buf len} {
    catch {set i 0; ProcessNRESCmd param i}
    # http has an error, so just clear
    InitXPAError $xpa
}

proc XPASendOrient {xpa cdata param} {
    global current

    InitXPAError $xpa
    catch {xpasetbuf $xpa "$current(orient)\n"}
    CatchXPAError $xpa
}

proc XPARcvdOrient {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessOrientCmd param i}
    CatchXPAError $xpa
}

proc XPASendPageSetup {xpa cdata param} {
    global ps

    InitXPAError $xpa
    catch {
	switch -- [string tolower $param] {
	    orientation {xpasetbuf $xpa "$ps(orient)\n"}
	    pagescale {xpasetbuf $xpa "$ps(scale)\n"}
	    pagesize {xpasetbuf $xpa "$ps(size)\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdPageSetup {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessPageSetupCmd param i}
    CatchXPAError $xpa
}

proc XPASendPan {xpa cdata param} {
    global current

    InitXPAError $xpa
    catch {
	set sys [lindex $param 0]
	set sky [lindex $param 1]
	set skyformat [lindex $param 2]
	if {$current(frame) != {}} {
	    switch -- $sys {
		{} {set sys physical}
		fk4 -
		fk5 -
		icrs -
		galactic -
		ecliptic {set skyformat $sky; set sky $sys; set sys wcs}
	    }

	    xpasetbuf $xpa "[$current(frame) get pan $sys $sky $skyformat]\n"
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdPan {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessPanCmd param i}
    CatchXPAError $xpa
}

proc XPASendPixelTable {xpa cdata param} {
    global pixel

    InitXPAError $xpa
    catch {xpasetbuf $xpa [ToYesNo $pixel(view)]}
    CatchXPAError $xpa
}

proc XPARcvdPixelTable {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessPixelTableCmd param i}
    CatchXPAError $xpa
}

proc XPASendPrefs {xpa cdata param} {
    global prefs
    global ds9

    InitXPAError $xpa
    catch {
	switch -- [string tolower $param] {
	    mosaicfast {xpasetbuf $xpa "$ds9(mosaic,fast)\n"}
	    bgcolor {xpasetbuf $xpa "$ds9(bg,color)\n"}
	    nancolor {xpasetbuf $xpa "$ds9(nan,color)\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdPrefs {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessPrefsCmd param i}
    CatchXPAError $xpa
}

proc XPASendPreserve {xpa cdata param} {
    global ds9
    global scale
    global panzoom
    global marker

    InitXPAError $xpa
    catch {
	switch -- [string tolower $param] {
	    scale {xpasetbuf $xpa [ToYesNo $scale(preserve)]}
	    pan {xpasetbuf $xpa [ToYesNo $panzoom(preserve)]}
	    regions {xpasetbuf $xpa [ToYesNo $marker(preserve)]}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdPreserve {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessPreserveCmd param i}
    CatchXPAError $xpa
}

proc XPASendPlot {xpa cdata param} {
    global current
    global ap

    InitXPAError $xpa
    catch {xpasetbuf $xpa "$ap(xpa)\n"}
    CatchXPAError $xpa
}

proc XPARcvdPlot {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessPlotCmd param i buf}
    CatchXPAError $xpa
}

proc XPASendPrint {xpa cdata param} {
    global ps

    InitXPAError $xpa
    catch {
	switch -- [string tolower $param] {
	    destination {xpasetbuf $xpa "$ps(dest)\n"}
	    command {xpasetbuf $xpa "$ps(cmd)\n"}
	    filename {xpasetbuf $xpa "$ps(filename)\n"}
	    palette {xpasetbuf $xpa "$ps(color)\n"}
	    level {xpasetbuf $xpa "$ps(level)\n"}
	    interpolate {xpasetbuf $xpa "0\n"}
	    resolution {xpasetbuf $xpa "$ps(resolution)\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdPrint {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessPrintCmd param i}
    CatchXPAError $xpa
}

proc XPARcvdRaise {xpa cdata param buf len} {
    global ds9

    InitXPAError $xpa
    catch {set i 0; ProcessRaiseCmd param i}
    CatchXPAError $xpa
}

proc XPASendRegions {xpa cdata param} {
    global current
    global marker

    InitXPAError $xpa
    catch {
	if {$current(frame) != {}} {
	    set item [lindex $param 0]
	    switch -- $item {
		format {xpasetbuf $xpa "$marker(format)\n"}
		coord -
		system {xpasetbuf $xpa "$marker(system)\n"}
		sky  {xpasetbuf $xpa "$marker(sky)\n"}
		coordformat -
		skyformat {xpasetbuf $xpa "$marker(skyformat)\n"}
		strip {xpasetbuf $xpa "[ToYesNo $marker(strip)]\n"}
		wcs {xpasetbuf $xpa "[ToYesNo $marker(wcs)]\n"}
		delim {
		    if {$marker(strip)} {
			xpasetbuf $xpa "semicolon\n"
		    } else {
			xpasetbuf $xpa "nl\n"
		    }
		}
		shape {xpasetbuf $xpa "$marker(shape)\n"}
		color {xpasetbuf $xpa "$marker(color)\n"}
		width {xpasetbuf $xpa "$marker(width)\n"}

		tag -
		tags -
		group -
		groups {xpasetbuf $xpa \
		"[lsort [$current(frame) get marker tag all]]\n"}

		default {
		    set format $marker(format)
		    set sys $marker(system)
		    set sky $marker(sky)
		    set skyformat $marker(skyformat)
		    set strip $marker(strip)
		    set wcs $marker(wcs)
		    set select {}
		    set props {}
		    set tags {}

		    set i 0
		    set l [llength $param]
		    while {$i < $l} {
			switch -- [lindex $param $i] {
			    -format {incr i; set format [lindex $param $i]}
			    -coord -
			    -system {
				incr i
				# for backward compatibility
				switch -- [lindex $param $i] {
				    fk4 -
				    fk5 -
				    icrs -
				    galactic -
				    ecliptic {
					set sys wcs
					set sky [lindex $param $i]
				    }
				    
				    default {set sys [lindex $param $i]}
				}
			    }
			    -sky {incr i; set sky [lindex $param $i]}
			    -coordformat -
			    -skyformat {
				incr i;
				set skyformat [lindex $param $i]
			    }
			    -strip {
				incr i; 
				set strip [FromYesNo [lindex $param $i]]
			    }
			    -wcs {
				incr i; 
				set wcs [FromYesNo [lindex $param $i]]
			    }
			    -delim  {
				incr i;
				if {[lindex $param $i] != "nl"} {
				    set strip 1
				} else {
				    set strip 0
				}
			    }

			    include {append props " include = 1"}
			    exclude {append props " include = 0"}
			    source {append props " source = 1"}
			    background {append props " source = 0"}
			    selected {set select "select"}
			    -prop {
				append props \
				    " [lindex $param [expr $i+1]] = [lindex $param [expr $i+2]]"
				incr i 2
			    }
			    -tag -
			    -group {
				incr i
				append tags "tag = \{[lindex $param $i]\}"
			    }
			    default {
				xpaerror $xpa \
				    "Illegal option: [lindex $param $i]"
			    }
			}
			incr i
		    }

		    xpasetbuf $xpa [$current(frame) marker list \
					$select \
					$format $sys $sky $skyformat \
					$strip $wcs $props $tags]
		}
	    }
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdRegions {xpa cdata param buf len} {
    global marker
    global current

    InitXPAError $xpa

    if {$param != {}} {
	catch {set i 0; ProcessMarkerCmd param i}
    } else {
	switch -- $marker(format) {
	    xy {
		$current(frame) marker load $marker(format) \
		    [xparec $xpa datafd] $marker(system) $marker(sky)
		UpdateGroupDialog
	    }
	    default {
		$current(frame) marker load $marker(format) \
		    [xparec $xpa datafd]
		UpdateGroupDialog
	    }
	}
    }

    CatchXPAError $xpa
}

proc XPASendRGB {xpa cdata param} {
    global current
    global rgb

    InitXPAError $xpa
    catch {
	switch -- [lindex $param 0] {
	    channel {xpasetbuf $xpa "$current(rgb)\n"}
	    lock {
		switch -- [string tolower [lindex $param 1]] {
		    bin {xpasetbuf $xpa "$rgb(lock,bin)\n"}
		    scale {xpasetbuf $xpa "$rgb(lock,scale)\n"}
		    colorbar {xpasetbuf $xpa "$rgb(lock,colorbar)\n"}
		}
	    }
	    system {xpasetbuf $xpa "$rgb(system)\n"}
	    view {
		switch -- [lindex $param 1] {
		    red {xpasetbuf $xpa [ToYesNo $rgb(red)]}
		    green {xpasetbuf $xpa [ToYesNo $rgb(green)]}
		    blue {xpasetbuf $xpa [ToYesNo $rgb(blue)]}
		}
	    }
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdRGB {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessRGBCmd param i}
    CatchXPAError $xpa
}

proc XPASendRotate {xpa cdata param} {
    global current

    InitXPAError $xpa
    catch {xpasetbuf $xpa "$current(rotate)\n"}
    CatchXPAError $xpa
}

proc XPARcvdRotate {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessRotateCmd param i}
    CatchXPAError $xpa
}

proc XPARcvdSaveImage {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessSaveImageCmd param i}
    CatchXPAError $xpa
}

proc XPARcvdSaveFits {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessSaveFitsCmd param i}
    CatchXPAError $xpa
}

proc XPARcvdSaveMPEG {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessSaveMPEGCmd param i}
    CatchXPAError $xpa
}

proc XPASendScale {xpa cdata param} {
    global current
    global scale

    InitXPAError $xpa
    catch {
	switch -- [string tolower $param] {
	    datasec {xpasetbuf $xpa "$scale(datasec)\n"}
	    limits {
		if {$current(frame) != {}} {
		    set lims [$current(frame) get clip]
		    xpasetbuf $xpa "[lindex $lims 0] [lindex $lims 1]\n"
		}
	    }
	    mode {xpasetbuf $xpa "$scale(mode)\n"}
	    scope {xpasetbuf $xpa "$scale(scope)\n"}
	    default {xpasetbuf $xpa "$scale(type)\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdScale {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessScaleCmd param i}
    CatchXPAError $xpa
}

proc XPASendSingle {xpa cdata param} {
    global ds9

    InitXPAError $xpa
    catch {
	if {$ds9(display,user) == "single"} {
	    xpasetbuf $xpa [ToYesNo 1]
	} else {
	    xpasetbuf $xpa [ToYesNo 0]
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdSingle {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessSingleCmd param i}
    CatchXPAError $xpa
}

proc XPASendShm {xpa cdata param} {
    global current

    InitXPAError $xpa
    catch {
	if {$current(frame) != {}} {
	    xpasetbuf $xpa "[$current(frame) get fits file name full]\n"
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdShm {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessShmCmd param i 0 foobar}
    CatchXPAError $xpa
}

proc XPASendSmooth {xpa cdata param} {
    global current
    global smooth

    InitXPAError $xpa
    catch {
	switch -- [lindex $param 0] {
	    function {xpasetbuf $xpa "$smooth(function)\n"}
	    radius {xpasetbuf $xpa "$smooth(radius)\n"}
	    default {xpasetbuf $xpa [ToYesNo $smooth(view)]}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdSmooth {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessSmoothCmd param i}
    CatchXPAError $xpa
}

proc XPARcvdSource {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessSourceCmd param i}
    CatchXPAError $xpa
}

proc XPARcvdTcl {xpa cdata param buf len} {
    global ds9
    global message

    InitXPAError $xpa
    if {$ds9(xpa,tcl)} {
	catch {
	    if [string compare $param {}] {
		uplevel #0 $param
	    } else {
		uplevel #0 $buf
	    }
	}
    } else {
	Error "$message(error,xpa,tcl)"
    }
    CatchXPAError $xpa
}

proc XPASendTile {xpa cdata param} {
    global ds9
    global tile

    InitXPAError $xpa
    catch {
	switch -- [lindex $param 0] {
	    mode {xpasetbuf $xpa "$tile(mode)\n"}
	    grid {
		switch -- [lindex $param 1] {
		    mode {xpasetbuf $xpa "$tile(grid,mode)\n"}
		    layout {xpasetbuf $xpa "$tile(grid,row) $tile(grid,col)\n"}
		    gap {xpasetbuf $xpa "$tile(grid,gap)\n"}
		}
	    }
	    default {
		if {$ds9(display,user)=="tile"} {
		    xpasetbuf $xpa [ToYesNo 1]
		} else {
		    xpasetbuf $xpa [ToYesNo 0]
		}
	    }
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdTile {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessTileCmd param i}
    CatchXPAError $xpa
}

proc XPARcvdUpdate {xpa cdata param buf len} {
    global current
    global ds9

    InitXPAError $xpa
    catch {
	if {$current(frame) != {}} {
	    switch -- [string tolower [lindex $param 0]] {
		now {
		    if {[lindex $param 1] != {}} {
			$current(frame) update now \
			    [lindex $param 1] \
			    [lindex $param 2] [lindex $param 3] \
			    [lindex $param 4] [lindex $param 5]
		    } else {
			$current(frame) update now
		    }
		}

		on -
		yes {set ds9(xpa,idletasks) 1}
		no -
		off {set ds9(xpa,idletasks) 0}

		{} {$current(frame) update}
		default {
		    $current(frame) update \
			[lindex $param 0] \
			[lindex $param 1] [lindex $param 2] \
			[lindex $param 3] [lindex $param 4]
		}
	    }
	}
    }
    CatchXPAError $xpa
}

proc XPASendView {xpa cdata param} {
    global view

    InitXPAError $xpa
    catch {
	switch -- [string tolower $param] {
	    info {xpasetbuf $xpa [ToYesNo $view(info)]}
	    panner {xpasetbuf $xpa [ToYesNo $view(panner)]}
	    magnifier {xpasetbuf $xpa [ToYesNo $view(magnifier)]}
	    buttons {xpasetbuf $xpa [ToYesNo $view(buttons)]}
	    colorbar {xpasetbuf $xpa [ToYesNo $view(colorbar)]}
	    horzgraph {xpasetbuf $xpa [ToYesNo $view(graph,horz)]}
	    vertgraph {xpasetbuf $xpa [ToYesNo $view(graph,vert)]}

	    filename {xpasetbuf $xpa [ToYesNo $view(info,filename)]}
	    object {xpasetbuf $xpa [ToYesNo $view(info,object)]}
	    minmax {xpasetbuf $xpa [ToYesNo $view(info,minmax)]}
	    frame {xpasetbuf $xpa [ToYesNo $view(info,frame)]}

	    detector {xpasetbuf $xpa [ToYesNo $view(info,detector)]}
	    amplifier {xpasetbuf $xpa [ToYesNo $view(info,amplifier)]}
	    physical {xpasetbuf $xpa [ToYesNo $view(info,physical)]}
	    image {xpasetbuf $xpa [ToYesNo $view(info,image)]}
	    wcs {xpasetbuf $xpa [ToYesNo $view(info,wcs)]}
	    wcsa {xpasetbuf $xpa [ToYesNo $view(info,wcsA)]}
	    wcsb {xpasetbuf $xpa [ToYesNo $view(info,wcsB)]}
	    wcsc {xpasetbuf $xpa [ToYesNo $view(info,wcsC)]}
	    wcsd {xpasetbuf $xpa [ToYesNo $view(info,wcsD)]}
	    wcse {xpasetbuf $xpa [ToYesNo $view(info,wcsE)]}
	    wcsf {xpasetbuf $xpa [ToYesNo $view(info,wcsF)]}
	    wcsg {xpasetbuf $xpa [ToYesNo $view(info,wcsG)]}
	    wcsh {xpasetbuf $xpa [ToYesNo $view(info,wcsH)]}
	    wcsi {xpasetbuf $xpa [ToYesNo $view(info,wcsI)]}
	    wcsj {xpasetbuf $xpa [ToYesNo $view(info,wcsJ)]}
	    wcsk {xpasetbuf $xpa [ToYesNo $view(info,wcsK)]}
	    wcsl {xpasetbuf $xpa [ToYesNo $view(info,wcsL)]}
	    wcsm {xpasetbuf $xpa [ToYesNo $view(info,wcsM)]}
	    wcsn {xpasetbuf $xpa [ToYesNo $view(info,wcsN)]}
	    wcso {xpasetbuf $xpa [ToYesNo $view(info,wcsO)]}
	    wcsp {xpasetbuf $xpa [ToYesNo $view(info,wcsP)]}
	    wcsq {xpasetbuf $xpa [ToYesNo $view(info,wcsQ)]}
	    wcsr {xpasetbuf $xpa [ToYesNo $view(info,wcsR)]}
	    wcss {xpasetbuf $xpa [ToYesNo $view(info,wcsS)]}
	    wcst {xpasetbuf $xpa [ToYesNo $view(info,wcsT)]}
	    wcsu {xpasetbuf $xpa [ToYesNo $view(info,wcsU)]}
	    wcsv {xpasetbuf $xpa [ToYesNo $view(info,wcsV)]}
	    wcsw {xpasetbuf $xpa [ToYesNo $view(info,wcsW)]}
	    wcsx {xpasetbuf $xpa [ToYesNo $view(info,wcsX)]}
	    wcsy {xpasetbuf $xpa [ToYesNo $view(info,wcsY)]}
	    wcsz {xpasetbuf $xpa [ToYesNo $view(info,wcsZ)]}
	    default {xpasetbuf $xpa "$view(layout)\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdView {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessViewCmd param i}
    CatchXPAError $xpa
}

proc XPASendVersion {xpa cdata param} {
    global ds9

    InitXPAError $xpa
    catch {xpasetbuf $xpa "$ds9(title) $ds9(version)\n"}
    CatchXPAError $xpa
}

proc XPASendVO {xpa cdata param} {
    global vo

    InitXPAError $xpa
    set l [llength $vo(server,host)]
    for {set ii 0} {$ii<$l} {incr ii} {
	xpasetbuf $xpa \
	    "$vo(server,host) $vo(server,title) $vo(server,url) $vo(b$ii)\n"
    }
    CatchXPAError $xpa
}

proc XPARcvdVO {xpa cdata param buf len} {
    catch {set i 0; ProcessVOCmd param i}
    # someone is setting the error state
    InitXPAError $xpa
}

proc XPASendWCS {xpa cdata param} {
    global wcs

    InitXPAError $xpa
    catch {
	switch -- [string tolower $param] {
	    align {xpasetbuf $xpa [ToYesNo $wcs(align)]}
	    system {xpasetbuf $xpa "$wcs(system)\n"}
	    sky  {xpasetbuf $xpa "$wcs(sky)\n"}
	    format -
	    skyformat {xpasetbuf $xpa "$wcs(skyformat)\n"}
	    default {xpasetbuf $xpa "$wcs(system)\n"}
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdWCS {xpa cdata param buf len} {
    global wcs
    global current

    InitXPAError $xpa
    set which [lindex $param 1]
    if {($which=="replace" || $which=="append") && [lindex $param 2]=={}} {
	$current(frame) wcs $which [xparec $xpa datafd]
    } else {
	set i 0
	ProcessWCSCmd param i
    }
    CatchXPAError $xpa
}

proc XPASendWeb {xpa cdata param} {
    global hv

    InitXPAError $xpa
    if {[info exists hv(web,url)]} {
	xpasetbuf $xpa "$hv(web,url)\n"
    }
    CatchXPAError $xpa
}

proc XPARcvdWeb {xpa cdata param buf len} {
    catch {set i 0; ProcessWebCmd param i}
    # someone is setting an error state
    InitXPAError $xpa
}

proc XPASendWidth {xpa cdata param} {
    global canvas

    InitXPAError $xpa
    catch {xpasetbuf $xpa "$canvas(width)\n"}
    CatchXPAError $xpa
}

proc XPARcvdWidth {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessWidthCmd param i}
    CatchXPAError $xpa
}

proc XPASendZoom {xpa cdata param} {
    global current

    InitXPAError $xpa
    catch {
	set z1 [lindex $current(zoom) 0]
	set z2 [lindex $current(zoom) 1]
	if {$z1 != $z2} {
	    xpasetbuf $xpa "$current(zoom)\n"
	} else {
	    xpasetbuf $xpa "$z1\n"
	}
    }
    CatchXPAError $xpa
}

proc XPARcvdZoom {xpa cdata param buf len} {
    InitXPAError $xpa
    catch {set i 0; ProcessZoomCmd param i}
    CatchXPAError $xpa
}

proc DisplayXPAInfo {} {
    global xpa
    global message

    set r {}
    if { ([info exists xpa] && ($xpa != {})) } {
	append r "[format "XPA_VERSION:\t%s" [xparec $xpa version]]\n"
	append r "[format "XPA_CLASS:\t%s"   [xparec $xpa class]]\n"
	append r "[format "XPA_NAME:\t%s"    [xparec $xpa name]]\n"
	append r "[format "XPA_METHOD:\t%s"  [xparec $xpa method]]\n"
	SimpleTextDialog xpa "XPA Information" 80 20 append bottom $r
    } else {
	Error "$message(error,xpa,init)"
    }
}

proc ProcessXPACmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ds9
    global env

    switch -- [string tolower [lindex $var $i]] {
	unix -
	inet -
	local -
	localhost {set env(XPA_METHOD) [lindex $var $i]}
	noxpans {set env(XPA_NSREGISTER) false}

	tcl {incr i; set ds9(xpa,tcl) [FromYesNo [lindex $var $i]]}

	yes -
	true -
	1 -
	no -
	false -
	0 {set ds9(xpa) [FromYesNo [lindex $var $i]]}
    }
}
